/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation.solr;

import java.util.Map;

import org.ametys.cms.ObservationConstants;
import org.ametys.core.observation.Event;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;

/**
 * Observes ODF {@link Program} deletion or unpublishing in order to synchronize the Solr index.
 */
public class UnindexProgramObserver extends AbstractSolrODFObserver
{
    @Override
    public boolean supports(Event event)
    {
        return (event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING) || event.getId().equals(ObservationConstants.EVENT_CONTENT_UNTAG_LIVE))
                && event.getArguments().get(ObservationConstants.ARGS_CONTENT) instanceof Program;
    }
    
    @Override
    public int getPriority()
    {
        // Will be processed before live synchronization observers
        return MAX_PRIORITY;
    }
    
    @Override
    protected void _updateIndex(Event event, Map<String, Object> transientVars, Page odfRootPage, Program program, SubProgram subProgram, Course course) throws Exception
    {
        Page programPage = _odfPageResolver.getProgramPage(odfRootPage, program);
        if (programPage != null)
        {
            String pageId = programPage.getId();
            getLogger().info("Updating Solr document with id: {}", pageId);
            
            // Delete the page document.
            if (event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING)) // otherwise it is an unpublishing, so keep page in 'default'
            {
                _solrPageIndexer.unindexPage(pageId, RepositoryConstants.DEFAULT_WORKSPACE, true, true);
            }
            _solrPageIndexer.unindexPage(pageId, WebConstants.LIVE_WORKSPACE, true, true);
        }
    }
}
