/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
 ODFCart = {
    
    isAnonymous: true,
    
    userItems: {},
    
    /**
     * @property {String} __CART_ITEMS_ID The id of cart for local storage
     */
    __CART_ITEMS_ID: "odfCartItems",
    
    /**
     * @property {Function[]} __listeners The registered listeners on cart modification
     */
    __listeners: [],
    
    localCart: {
        
        /**
	     * Add a item to the cart (local storage only)
	     * @param {String} id the item's id
	     * @param {String} title the content's title
         * @param {String} pageId the page's id
	     */
	    addItem: function(id, title)
	    {
	        var elmts = ODFCart.localCart.getItems();
	        
	        elmts[id] = {
                id: id,
	            title: title
	        };
	        
	        localStorage.setItem(ODFCart.__CART_ITEMS_ID, JSON.stringify(elmts));
	    },
	    
	    /**
	     * Get the item's store into the cart (local storage)
	     * @return {Object} The cart's items
	     */
	    getItems: function()
	    {
	        var elmts = localStorage.getItem(ODFCart.__CART_ITEMS_ID) || '{}';
	        return elmts ? JSON.parse(elmts) : {}; 
	    },
        
        /**
	     * Empty the cart (local storage only)
	     */
	    removeAll: function()
	    {
	        localStorage.setItem(ODFCart.__CART_ITEMS_ID, '{}');
	    },
	    
	    /**
	     * Remove a item from the cart (local storage only)
	     * @param {String} id the id of item to remove
	     */
	    removeItem: function(id)
	    {
	        var elmts = ODFCart.localCart.getItems();
	        delete elmts[id];
	        localStorage.setItem(ODFCart.__CART_ITEMS_ID, JSON.stringify(elmts));
	    },
	    
	    /**
	     * Get the item stored into the cart
	     * @param {String} id the id of item
	     * @return {Object} the item or null if not found
	     */
	    getItem: function(id)
	    {
	        var elmts = ODFCart.localCart.getItems();
	        return elmts.hasOwnProperty(id) ? elmts[id] : null;
	    },
	    
	    hasItem: function(id)
	    {
	        var elmts = ODFCart.localCart.getItems();
	        return elmts.hasOwnProperty(id)
	    }
    },
    
    /**
     * Register a listener on cart modification
     * @param {Function} listener the function to call on cart modification
     */
    registerListener: function(listener)
    {
        ODFCart.__listeners.push(listener);
    },
    
    _notifyListeners: function()
    {
        for (var i=0; i < this.__listeners.length; i++)
        {
            this.__listeners[i]();
        }
    },
    
    /**
     * Empty callback
     */
    emptyCallback: function() {},
    
    /**
     * Add a item into cart
     * @param {String} id the id of item to add
     * @param {String} title the title of item to add
     * @param {Function} [callback] Optional callback to invoked after adding item. By default, reload the page.
     */
    addItem: function(id, title, callback)
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
            if (!me.isAnonymous)
	        {
	            me.userItems[id] = {
	                id: id,
	                title: title
	            };
	            me.save(callback);
	        }
	        else
	        {
	            me.localCart.addItem(id);
	            if (typeof (callback) == 'function')
	            {
	                callback();
	            }
	            else
	            {            
	                 document.location.reload(false);
                     return;
	            }
	        }
            
            me._notifyListeners();
        });
    },
    
    /**
     * Remove a item from cart
     * @param {String} id the id of item to remove
     * @param {Function} [callback] Optional callback to invoked after removing item. By default, reload the page.
     */
    removeItem: function(id, callback)
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
            if (!me.isAnonymous)
            {
                delete me.userItems[id];
                me.save(callback);
            }
            else
            {
                me.localCart.removeItem(id);
                if (typeof (callback) == 'function')
                {
                    callback();
                }
                else
                {            
                     document.location.reload(false);
                     return;
                }
            }
            
            me._notifyListeners();
        });
    },
    
    /**
     * Empty the cart
     * @param {Function} [callback] Optional callback to invoked after removing all. By default, reload the page.
     */
    removeAll: function(callback)
    {
        var me = this;
        me._internalGetItems(function(itemIds){
            if (!me.isAnonymous)
            {
                me.userItems = {};
                me.save(callback);
            }
            else
            {
                me.localCart.removeAll();
                if (typeof (callback) == 'function')
                {
                    callback();
                }
                else
                {            
                     document.location.reload(false);
                     return;
                }
            }
            
            me._notifyListeners();
        });
    },
    
    /**
     * Determines if the cart contains a items
     * @param {String} id the id of the item
     * @return {Boolean} true if the item is in cart
     */
    hasItem: function(id, callback)
    {
        var me = this;
        me._internalGetItems(function(itemIds){
            
            var answer = me.isAnonymous ? me.localCart.hasItem(id) : me.userItems.hasOwnProperty(id);
            if (typeof (callback) == 'function')
            {
                callback(answer);
            }
        });
    },
    
    /**
     * Save the cart if a user is currently connected
     * @param {Function} [callback] the callback function
     */
    save: function(callback)
    {
        if (this.isAnonymous)
        {
            // No user connected
            return;
        }
        
        var items = this.userItems;
        var itemIds = Object.keys(this.userItems);
        
	    function success(data)
	    {
            if (data && data.success)
            {
                if (typeof (callback) == 'function')
	            {
	                callback(data);
	            }
	            else
	            {            
	                 document.location.reload(false);
	            }
            }
	    }
	    
	    function error(jqXHR, textStatus, errorThrown)
	    {
	        ODFCart.error("{{i18n PLUGINS_ODFWEB_CART_SAVE_ERROR}}");
	    }
	
	    $j.ajax({
	        url: contextPath + "/_plugins/odf-web/cart/save",
	        data: {
               "itemIds": itemIds.join(",")
            },
            dataType: "json",
	        type: 'post',
	        success: success,
	        error: error
	    });
    },
    
    getItemsCount: function(callback)
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
            if (typeof(callback) == "function")
            {
                callback(itemIds.length);
            }
        });
    },
    
    /**
     * Get the cart's items
     * @param {String} lang The current language
     * @param {String} [view] The view to use for cart's elements. Defaults to 'cart'.
     * @param {Function} [callback] the callback function to call after getting items.
     */
    getItems: function(lang, view, callback)
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
            // Get items view
            me._getItemsView(itemIds, lang, view, callback);
        });
    },
    
    _initialized: false,
    _isRunning: false,
    _callbacks: [],
    
    _internalGetItems: function(callback)
    {
        if (this._initialized)
        {
            var itemIds = [];
            if (this.isAnonymous)
            {
                // Get items from local storage
                itemIds = Object.keys(this.localCart.getItems());
            }
            else
            {
                itemIds = Object.keys(this.userItems);
            }
            
            callback(itemIds);
            return;
        }
        
        this._callbacks.push(callback);
        
        var me = this;
        
        function success(data)
        {
            me._isRunning = false;
            
            if (data)
            {
                me.isAnonymous = data.isAnonymous;
                
                var itemIds = [];
                if (me.isAnonymous)
                {
                    // Get items from local storage
                    me.userItems = {};
                    itemIds = Object.keys(me.localCart.getItems());
                }
                else
                {
                    // Merge saved items with localStorage
                    me._mergeUserPrefs(data.items);
                    itemIds = Object.keys(me.userItems);
                }
                
                $j.each(me._callbacks, function(index, cb) {
                    cb(itemIds);
                });
                
                me._initialized = true;
                me._callbacks = [];
            }
        }
        
        function handleError(jqXHR, textStatus, errorThrown)
        {
            ODFCart.error("{{i18n PLUGINS_ODFWEB_CART_GET_ITEMS_ERROR}}");
            me._isRunning = false;
        }
        
        if (!this._isRunning)
        {
            this._isRunning = true;
            
            $j.ajax({
	            url: contextPath + "/_plugins/odf-web/cart/saved-items",
	            dataType: "json",
	            type: 'post',
	            success: success,
	            error: handleError
	        });
        }
    },
    
    _getItemsView: function(itemIds, lang, view, callback)
    {
         if (itemIds.length > 0)
         {
            function success(data)
            {
                if (data && data.items)
                {
                    $j(data.items).each(function (index, item) {
                        if (item.view)
                        {
                            var dom = new DOMParser().parseFromString(item.view, "text/html");
                            item.view = $j(dom.body).find(">*").html();
                        }
                    });

                    if (typeof(callback) == "function")
	                {
	                    callback(data.items);
	                }
                }
            }
            
            function handleError(jqXHR, textStatus, errorThrown)
            {
                ODFCart.error("{{i18n PLUGINS_ODFWEB_CART_GET_ITEMS_ERROR}}");
            }
            
            $j.ajax({
                url: contextPath + "/_plugins/odf-web/cart/items/view",
                data: {
                    itemIds: itemIds.join(','),
                    lang: lang,
                    view: view || 'cart'
                },
                dataType: "json",
                type: 'post',
                success: success,
                error: handleError
            });       
         }
         else
         {
            if (typeof(callback) == "function")
            {
                callback([]);
            }
         }
    },
    
    /**
     * @private
     * Merge the user preferences with the local storage
     * @param {Object[]} savedItems The saved items (in user prefs)
     */
    _mergeUserPrefs: function(savedItems)
    {
        var localItems = this.localCart.getItems();
        var localSize = Object.keys(localItems).length;

        var me = this;
        $j.each(savedItems, function(index, savedItem) {
            
            if (!me.localCart.hasItem(savedItem.id))
            {
                // Update local storage
                me.localCart.addItem(savedItem.id, savedItem.title);
            }
            else
            {
                // The item is already in current cart
                localSize--;
            }
        });
        
        this.userItems = this.localCart.getItems();
        
        if (localSize > 0)
	    {
	        // at least there is a local element not saved on server side
	        this.save();
	    }
        
        // Empty local storage
        this.localCart.removeAll();
    },
    
    /**
     * Subscribe to a item of the cart
     * @param {String} id the id of item to subscribe to
     * @param {Function} [callback] Optional callback to invoked after subscribing to the item. By default, reload the page.
     */
    subscribe: function(id, subscribe, callback)
    {
        if (this.isAnonymous)
        {
            // No user connected
            return;
        }
        
        function success(data)
        {
            if (data && data.success)
            {
                if (typeof (callback) == 'function')
                {
                    callback(data);
                }
                else
                {            
                     document.location.reload(false);
                }
            }
        }
        
        function error(jqXHR, textStatus, errorThrown)
        {
            ODFCart.error("{{i18n PLUGINS_ODFWEB_CART_SUBSCRIBE_ERROR}}");
        }
    
        $j.ajax({
            url: contextPath + "/_plugins/odf-web/cart/subscribe",
            data: {
               "itemId": id,
               subscribe: subscribe
            },
            dataType: "json",
            type: 'post',
            success: success,
            error: error
        });
    },
    
    /**
     * Unubscribe to a item of the cart
     * @param {String} id the id of item to unsubscribe to
     * @param {Function} [callback] Optional callback to invoked after removing subscription. By default, reload the page.
     */
    unsubscribe: function(id, callback)
    {
        var me = this;
        me._internalSubscription(id, false, callback);
    },
    
    /**
     * Download the cart (ZIP with each PDF file)
     */
    download: function(siteName)
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
	        if (itemIds.length > 0)
	        {
	            var form = document.createElement("form");
		        form.action = contextPath + "/_plugins/odf-web/cart/download.zip";
		        form.method = "POST";
		        form.target = "_blank";
		       
	            var input = document.createElement("input");
	            input.type = 'hidden';
	            input.name = 'siteName';
	            input.value = siteName;
	            form.appendChild(input);
	                
		       $(itemIds).each(function (index, itemId) {
	                input = document.createElement("input");
	                input.type = 'hidden';
	                input.name = 'itemId';
	                input.value = itemId;
	                form.appendChild(input);
	            });
	   
		        form.style.position = "absolute";
		        form.style.left = "-10000px";
		        
		        document.body.appendChild(form);
		        
		        try
		        {
		            form.submit();
		        }
		        catch(e)
		        {
		            ODFCart.error("{{i18n plugin.core-ui:PLUGINS_CORE_UI_OPEN_WINDOW_POPUP_BLOCKED}}", "{{i18n plugin.core-ui:PLUGINS_CORE_UI_OPEN_WINDOW_POPUP_BLOCKED_TITLE}}");
		        }
		        
		        // delete form
		        window.setTimeout(function () {document.body.removeChild(form)}, 1);
	        }
        });
    },
    
    share: function(siteName, sharedDialogId)
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
            if (itemIds.length > 0)
            {
                 var dialog = $j('#' + sharedDialogId),
		            inputEmail = dialog.find("input[name='email']"),
		            inputMsg = dialog.find("textarea[name='message']");
                    
                 me._clearInvalid(dialog);
                 
                 inputEmail.val('');
                 inputMsg.text('');
                 
                 dialog.dialog({
		            // Remove the closing 'X' from the dialog
		            open: function(event, ui) { $(".ui-dialog-titlebar-close").hide(); }, 
		            modal: true,
		            width: 450,
		            buttons: [{
		                text: "{{i18n plugin.odf-web:PLUGINS_ODFWEB_SERVICE_CART_SHARE_DIALOG_OK_BUTTON}}",
		                click: function() {
		                    if (!me._validateEmailInput(inputEmail))
		                        return;
		                    
		                    var params = {
		                        emails: $.map(inputEmail.val().split(","), $.trim),
		                        message: inputMsg.val(),
		                        itemIds: itemIds
		                    }
		                    
		                    $(this).dialog( "close" );
		                    me._sendSharedCart(params);
		                }
		            }, {
		                text: "{{i18n plugin.odf-web:PLUGINS_ODFWEB_SERVICE_CART_SHARE_DIALOG_CANCEL_BUTTON}}",
		                click: function() {
		                    $(this).dialog( "close" );
		                }
		            }]
		        });
            }
        });
    },
    
    _clearInvalid: function(dialog)
    {
        dialog.find(".field").removeClass("error");
        dialog.find(".error-hint").remove();
    },
    
    _validateEmailInput: function (input)
    {
        input.parents('.field').find(".error-hint").remove();
        
        if (input.val() == '')
        {
            input.focus();
            input.parents('.field').addClass("error");
            input.parents('.field').append("<p class=\"error-hint\">{{i18n plugin.odf-web:PLUGINS_ODFWEB_SERVICE_CART_SHARE_DIALOG_MANDATORY_FIELD_ERROR}}</p>");
            return false;            
        }
        
        if (!/^([a-z0-9._-]+@[a-z0-9.-]{2,}[.][a-z]{2,})?(\s*,\s*([a-z0-9._-]+@[a-z0-9.-]{2,}[.][a-z]{2,})?)*$/.test(input.val()))
        {
            input.focus();
            input.parents('.field').addClass("error");
            input.parents('.field').append("<p class=\"error-hint\">{{i18n plugin.odf-web:PLUGINS_ODFWEB_SERVICE_CART_SHARE_DIALOG_INVALID_EMAIL}}</p>");
            return false;
        }
        
        input.parents('.field').removeClass("error");
        
        return true;
    },
    
    _sendSharedCart: function(params)
    {
        function success(data)
        {
            if (data && data.success)
            {
                ODFCart.info("{{i18n PLUGINS_ODFWEB_CART_SHARE_SUCCESS}}");
            }
            else if (data.mailsInError)
            {
                ODFCart.error("{{i18n PLUGINS_ODFWEB_CART_SHARE_MAILS_IN_ERROR}}" + data.mailsInError.join(', '));
            }
            else
            {
                ODFCart.error("{{i18n PLUGINS_ODFWEB_CART_SHARE_ERROR}}");
            }
        }
        
        function handleError(jqXHR, textStatus, errorThrown)
        {
            ODFCart.error("{{i18n PLUGINS_ODFWEB_CART_SHARE_ERROR}}");
        }
        
        $j.ajax({
            url: contextPath + "/_plugins/odf-web/cart/share",
            data: {
                itemIds: params.itemIds.join(','),
                emails: params.emails.join(','),
                message: params.message || ''
            },
            dataType: "json",
            type: 'post',
            success: success,
            error: handleError
        });       
    },
    
    /**
     * Display a info message
     * @param {String} message the info message
     * @param {String} [title] the title
     */
    info: function(message, title)
    {
        if ( !title )
            title = "{{i18n plugin.odf-web:PLUGINS_ODFWEB_CART_INFO_TITLE}}";

        this._showMessage(title, message, 'ametys-dialog-info');
    },
    
    /**
     * Display a error message
     * @param {String} message the error message
     * @param {String} [title] the title
     */
    error: function(message, title)
    {
        if ( !title )
            title = "{{i18n plugin.odf-web:PLUGINS_ODFWEB_CART_ERROR_TITLE}}";

        this._showMessage(title, message, 'ametys-dialog-error');
    },
    
    _showMessage: function(title, message, dialogCls)
    {
        if ( !message )
            message = 'No Message to Display.';
            
        $('<div class="ui-corner-all"></div>').html( message ).dialog({
            // Remove the closing 'X' from the dialog
            open: function(event, ui) { $(".ui-dialog-titlebar-close").hide(); }, 
            dialogClass: dialogCls,
            title: title,
            resizable: true,
            minWidth: 400,
            maxWidth: 500,
            modal: true,
            zIndex: 10000,
            buttons: {
                "{{i18n plugin.odf-web:PLUGINS_ODFWEB_CART_INFO_OK_BTN}}": function()  {
                    $( this ).dialog( 'close' );
                }
            }
        });
    }
    
 }