/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.program;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.model.ModelItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Generates the exhaustive list of programs.
 */
public class ProgramListGenerator extends ServiceableGenerator
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    
    /** Handler for root */
    protected OdfPageHandler _odfPageHandler;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _odfPageHandler = (OdfPageHandler) serviceManager.lookup(OdfPageHandler.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String site = parameters.getParameter("site", (String) request.getAttribute("site"));
        String lang = parameters.getParameter("lang", (String) request.getAttribute("sitemapLanguage"));
        String catalog = parameters.getParameter("catalog", (String) request.getAttribute("catalog"));
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        String[] filteredByOrgUnits = zoneItem.getServiceParameters().getValue("orgunitlist", false, new String[0]);

        Page rootPage = _odfPageHandler.getOdfRootPage(site, lang, catalog);
        if (rootPage == null)
        {
            throw new IllegalArgumentException("There is no ODF root page for catalog '" + catalog + "' and language '" + lang + "'.");
        }
        
        AmetysObjectIterable<Program> programs = _odfPageHandler.getProgramsWithRestrictions(rootPage, null, null, null, null);
        
        contentHandler.startDocument();
        
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("root-page-path", rootPage.getPathInSitemap());
        
        XMLUtils.startElement(contentHandler, "programs", atts);
        
        HashMap<String, List<String>> metadataFilters = new HashMap<>(); 
       
        if (filteredByOrgUnits.length > 0)
        {
            List<String> filteredByOrgUnitsAsList = Arrays.asList(filteredByOrgUnits);
            metadataFilters.put("orgUnit", filteredByOrgUnitsAsList);
            
            for (Program program : programs)
            {
                // Check at least one program's org units belongs to the filtered list 
                List<String> orgUnits = program.getOrgUnits();
                
                boolean found = false;
                for (String ouId : orgUnits)
                {
                    try
                    {
                        OrgUnit ou = _ametysResolver.resolveById(ouId);
                        if (_isEqualOrHasParentInList(ou, filteredByOrgUnitsAsList))
                        {
                            found = true;
                            break;
                        }
                    }
                    catch (UnknownAmetysObjectException e)
                    {
                        getLogger().error("The OrgUnit with id '" + ouId + "' does not exist anymore. It will be ignored.", e);
                    }
                    
                }
                
                if (found)
                {
                    saxProgram(program, metadataFilters);
                }
            }
        }
        else
        {
            for (Program program : programs)
            {
                saxProgram(program, metadataFilters);
            }            
        }
        XMLUtils.endElement(contentHandler, "programs");
        
        contentHandler.endDocument();
    }
    
    private boolean _isEqualOrHasParentInList (OrgUnit ou, List<String> filteredOrgUnits)
    {
        if (filteredOrgUnits.contains(ou.getId()))
        {
            return true;
        }
        
        // Check parents
        OrgUnit parentOrgUnit = ou.getParentOrgUnit();
        while (parentOrgUnit != null)
        {
            if (filteredOrgUnits.contains(parentOrgUnit.getId()))
            {
                return true;
            }
            parentOrgUnit = parentOrgUnit.getParentOrgUnit();
        }
        
        return false;
    }
    
    /**
     * SAX a program.
     * @param program The program to SAX.
     * @param attributeFilters Filters to apply to the attributes
     * @throws SAXException If an error occurs while SAXing
     * @throws IOException If an error occurs while retrieving content.
     */
    public void saxProgram(Program program, HashMap<String, List<String>> attributeFilters) throws SAXException, IOException
    {
        Map<String, ModelItem> enumeratedAttributes = _odfPageHandler.getEnumeratedAttributes(ProgramFactory.PROGRAM_CONTENT_TYPE, true);
        
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", program.getId());
        atts.addCDATAAttribute("name", program.getName());
        atts.addCDATAAttribute("title", program.getTitle());
        
        XMLUtils.startElement(contentHandler, "program", atts);
        
        for (String attributePath : enumeratedAttributes.keySet())
        {
            List<String> values = Optional.ofNullable(program.getValue(attributePath))
                                          .map(this::_valuesAsList)
                                          .orElse(Collections.emptyList())
                                          .stream()
                                          .filter(value -> value instanceof String || value instanceof ContentValue)
                                          .map(value -> value instanceof ContentValue ? ((ContentValue) value).getContentId() : (String) value)
                                          .collect(Collectors.toList());
                    
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("path", attributePath);
            
            XMLUtils.startElement(contentHandler, "metadata", attrs);
            for (String value : values)
            {
                if (_filterAttributes(attributePath, value, attributeFilters))
                {
                    XMLUtils.createElement(contentHandler, "value", value);
                }
            }
            XMLUtils.endElement(contentHandler, "metadata");
        }
        
        for (ProgramPart programPart : program.getProgramPartChildren())
        {
            if (programPart instanceof SubProgram)
            {
                saxSubProgram((SubProgram) programPart);
            }
        }
        
        XMLUtils.endElement(contentHandler, "program");
    }
    
    private List<Object> _valuesAsList(Object values)
    {
        if (values instanceof Collection<?>)
        {
            return new ArrayList<>((Collection<?>) values); 
        }
        else if (values instanceof Object[])
        {
            return Arrays.asList((Object[]) values);
        }
        else
        {
            return Arrays.asList(values);
        }
    }
    
    private boolean _filterAttributes(String attributePath, String value, HashMap<String, List<String>> attributeFilters)
    {
        if (attributePath.equals("orgUnit") && attributeFilters.containsKey("orgUnit"))
        {
            try
            {
                OrgUnit ou = _ametysResolver.resolveById(value);
                return _isEqualOrHasParentInList(ou, attributeFilters.get("orgUnit"));
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().error("The OrgUnit with id '" + value + "' does not exist anymore. It will be ignored.", e);
                return false;
            }
            
        }
        return true;
    }
    
    /**
     * SAX a subprogram.
     * @param subProgram The subprogram to SAX.
     * @throws SAXException If an error occurs while SAXing
     */
    protected void saxSubProgram (SubProgram subProgram) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        
        atts.addCDATAAttribute("id", subProgram.getId());
        atts.addCDATAAttribute("name", subProgram.getName());
        atts.addCDATAAttribute("title", subProgram.getTitle());
        
        XMLUtils.createElement(contentHandler, "subprogram", atts);
    }
    
    /**
     * SAX a value.
     * @param name the criterion name.
     * @param value the criterion value.
     * @throws SAXException SAX error
     */
    protected void saxCriterionValue(String name, String value) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("name", name);
        attrs.addCDATAAttribute("value", value);
        XMLUtils.createElement(contentHandler, "criterion", attrs);
    }
}
