/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfweb.repository.FirstLevelPageFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
/**
 * Abstract {@link Observer} for observing events on a Program or Course.
 */
public abstract class AbstractODFObserver extends AbstractLogEnabled implements Observer, Serviceable, Contextualizable
{
    /** The context. */
    protected org.apache.avalon.framework.context.Context _context;
    /** Cocoon context. */
    protected Context _cocoonContext;
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (Context) context.get(org.apache.cocoon.Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    /**
     * The workspace to use. Default to {@link RepositoryConstants#DEFAULT_WORKSPACE default}, 
     * override this method to work on a different workspace
     * @return The workspace to use
     */
    protected String _workspaceToUse()
    {
        return RepositoryConstants.DEFAULT_WORKSPACE;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve current workspace
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Switch to required workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, _workspaceToUse());
            
            AmetysObjectIterable<Page> rootPages = _getODFRootPages();
            if (rootPages.getSize() == 0)
            {
                getLogger().debug("There is no ODF root page, nothing to invalidate");
                return;
            }
            
            Set<Program> programs = _getPrograms(event);
            if (programs.size() == 0)
            {
                getLogger().debug("There is no concerned programs");
                return;
            }
            
            for (Page odfRootPage : rootPages)
            {
                _internalObserve(event, transientVars, odfRootPage, programs, _getSubProgram(event), _getCourse(event));
            }
        }
        catch (Exception e)
        {
            getLogger().error("Unable to observe event: {}", event, e);
        }
        finally
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Do the actual work.
     * @param event the event.
     * @param transientVars transientVars passed from one Observer to another when processing a single Event. 
     * This may allow optimizations between observers.
     * @param odfRootPage the Page holding the virtual factory.
     * @param rootPrograms the root programs.
     * @param subProgram The subprogram. Can be null
     * @param course The course. Can be null.
     * @throws Exception if an error occured
     */
    protected abstract void _internalObserve(Event event, Map<String, Object> transientVars, Page odfRootPage, Set<Program> rootPrograms, SubProgram subProgram, Course course) throws Exception;
    
    /**
     * Get the ODF root pages
     * @return the ODF root pages
     */
    protected AmetysObjectIterable<Page> _getODFRootPages()
    {
        Expression expression = new VirtualFactoryExpression(FirstLevelPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, expression, null);
        
        return _resolver.query(query);
    }
    
    /**
     * Retrieve the target of the observer
     * @param event The event
     * @return The target
     * @throws Exception if failed to get content
     */ 
    protected Content _getContentArgument(Event event) throws Exception
    {
        return (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
    }
    
    /**
     * Get the target abstract program
     * @param event the event
     * @return the target abstract program or <code>null</code>
     * @throws Exception if failed to get content
     */
    protected SubProgram _getSubProgram(Event event) throws Exception
    {
        Content content = _getContentArgument(event);
        
        if (content instanceof SubProgram)
        {
            return (SubProgram) content;
        }
        return null;
    }
    
    /**
     * Get the target course
     * @param event the event
     * @return the target course or <code>null</code>
     * @throws Exception if failed to get content
     */
    protected Course _getCourse(Event event) throws Exception
    {
        Content content = _getContentArgument(event);
        
        if (content instanceof Course)
        {
            return (Course) content;
        }
        return null;
    }
    
    /**
     * Get the target programs
     * @param event the event
     * @return the target programs in a List
     * @throws Exception if failed to get content 
     */
    protected Set<Program> _getPrograms(Event event) throws Exception
    {
        Content content = _getContentArgument(event);
        
        // ODF-1825 We should invalidate the cache for any program or course item because it can impacts the display of any other ODF content
        
        if (content instanceof Program program)
        {
            return Collections.singleton(program);
        }
        else if (content instanceof ProgramItem programItem)
        {
            return _odfHelper.getParentPrograms(programItem);
        }
        else if (content instanceof CoursePart coursePart)
        {
            return _odfHelper.getParentPrograms(coursePart);
        }
        else
        {
            getLogger().debug("This observer only handles ODF contents for virtual pages");
            return Collections.emptySet();
        }
    }
}

