/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;

/**
 * Resolves an ODF page path from the associated ODF content.
 */
public class OdfPageResolver extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The avalon role. */
    public static final String ROLE = OdfPageResolver.class.getName();
    
    private static final String __PATH_IN_SITEMAP_CACHE = OdfPageResolver.class.getName() + "$pathInSitemap";
    private static final String __RESOLVED_PATH_CACHE = OdfPageResolver.class.getName() + "$resolvedPath";
    private static final String __ODF_ROOT_PROGRAM_CACHE = OdfPageResolver.class.getName() + "$rootProgram";
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    /** The odf page handler */
    protected OdfPageHandler _odfPageHandler;
    /** ODF helper */
    protected ODFHelper _odfHelper;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    /** The course page factory */
    protected CoursePageFactory _coursePageFactory;
    /** The program page factory */
    protected ProgramPageFactory _programPageFactory;
    
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _odfPageHandler = (OdfPageHandler) serviceManager.lookup(OdfPageHandler.ROLE);
        _odfHelper = (ODFHelper) serviceManager.lookup(ODFHelper.ROLE);
        _cacheManager = (AbstractCacheManager) serviceManager.lookup(AbstractCacheManager.ROLE);
        AmetysObjectFactoryExtensionPoint ametysObjectFactoryEP = (AmetysObjectFactoryExtensionPoint) serviceManager.lookup(AmetysObjectFactoryExtensionPoint.ROLE);
        _coursePageFactory = (CoursePageFactory) ametysObjectFactoryEP.getExtension(CoursePageFactory.class.getName());
        _programPageFactory = (ProgramPageFactory) ametysObjectFactoryEP.getExtension(ProgramPageFactory.class.getName());
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__PATH_IN_SITEMAP_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PATH_IN_SITEMAP_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PATH_IN_SITEMAP_DESCRIPTION"),
                false);
        
        _cacheManager.createRequestCache(__RESOLVED_PATH_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_RESOLVED_PATH_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_RESOLVED_PATH_DESCRIPTION"),
                false);
        
        _cacheManager.createRequestCache(__ODF_ROOT_PROGRAM_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ROOT_PROGRAM_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ROOT_PROGRAM_DESCRIPTION"),
                false);
    }
    
    /**
     * Get all referencing pages for this program item, in all sites and all sitemaps
     * @param programItem The program item
     * @return the referencing pages
     */
    public Set<Page> getReferencingPages(ProgramItem programItem)
    {
        return getReferencingPages(programItem, null, ((Content) programItem).getLanguage());
    }
    
    /**
     * Get all referencing pages for this program item
     * @param programItem The program item
     * @param siteName The site name. Can be null to search on all sites
     * @param lang The sitemap language. Can be null to search on all sitemaps
     * @return the referencing pages
     */
    public Set<Page> getReferencingPages(ProgramItem programItem, String siteName, String lang)
    {
        Set<Page> refPages = new HashSet<>();
        
        Set<Page> odfRootPages = _odfPageHandler.getOdfRootPages(siteName, lang);
        
        for (Page rootPage : odfRootPages)
        {
            // Ignore ODF root pages that does not belong to the same catalog as the program item
            if (_odfPageHandler.getCatalog(rootPage).equals(programItem.getCatalog()))
            {
                if (programItem instanceof Program program)
                {
                    ProgramPage programPage = getProgramPage(rootPage, program);
                    if (programPage != null)
                    {
                        refPages.add(programPage);
                    }
                }
                else if (programItem instanceof SubProgram subProgram)
                {
                    Set<Program> parentPrograms = _odfHelper.getParentPrograms(programItem);
                    for (Program parentProgram : parentPrograms)
                    {
                        ProgramPage subProgramPage = getSubProgramPage(rootPage, subProgram, parentProgram);
                        if (subProgramPage != null)
                        {
                            refPages.add(subProgramPage);
                        }
                    }
                }
                else if (programItem instanceof Course course)
                {
                    List<CourseList> parentCourseLists = course.getParentCourseLists();
                    for (CourseList courseList : parentCourseLists)
                    {
                        List<Course> parentCourses = courseList.getParentCourses();
                        for (Course parentCourse : parentCourses)
                        {
                            CoursePage coursePage = getCoursePage(rootPage, course, parentCourse);
                            if (coursePage != null)
                            {
                                refPages.add(coursePage);
                            }
                        }
                        
                        List<AbstractProgram> parentAbstractPrograms = getNearestAncestorAbstractPrograms(courseList);
                        for (AbstractProgram parentAbstractProgram : parentAbstractPrograms)
                        {
                            CoursePage coursePage = getCoursePage(rootPage, course, parentAbstractProgram);
                            if (coursePage != null)
                            {
                                refPages.add(coursePage);
                            }
                        }
                    }
                }
            }
        }
        
        return refPages;
    }
    
    /**
     * Return the program page
     * @param program the program
     * @return the page program or null
     */
    public ProgramPage getProgramPage(Program program)
    {
        return getProgramPage(program, null);
    }
    
    /**
     * Return the program page
     * @param program the program
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the page program or null
     */
    public ProgramPage getProgramPage(Program program, String siteName)
    {
        Page odfRootPage = getOdfRootPage(siteName, program.getLanguage(), program.getCatalog());
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getProgramPage(odfRootPage, program);
    }
    
    /**
     * Return the program page
     * @param odfRootPage the odf root page
     * @param program the program
     * @return the page program or null
     */
    public ProgramPage getProgramPage (Page odfRootPage, Program program)
    {
        // E.g: program://_root?rootId=xxxx&programId=xxxx
        String pageId =  "program://_root?rootId=" + odfRootPage.getId() + "&programId=" + program.getId();
        try
        {
            return _ametysResolver.resolveById(pageId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Return the subprogram page
     * @param subProgram the subprogram
     * @param parentProgram The parent program
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the subprogram page or null
     */
    public ProgramPage getSubProgramPage(SubProgram subProgram, AbstractProgram parentProgram, String siteName)
    {
        Page odfRootPage = getOdfRootPage(siteName, subProgram.getLanguage(), subProgram.getCatalog());
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getSubProgramPage(odfRootPage, subProgram, parentProgram);
    }
    
    /**
     * Return the subprogram page
     * @param odfRootPage the odf root page
     * @param subProgram the subprogram
     * @param parentAbstractProgram The parent program or subprogram
     * @return the subprogram page or null
     */
    public ProgramPage getSubProgramPage (Page odfRootPage, SubProgram subProgram, AbstractProgram parentAbstractProgram)
    {
        try
        {
            // Get first parent program matching an existing program page
            Program parentProgram = getRootProgram(odfRootPage, subProgram, parentAbstractProgram);
            if (parentProgram == null)
            {
                // No program page
                return null;
            }
            
            AbstractProgram nearestParentAbstractProgram = Optional.ofNullable(parentAbstractProgram).orElse(parentProgram);
            AbstractProgram parent = getNearestAncestorAbstractProgram(subProgram, nearestParentAbstractProgram);
            if (parent == null)
            {
                return null; // no page
            }
         
            String path = getPathInProgram(parent, parentProgram);
            if (path == null)
            {
                // Subprogram is not part of the selected parent program
                return null;
            }
            return _programPageFactory.createProgramPage(odfRootPage, subProgram, path, parentProgram, null);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Returns the subprogram page
     * @param odfRootPage the odf root page
     * @param subProgram the subprogram
     * @param path a full or partial path of subprogram
     * @param checkHierarchy set to true to check that the given path is a valid hierarchy
     * @return the subprogram page or null if not found
     */
    public ProgramPage getSubProgramPage(Page odfRootPage, SubProgram subProgram, List<String> path, boolean checkHierarchy)
    {
        // Possible paths are :
        // [subprogramContent://UUID, subprogramContent://UUID, subprogramContent://UUID]
        // [programContent://UUID, subprogramContent://UUID, subprogramContent://UUID]
        
        try
        {
            
            ProgramItem lastParent = _ametysResolver.resolveById(path.get(0));
            Program parentProgram = getRootProgram(odfRootPage, lastParent, null);
            if (parentProgram == null)
            {
                // No page
                return null;
            }
            
            List<String> reversedPath = path.reversed();
            if (checkHierarchy)
            {
                ProgramItem nextProgramItem = _ametysResolver.resolveById(path.getLast());
                if (!_checkHierarchy(subProgram, nextProgramItem))
                {
                    // Parent item in path is not part of program item's parents
                    getLogger().warn(reversedPath + " is not valid hierarchy");
                    throw new UnknownAmetysObjectException(reversedPath + " is not valid hierarchy");
                }
            }
            
            String pagePath = _resolvePagePath(reversedPath, checkHierarchy);
            if (pagePath == null)
            {
                // No page
                return null;
            }
            
            return _programPageFactory.createProgramPage(odfRootPage, subProgram, pagePath, parentProgram, null);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Return the course page
     * @param course the course
     * @param parentProgram the parent program or subprogram. Can be null.
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage(Course course, AbstractProgram parentProgram, String siteName)
    {
        String catalog = course.getCatalog();
        Page odfRootPage = getOdfRootPage(siteName, course.getLanguage(), catalog);
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getCoursePage(odfRootPage, course, parentProgram);
    }
    
    /**
     * Return the course page
     * @param odfRootPage the odf root page
     * @param course the course
     * @param parentAbstractProgram the parent program or subprogram. Can be null.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage (Page odfRootPage, Course course, AbstractProgram parentAbstractProgram)
    {
        try
        {
            // Get first parent program matching an existing program page
            Program parentProgram = getRootProgram(odfRootPage, course, parentAbstractProgram);
            if (parentProgram == null)
            {
                // No program page
                return null;
            }
            
            AbstractProgram nearestParentAbstractProgram = Optional.ofNullable(parentAbstractProgram).orElse(parentProgram);
            
            ProgramItem parent = null;
            
            Course parentCourse = getNearestAncestorCourse(course, nearestParentAbstractProgram);
            if (parentCourse != null)
            {
                parent = parentCourse;
            }
            else
            {
                parent = getNearestAncestorAbstractProgram(course, nearestParentAbstractProgram);
            }
            
            if (parent == null)
            {
                return null; // no page
            }
         
            String path = getPathInProgram(parent, parentProgram);
            if (path == null)
            {
                // Course is not part of the selected parent program
                return null;
            }
            
            return _coursePageFactory.createCoursePage(odfRootPage, course, parentProgram, path, null);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Return the course page
     * @param course the course
     * @param parentCourse the parent course. Can NOT be null.
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage (Course course, Course parentCourse, String siteName)
    {
        String catalog = course.getCatalog();
        Page odfRootPage = getOdfRootPage(siteName, course.getLanguage(), catalog);
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getCoursePage(odfRootPage, course, parentCourse);
    }
    
    /**
     * Return the course page
     * @param odfRootPage the odf root page
     * @param course the course
     * @param parentCourse the parent course. Can NOT be null.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage (Page odfRootPage, Course course, Course parentCourse)
    {
        try
        {
            // Get first parent program matching an existing program page
            Program parentProgram = getRootProgram(odfRootPage, parentCourse, null);
            if (parentProgram == null)
            {
                // No page
                return null;
            }
         
            String path = getPathInProgram(parentCourse, parentProgram);
            if (path == null)
            {
                // Course is not part of the selected parent program
                return null;
            }
            return _coursePageFactory.createCoursePage(odfRootPage, course, parentProgram, path, null);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Returns the course page in given ODF root page,
     * @param odfRootPage the odf root page
     * @param course the course
     * @param path a (partial) education path or a (partial) sitemap path. Be careful, assume that the given path correspond to a valid path in ODF root. Use {@link #getCoursePage(Page, Course, List, boolean)} with true if not sure.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage(Page odfRootPage, Course course, List<String> path)
    {
        return getCoursePage(odfRootPage, course, path, false);
    }
    
    /**
     * Returns the course page in given ODF root page,
     * @param odfRootPage the odf root page
     * @param course the course
     * @param path a (partial) education path or a (partial) sitemap path
     * @param checkHierarchy set to true to check that the given path correspond to a valid hierarchy
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage(Page odfRootPage, Course course, List<String> path, boolean checkHierarchy)
    {
        // Possible paths are :
        // [(sub)programContent://UUID, container://UUID, courseContent://UUID1, courseContent://UUID2]
        // [courseContent://UUID1, courseContent://UUID2, ..., courseContent://UUID3]
        // [subprogramContent://UUID2, ..., (sub)programContent://UUID3, courseContent://UUID1]
        
        try
        {
            // Get first parent program matching an existing program page
            ProgramItem lastParent = _ametysResolver.resolveById(path.get(0));
            Program rootProgram = getRootProgram(odfRootPage, lastParent, null);
            if (rootProgram == null)
            {
                // No page
                return null;
            }
            
            List<String> reversedPath = path.reversed();
            if (checkHierarchy)
            {
                ProgramItem nextProgramItem = _ametysResolver.resolveById(path.getLast());
                if (!_checkHierarchy(course, nextProgramItem))
                {
                    // Parent item in path is not part of program item's parents
                    getLogger().warn(reversedPath + " is not valid hierarchy");
                    throw new UnknownAmetysObjectException(reversedPath + " is not valid hierarchy");
                }
            }
            
            String pagePath = _resolvePagePath(reversedPath, checkHierarchy);
            if (pagePath == null)
            {
                // No page
                return null;
            }
            
            return _coursePageFactory.createCoursePage(odfRootPage, course, rootProgram, pagePath, null);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Determines if a program page exists
     * @param odfRootPage the ODF root page
     * @param program the program page
     * @return true if program page existes, false otherwise
     */
    public boolean isProgramPageExist(Page odfRootPage, Program program)
    {
        if (program == null || !_odfPageHandler.isValidRestriction(odfRootPage, program))
        {
            // No program page
            return false;
        }
        
        String levelsPath = _odfPageHandler.computeLevelsPath(odfRootPage, program);
        if (levelsPath == null)
        {
            // The current program has no valid attributes for the levels selected in the ODF root
            return false;
        }
        
        return true;
    }
    
    /**
     * Get the ODF root page, either in the given site if it exists, or in the default ODF site.
     * @param siteName the desired site name.
     * @param language the sitemap language to search in.
     * @param catalog The ODF catalog
     * @return the ODF root page, either in the given site if it exists, or in the default ODF site.
     */
    public Page getOdfRootPage(String siteName, String language, String catalog)
    {
        Page odfRootPage = null;
        
        if (StringUtils.isNotEmpty(siteName))
        {
            odfRootPage = _odfPageHandler.getOdfRootPage(siteName, language, catalog);
        }
        
        if (odfRootPage == null)
        {
            String odfSiteName = Config.getInstance().getValue("odf.web.site.name");
            odfRootPage = _odfPageHandler.getOdfRootPage(odfSiteName, language, catalog);
        }
        
        return odfRootPage;
    }
    
    private String _resolvePagePath(List<String> reversedPath, boolean checkHierarchy)
    {
        // Possible reversed paths are :
        // [courseContent://UUID1, courseContent://UUID2, ..., courseContent://UUID3, (sub)programContent://UUID]
        // [courseContent://UUID1, courseContent://UUID2, ..., courseContent://UUID3]
        // [courseContent://UUID1, subprogramContent://UUID2, ..., (sub)programContent://UUID3]
        
        Cache<String, String> cache = _cacheManager.get(__RESOLVED_PATH_CACHE);
        return cache.get(StringUtils.join(reversedPath, ";"), item -> {
            String pagePath = null;
            
            try
            {
                if (reversedPath.size() == 1)
                {
                    ProgramItem programItem = _ametysResolver.resolveById(reversedPath.get(0));
                    pagePath = getPathInProgram(programItem, null);
                }
                else
                {
                    String parentPath = _resolvePagePath(reversedPath.subList(1, reversedPath.size()), checkHierarchy);
                    if (parentPath != null)
                    {
                        ProgramItem programItem = _ametysResolver.resolveById(reversedPath.get(0));
                        if (checkHierarchy)
                        {
                            // Get next parent given in path
                            ProgramItem nextProgramItem = _ametysResolver.resolveById(reversedPath.get(1));
                            // Check that next parent is a parent item
                            if (!_checkHierarchy(programItem, nextProgramItem))
                            {
                                // Parent item in path is not part of program item's parents
                                getLogger().warn(reversedPath + " is not valid hierarchy");
                                throw new UnknownAmetysObjectException(reversedPath + " is not valid hierarchy");
                            }
                        }
                        if (programItem instanceof AbstractProgram || programItem instanceof Course)
                        {
                            parentPath += '/' + _odfPageHandler.getPageName(programItem);
                        }
                        
                        pagePath = parentPath;
                    }
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // Nothing
            }
            
            cache.put(StringUtils.join(reversedPath, ";"), pagePath);
            return pagePath;
        });
    }
    
    private boolean _checkHierarchy(ProgramItem childProgramItem, ProgramItem nextParentProgramItem)
    {
        List<ProgramItem> parentProgramItems = _odfHelper.getParentProgramItems(childProgramItem);
        if (parentProgramItems.contains(nextParentProgramItem))
        {
            return true;
        }
        
        for (ProgramItem parentProgramItem : parentProgramItems)
        {
            if (!(parentProgramItem instanceof AbstractProgram) && !(parentProgramItem instanceof Course) && _checkHierarchy(parentProgramItem, nextParentProgramItem))
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get the path in sitemap of a ODF content into a {@link Program} or {@link SubProgram}<br>
     * Be careful, this is the path in sitemap, to get the path of a item into a Program, use {@link ODFHelper#getPathInProgram} instead.
     * @param programItem The program item
     * @param parentProgram The parent root (sub)program. Can be null.
     * @return the path in sitemap from the parent program or null if no found ODF path for those program item and parent program
     */
    public String getPathInProgram (ProgramItem programItem, AbstractProgram parentProgram)
    {
        Cache<PathInProgramCacheKey, String> cache = _cacheManager.get(__PATH_IN_SITEMAP_CACHE);
        
        return cache.get(PathInProgramCacheKey.of(programItem.getId(), parentProgram != null ? parentProgram.getId() : "__NOPARENT"), item -> {
            
            if (programItem instanceof Program || programItem.equals(parentProgram))
            {
                // The program item is already the program it self
                return _odfPageHandler.getPageName(programItem);
            }
            
            List<String> paths = new ArrayList<>();

            // Add the parent path in program if exists
            ProgramItem parent = _odfHelper.getParentProgramItem(programItem, parentProgram);
            if (parent != null)
            {
                String parentPath = getPathInProgram(parent, parentProgram);
                if (parentPath != null)
                {
                    paths.add(parentPath);
                }
            }
            
            // Add the current page name (if it is an item with a page (only course, subprogram and program)
            if (programItem instanceof AbstractProgram || programItem instanceof Course)
            {
                paths.add(_odfPageHandler.getPageName(programItem));
            }

            // If the path is empty, return null
            return paths.isEmpty() ? null : StringUtils.join(paths, "/");
        });
    }
    
    private static class RootProgramCacheKey extends AbstractCacheKey
    {
        protected RootProgramCacheKey(Page odfRootPage, String programItemId, String parentProgramId)
        {
            super(odfRootPage, programItemId, parentProgramId);
        }
        
        public static RootProgramCacheKey of(Page odfRootPage, String programItemId, String parentProgramId)
        {
            return new RootProgramCacheKey(odfRootPage, programItemId, parentProgramId);
        }
    }
    
    private static class PathInProgramCacheKey extends AbstractCacheKey
    {
        protected PathInProgramCacheKey(String programItemId, String parentProgramId)
        {
            super(programItemId, parentProgramId);
        }
        
        public static PathInProgramCacheKey of(String programItemId, String parentProgramId)
        {
            return new PathInProgramCacheKey(programItemId, parentProgramId);
        }
    }
    
    /**
     * Returns the first {@link Program} ancestor matching an existing {@link ProgramPage} in given ODF root page, ensuring that the given parent content 'parentProgram' is in the hierarchy (if not null)<br>
     * If 'parentProgram' is null, the first {@link Program} ancestor will be returned regardless of parent hierarchy.<br>
     * If 'parentProgram' is a {@link SubProgram}, the first {@link Program} ancestor from this {@link SubProgram} will be returned regardless of parent hierarchy
     * @param odfRootPage The ODf root page. Cannot be null.
     * @param programItem a {@link ProgramItem}
     * @param parentAbstractProgram The parent program or subprogram. Can be null.
     * @return the parent {@link Program} into this (sub)program that matchs an existing program page, or null if not found
     */
    public Program getRootProgram(Page odfRootPage, ProgramItem programItem, AbstractProgram parentAbstractProgram)
    {
        Cache<RootProgramCacheKey, Program> rootCache = _cacheManager.get(__ODF_ROOT_PROGRAM_CACHE);
        
        return rootCache.get(RootProgramCacheKey.of(odfRootPage, programItem.getId(), parentAbstractProgram != null ? parentAbstractProgram.getId() : "__NOPARENT"), k -> {
            // Get all parent programs
            Set<Program> parentPrograms = _getParentPrograms(programItem, parentAbstractProgram);
            
            // Get first parent program matching an existing program page
            Optional<Program> parentProgram = parentPrograms.stream()
                    .filter(p -> isProgramPageExist(odfRootPage, p))
                    .findFirst();
            
            return parentProgram.orElse(null);
        });
    }
    
    /**
     * Returns all parent {@link Program} ancestors, ensuring that the given parent content 'parentProgram' is in the hierarchy, if not null.<br>
     * If 'parentProgram' is null, the all {@link Program} ancestors will be returned regardless of parent hierarchy.<br>
     * If 'parentProgram' is a {@link SubProgram}, the {@link Program} ancestors from this {@link SubProgram} will be returned regardless of parent hierarchy
     * @param programItem a {@link ProgramItem}
     * @param parentProgram The parent program or subprogram. Can be null.
     * @return the parent {@link Program}s into this (sub)program or empty if not found
     */
    private Set<Program> _getParentPrograms(ProgramItem programItem, AbstractProgram parentProgram)
    {
        if (programItem instanceof Program program)
        {
            return Set.of(program);
        }
        
        Set<Program> parentPrograms = new HashSet<>();
        
        AbstractProgram parent = parentProgram;
        
        List<ProgramItem> parentItems = _odfHelper.getParentProgramItems(programItem, parentProgram);
        
        for (ProgramItem parentItem : parentItems)
        {
            if (parentItem instanceof Program program)
            {
                parentPrograms.add(program);
            }
            else
            {
                if (parent != null && parentItem.equals(parent))
                {
                    // Once the desired abstract program parent is passed, the parent is null
                    parent = null;
                }
                
                parentPrograms.addAll(_getParentPrograms(parentItem, parent));
            }
        }
        
        return parentPrograms;
    }
    
    /**
     * Returns the nearest {@link AbstractProgram} ancestors.
     * @param programPart a {@link ProgramPart}
     * @return the nearest {@link AbstractProgram} ancestors containing this program part
     */
    public List<AbstractProgram> getNearestAncestorAbstractPrograms (ProgramPart programPart)
    {
        List<AbstractProgram> ancestors = new ArrayList<>();
        
        List<ProgramPart> parents = programPart.getProgramPartParents();
        for (ProgramPart parent : parents)
        {
            if (parent instanceof AbstractProgram)
            {
                ancestors.add((AbstractProgram) parent);
            }
            else
            {
                ancestors.addAll(getNearestAncestorAbstractPrograms(parent));
            }
        }
        
        return ancestors;
    }
    
    /**
     * Returns the nearest {@link AbstractProgram} ancestor.
     * @param programItem a {@link ProgramItem}
     * @param parentProgram The parent program or subprogram
     * @return the nearest {@link AbstractProgram} ancestor into this (sub)program or null if not found
     */
    public AbstractProgram getNearestAncestorAbstractProgram (ProgramItem programItem, AbstractProgram parentProgram)
    {
        ProgramItem parentItem = _odfHelper.getParentProgramItem(programItem, parentProgram);
        while (parentItem != null && !(parentItem instanceof AbstractProgram))
        {
            parentItem = _odfHelper.getParentProgramItem(parentItem, parentProgram);
        }
        
        return parentItem != null ? (AbstractProgram) parentItem : null;
    }
    
    /**
     * Returns the nearest {@link Course} ancestor.
     * @param course a {@link Course}
     * @param parentProgram The parent program or subprogram
     * @return the nearest {@link Course} ancestor into this (sub)program or null if not found
     */
    public Course getNearestAncestorCourse (Course course, AbstractProgram parentProgram)
    {
        ProgramItem parentItem = _odfHelper.getParentProgramItem(course, parentProgram);
        while (parentItem != null && !(parentItem instanceof Course) && !(parentItem instanceof AbstractProgram))
        {
            parentItem = _odfHelper.getParentProgramItem(parentItem, parentProgram);
        }
        
        return parentItem != null && parentItem instanceof Course ? (Course) parentItem : null;
    }
    
    /**
     * Get the path of a {@link ProgramItem} page into the given {@link Program}
     * @param siteName the site name
     * @param language the language
     * @param programItem the subprogram.
     * @param parentProgram The parent program
     * @return the page path or empty if no page matches
     */
    public String getProgramItemPagePath(String siteName, String language, ProgramItem programItem, Program parentProgram)
    {
        Page rootPage = _odfPageHandler.getOdfRootPage(siteName, language, programItem.getCatalog());
        if (rootPage != null)
        {
            Page page = null;
            if (programItem instanceof Program program)
            {
                page = getProgramPage(rootPage, program);
            }
            else if (programItem instanceof SubProgram subProgram)
            {
                page = getSubProgramPage(rootPage, subProgram, parentProgram);
            }
            else if (programItem instanceof Course course)
            {
                page = getCoursePage(rootPage, course, parentProgram);
            }
            
            if (page != null)
            {
                return rootPage.getSitemapName() + "/" + page.getPathInSitemap();
            }
        }
        
        return StringUtils.EMPTY;
    }
}
