/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.tree.OdfClassificationHandler;
import org.ametys.plugins.odfweb.repository.FirstLevelPageFactory;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;

/**
 * This observer clears the cache of enumerated {@link OrgUnit} values or {@link Content}s which are {@link ContentHelper#isReferenceTable(Content) part of reference table}
 * when an {@link OrgUnit} or reference table content is added, modified, moved or deleted.
 */
public class ClearLevelCacheObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    private ContentHelper _contentHelper;
    private OdfClassificationHandler _odfClassificationHandler;
    private AmetysObjectResolver _resolver;
    private OdfPageHandler _odfPageHandler;
    private ContentTypeExtensionPoint _cTypeEP;
    private OdfReferenceTableHelper _odfReferenceTableHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _odfClassificationHandler = (OdfClassificationHandler) manager.lookup(OdfClassificationHandler.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfPageHandler = (OdfPageHandler) manager.lookup(OdfPageHandler.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _odfReferenceTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        String eventId = event.getId();
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        return (eventId.equals(ObservationConstants.EVENT_CONTENT_DELETING)
                || eventId.equals(ObservationConstants.EVENT_CONTENT_MODIFIED/* handles moved orgUnit too */)
                || eventId.equals(ObservationConstants.EVENT_CONTENT_ADDED))
                && content != null && (content instanceof OrgUnit || _contentHelper.isReferenceTable(content));
    }
    
    @Override
    public int getPriority()
    {
        return MAX_PRIORITY;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        if (content instanceof OrgUnit)
        {
            _odfClassificationHandler.clearLevelValues(ProgramItem.ORG_UNITS_REFERENCES, content.getLanguage());
        }
        else
        {
            AmetysObjectIterable<Page> rootPages = _getODFRootPages();
            if (rootPages.getSize() == 0)
            {
                getLogger().debug("There is no ODF root page, nothing to invalidate");
                return;
            }
            
            if (_odfReferenceTableHelper.isTableReferenceEntry(content))
            {
                // The target content is an entry of a reference table
                for (Page rootPage : rootPages)
                {
                    String level1Attribute = _odfPageHandler.getLevel1Metadata(rootPage);
                    if (_isPartOfClassification(content, level1Attribute))
                    {
                        _odfClassificationHandler.clearLevelValues(level1Attribute, null);
                    }
                    
                    String level2Attribute = _odfPageHandler.getLevel2Metadata(rootPage);
                    if (_isPartOfClassification(content, level2Attribute))
                    {
                        _odfClassificationHandler.clearLevelValues(level2Attribute, null);
                    }
                }
                
            }
        }
    }
    
    private boolean _isPartOfClassification(Content targetContent, String attributName)
    {
        if (StringUtils.isNotEmpty(attributName))
        {
            ContentType programCType = _cTypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
            
            if (programCType.hasModelItem(attributName))
            {
                ModelItem modelItem = programCType.getModelItem(attributName);
                
                if (modelItem instanceof ContentAttributeDefinition)
                {
                    return ArrayUtils.contains(targetContent.getTypes(), ((ContentAttributeDefinition) modelItem).getContentTypeId());
                }
            }
        }
        
        return false;
    }
    
    /**
     * Get the ODF root pages
     * @return the ODF root pages
     */
    protected AmetysObjectIterable<Page> _getODFRootPages()
    {
        Expression expression = new VirtualFactoryExpression(FirstLevelPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, expression, null);
        
        return _resolver.query(query);
    }
}
