/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * This controller is used for the shareable course workflow buttons
 */
Ext.define('Ametys.plugins.odf.course.controllers.ShareableCourseStatusButtonController', {
    extend: 'Ametys.plugins.cms.content.controller.SmartContentController',
    
    /**
	 * @cfg {String} shareable-course-status The shareable course status of the button (none, proposed, validated)
	 */
    /**
     * @property {String} status See #cfg-shareable-course-status
     */
    
    constructor: function(config)
    {
        this.callParent(arguments);
        
        this.status = this.getInitialConfig("shareable-course-status");
		
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
    },
    
    /**
     * @private
     * Listener handler for modified messages
     * @param {Ametys.message.Message} message the message
     */
    _onModified: function(message)
    {
        if (this.updateTargetsInCurrentSelectionTargets(message))
        {
            this.refresh();
        }
    },
    
    /**
	 * Refresh the controller from the content informations state of given targets
	 * @param targets The content targets
	 * @protected
	 */
	_getStatus: function (targets)
	{
		this.callParent(arguments);
		
		if (!this.isDisabled())
		{
			var isActiveButton = false;
			var contentTitles = [];
			Ext.Array.each(targets, function(matchingTarget)
				{
					var parameters = matchingTarget.getParameters();
					if (parameters && parameters.content && parameters.additionalData)
	                {
						var content = parameters.content;
						var additionalData = parameters.additionalData;
						var contentStatus = additionalData.shareableCourseStatus;
						if (contentStatus != null && this._isActiveButton(this.status, contentStatus))
						{
							isActiveButton = true;
							contentTitles.push(content.getTitle());
						}
	                }
				},
				this
			);
			
			var description = "";
			if (isActiveButton)
			{
				description += this.getInitialConfig("allright-start-description");
	    		description += "<br/>";
	    		description += this._getContentTitles(contentTitles);
	    		description += this.getInitialConfig("allright-end-description");
			}
			else
			{
				description += this.getInitialConfig("workflowaction-disable-description");
				this.disable();
			}
			
			this.setDescription(description);
		}
	},
	
	/**
     * @private
     * True if the button is active for the content
     * @param {String} buttonStatus The button status
     * @param {String} contentStatus The content status
     */
	_isActiveButton: function(buttonStatus, contentStatus)
	{
		if (buttonStatus == "NONE")
		{
			return contentStatus != "NONE";
		}
		else if (buttonStatus == "PROPOSED")
		{
			return contentStatus == "NONE" || contentStatus == "REFUSED";
		}
		else if (buttonStatus == "VALIDATED")
		{
			return contentStatus != "VALIDATED";
		}
	},
    
    /**
     * @private
     * Get concerned content titles
     * @param {String[]} contentTitles The contents' titles
     */
    _getContentTitles: function(contentTitles)
    {
		var contentTitlesAsString = "";
		Ext.Array.each(contentTitles, function(title)
			{
				if (contentTitlesAsString != "")
				{
					contentTitlesAsString += ", '" + title + "'";
			 	}
				else
				{
					contentTitlesAsString += "'" + title + "'";
				}
			}
    	);
		
		return contentTitlesAsString;
    }
});
