/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.odf.program.Program;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.util.AmetysHomeHelper;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Generate and put CDM-fr file in an output folder
 *
 */
public class DepositCDMFRFunction extends AbstractContentWorkflowComponent implements EnhancedFunction, Initializable
{
    private File _outputFolder;
    private boolean _isActive;
    private SourceResolver _sourceResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue("odf.publish.cdm-fr.output.folder");
        
        if (_isActive)
        {
            _outputFolder = new File(AmetysHomeHelper.getAmetysHomeData(), "/odf/cdmfr");
            FileUtils.forceMkdir(_outputFolder);
        }
    }

    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        if (!_isActive || _outputFolder == null)
        {
            // Do nothing
            return;
        }
        
        // Retrieve current content
        Program program = (Program) getContent(transientVars);
        
        try
        {
            // Generate CDM-FR file
            Source cdmfrSource = _sourceResolver.resolveURI("cocoon://_plugins/odf/export-cdmfr.xml?id=" + program.getId() + "&" + ExportCDMfrHelper.REQUEST_PARAM_VALID_LABEL + "=true");

            // Save file
            try (InputStream is = cdmfrSource.getInputStream())
            {
                String filename = program.getCDMId() + ".xml";
                
                // Delete existing file
                File file = new File(_outputFolder, filename);
                if (file.exists())
                {
                    file.delete();
                }
             
                try (OutputStream os = new FileOutputStream(file))
                {
                    IOUtils.copy(is, os);
                }
            }
        }
        catch (IOException e)
        {
            addWorkflowWarning(transientVars, new I18nizableText("plugin.odf", "PLUGINS_ODF_PUBLISH_PROGRAM_CDMFR_ERROR", Collections.singletonList(program.getTitle())));
            _logger.error("Unable to generate and copy the CDM-fr file", e);
        }
    }
    
    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_DEPOSIT_CDMFR_FUNCTION_LABEL");
    }
}
