/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.oai;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.cdmfr.CDMEntity;
import org.ametys.odf.cdmfr.ExportCDMfrManager;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * Generator for the <code>GetRecord</code> verb.
 */
public class GetRecordGenerator extends AbstractOAIVerbGenerator implements Serviceable
{
    private RootOrgUnitProvider _rootOrgUnitProvider;
    private ExportCDMfrManager _exportCDMfrManager;
    private ODFHelper _odfHelper;
    private CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _rootOrgUnitProvider = (RootOrgUnitProvider) serviceManager.lookup(RootOrgUnitProvider.ROLE);
        _exportCDMfrManager = (ExportCDMfrManager) serviceManager.lookup(ExportCDMfrManager.ROLE);
        _odfHelper = (ODFHelper) serviceManager.lookup(ODFHelper.ROLE);
        _catalogsManager = (CatalogsManager) serviceManager.lookup(CatalogsManager.ROLE);
    }
    
    @Override
    protected Collection<String> getRequiredParameters()
    {
        return Arrays.asList("verb", "identifier", "metadataPrefix");
    }

    @Override
    protected Collection<String> getAllowedParameters()
    {
        return Arrays.asList("verb", "identifier", "metadataPrefix");
    }

    @Override
    protected void generateVerb() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String identifier = request.getParameter("identifier");
        String metadataPrefix = request.getParameter("metadataPrefix");
        
        if (!metadataPrefix.equals("cdm") && !metadataPrefix.equals("oai_dc"))
        {
            generateError("cannotDisseminateFormat", "The value of the metadataPrefix argument is not supported by the repository.");
            return;
        }
        
        String prefix = "FRUAI" + _rootOrgUnitProvider.getRoot().getUAICode() + "PR";
        String code = identifier.startsWith(prefix) ? identifier.substring(prefix.length()) : "";
        
        Expression cdmCodeExpr = new StringExpression(CDMEntity.CDM_CODE, Operator.EQ, identifier);
        Expression codeExpr = new StringExpression(ProgramItem.CODE, Operator.EQ, code);
        
        OrExpression codesExpr = new OrExpression(codeExpr, cdmCodeExpr);
        
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        AmetysObjectIterable<Program> programs = _odfHelper.getProgramItems(ProgramFactory.PROGRAM_CONTENT_TYPE, null, _catalogsManager.getDefaultCatalogName(), "fr", codesExpr, sortCriteria);
        
        if (!programs.iterator().hasNext())
        {
            generateError("idDoesNotExist", "The value of the identifier argument is unknown");
            return;
        }
        
        Program program = programs.iterator().next();
        
        try
        {
            String[] labels = program.getAllLabels();
            
            if (Arrays.asList(labels).contains(CmsConstants.LIVE_LABEL))
            {
                program.switchToLabel(CmsConstants.LIVE_LABEL);
                XMLUtils.startElement(contentHandler, "GetRecord");
                
                RecordHelper.saxProgram(program, contentHandler, metadataPrefix, _exportCDMfrManager);
                
                XMLUtils.endElement(contentHandler, "GetRecord");
            }
            else
            {
                // no "Live" label
                generateError("idDoesNotExist", "There is no Live version for the identifier");
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            // no "Live" label
            generateError("idDoesNotExist", "There is no Live version for the identifier argument");
        }
    }
}
