/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program.properties;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.List;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.type.ModelItemTypeExtensionPoint;
import org.ametys.cms.model.properties.AbstractProperty;
import org.ametys.cms.model.properties.Property;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ElementType;

/**
 * {@link Property} for attribute name of the parent {@link Program} of a {@link SubProgram}
 */
public class SubProgramParentProgramProperty extends AbstractProperty<Object, Content> implements Serviceable
{
    /** The ODF Helper */
    protected ODFHelper _odfHelper;
    
    /** The Content Type Extension Point */
    protected ContentTypeExtensionPoint _ctypeEP;
    
    /** The Content Attribute Type Extension Point */
    protected ModelItemTypeExtensionPoint _contentAttributeTypeEP;
    
    /** The program attribute name to index for subProgram */
    protected String _programAttributeName;
    
    /** The program attribute type to index for subProgram */
    protected String _programAttributeType;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentAttributeTypeEP = (ModelItemTypeExtensionPoint) manager.lookup(ModelItemTypeExtensionPoint.ROLE_CONTENT_ATTRIBUTE);
        _ctypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        _programAttributeType = configuration.getAttribute("type");
        if (StringUtils.isBlank(_programAttributeType))
        {
            throw new ConfigurationException("Attribute 'type' is mandatory for " + SubProgramParentProgramProperty.class.getName());
        }
        
        _programAttributeName = configuration.getChild("program-attribute-name").getValue();
        if (StringUtils.isBlank(_programAttributeName))
        {
            throw new ConfigurationException("Child 'program-attribute-name' is mandatory for " + SubProgramParentProgramProperty.class.getName());
        }
    }
    
    @Override
    public Object getValue(Content content)
    {
        if (content instanceof SubProgram subProgram)
        {
            List<Object> values = _odfHelper.getParentPrograms(subProgram)
                .stream()
                .map(this::_getValues)
                .flatMap(List::stream)
                .toList();
            
            ContentType programCType = _ctypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
            ModelItem modelItem = programCType.getModelItem(_programAttributeName);
            ElementType attributeType = (ElementType) _contentAttributeTypeEP.getExtension(modelItem.getType().getId());
            
            return !values.isEmpty() ? values.toArray((Object[]) Array.newInstance(attributeType.getManagedClass(), values.size())) : null;
        }
        
        return null;
    }
    
    private List<Object> _getValues(Program program)
    {
        if (program.isMultiple(_programAttributeName))
        {
            Object[] values = program.getValue(_programAttributeName);
            
            return values != null ? Arrays.asList(values) : List.of();
        }
        else
        {
            Object value = program.getValue(_programAttributeName);
            return value != null ? List.of(value) : List.of();
        }
    }
    
    @Override
    public boolean isMultiple()
    {
        return true;
    }
    
    @Override
    protected String _getTypeId()
    {
        return _programAttributeType;
    }
}
