/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.util.Set;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.odf.course.Course;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;

/**
 * Interface for CMDfr extension.
 */
public interface CDMfrExtension
{
    /**
     * Send the content of an user defined ametys extension to be included in the CDM-fr representation the abstract program.<br>
     * @param contentHandler the receiving contentHandler.
     * @param program the abstract program (common for program and subprogram)
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void abstractProgram2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program, Set<String> persons, Set<String> orgUnits) throws SAXException;
    
    /**
     * Send the content of an user defined ametys extension to be included in the CDM-fr representation the program.<br>
     * @param contentHandler the receiving contentHandler.
     * @param program the program
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void program2CDM(ContentHandler contentHandler, Program program, Set<String> persons, Set<String> orgUnits) throws SAXException;
    
    /**
     * Send the content of an user defined ametys extension to be included in the CDM-fr representation the subProgram.<br>
     * @param contentHandler the receiving contentHandler.
     * @param subProgram the subProgram
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void subProgram2CDM(ContentHandler contentHandler, SubProgram subProgram, Set<String> persons, Set<String> orgUnits) throws SAXException;
    
    /**
     * Send the content of an user defined ametys extension to be included in the CDM-fr representation the course.<br>
     * @param contentHandler the receiving contentHandler.
     * @param course the course
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void course2CDM(ContentHandler contentHandler, Course course, Set<String> persons, Set<String> orgUnits) throws SAXException;
    
    /**
     * Send the content of an user defined ametys extension to be included in the CDM-fr representation the orgunit.<br>
     * @param contentHandler the receiving contentHandler.
     * @param orgunit the orgunit
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void orgunit2CDM(ContentHandler contentHandler, OrgUnit orgunit) throws SAXException;
    
    /**
     * Send the content of an user defined ametys extension to be included in the CDM-fr representation the orgunit.<br>
     * @param contentHandler the receiving contentHandler.
     * @param person the orgunit
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void person2CDM(ContentHandler contentHandler, Person person) throws SAXException;
    
    /**
     * Send the content of an user defined ametys extension to be included in the CDM-fr representation the container.<br>
     * @param contentHandler the receiving contentHandler.
     * @param container the container
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void container2CDM(ContentHandler contentHandler, Container container, Set<String> persons, Set<String> orgUnits) throws SAXException;
}
