/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.enumeration;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.ModelItem;

/**
 * Generate the values of an enumerated attribute of the program content type.
 */
public class EnumeratedAttributeValuesGenerator extends ServiceableGenerator
{
    /** The name of the parameter containing the path of the enumerated attribute */
    protected static final String ATTRIBUTE_PATH_PARAMETER_NAME = "attributePath";
    
    /** The content type extension point. */
    protected ContentTypeExtensionPoint _cTypeEP;
    /** The ODf enumeration helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    /** The Ametys resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) serviceManager.lookup(OdfReferenceTableHelper.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        ContentType programCType = _cTypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
        
        String attributePath = parameters.getParameter(ATTRIBUTE_PATH_PARAMETER_NAME, StringUtils.EMPTY);
        
        contentHandler.startDocument();
        
        try
        {
            ElementDefinition elementDefinition = _getElementDefinition(programCType, attributePath);
            if (elementDefinition != null)
            {
                Enumerator enumerator = elementDefinition.getEnumerator();
                if (enumerator != null)
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("metadataName", elementDefinition.getName());
                    attrs.addCDATAAttribute("metadataPath", elementDefinition.getPath());
                    XMLUtils.startElement(contentHandler, "items", attrs);
                    
                    Map<Object, I18nizableText> entries = enumerator.getTypedEntries();
                    for (Object code : entries.keySet())
                    {
                        String codeStr = code.toString();
                        if (StringUtils.isNotEmpty(codeStr))
                        {
                            I18nizableText value = entries.get(code);
                            
                            AttributesImpl valueAttrs = new AttributesImpl();
                            valueAttrs.addCDATAAttribute("code", codeStr);
                            
                            XMLUtils.startElement(contentHandler, "item", valueAttrs);
                            value.toSAX(contentHandler);
                            XMLUtils.endElement(contentHandler, "item");
                        }
                    }
                    
                    XMLUtils.endElement(contentHandler, "items");
                }
                else if (elementDefinition instanceof ContentAttributeDefinition)
                {
                    ContentAttributeDefinition contentAttributeDefinition = (ContentAttributeDefinition) elementDefinition;
                    if (_odfRefTableHelper.isTableReference(contentAttributeDefinition.getContentTypeId()))
                    {
                        String lang = parameters.getParameter("lang", null);
                        
                        _odfRefTableHelper.saxItems(contentHandler, contentAttributeDefinition, lang);
                    }
                    else if (StringUtils.isNotEmpty(contentAttributeDefinition.getContentTypeId()))
                    {
                        _saxContents(contentAttributeDefinition);
                    }
                }
            }
            else
            {
                // no attribute definition found or no attribute path has been defined
                XMLUtils.startElement(contentHandler, "items");
                XMLUtils.endElement(contentHandler, "items");
            }
            
        }
        catch (Exception e)
        {
            throw new ProcessingException("Error retrieving the values of metadata '" + attributePath + "'", e);
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * Get the element definition from its path.
     * @param cType the content type.
     * @param attributePath the attribute path.
     * @return the element definition or null if not found.
     */
    protected ElementDefinition _getElementDefinition(ContentType cType, String attributePath)
    {
        if (StringUtils.isNotEmpty(attributePath))
        {
            if (cType.hasModelItem(attributePath))
            {
                ModelItem modelItem = cType.getModelItem(attributePath);
                if (modelItem instanceof ElementDefinition)
                {
                    return (ElementDefinition) modelItem;
                }
                else
                {
                    getLogger().warn("The attribute path '" + attributePath + "' does refers an element definition");
                }
            }
            else
            {
                getLogger().warn("There is no attribute defined at path '" + attributePath + "' for content type '" + cType.getId() + "'.");
            }
            
        }
        
        // no attribute definition found or no attribute path has been defined
        return null;
    }
    
    /**
     * SAX contents of the type defined by the attribute definition
     * @param attributeDefinition The attribute definition 
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxContents(ContentAttributeDefinition attributeDefinition) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("metadataName", attributeDefinition.getName());
        attrs.addCDATAAttribute("metadataPath", attributeDefinition.getPath());
        attrs.addCDATAAttribute("contentTypeId", attributeDefinition.getContentTypeId());
        XMLUtils.startElement(contentHandler, "items", attrs);
        
        ContentTypeExpression cTypeExpr = new ContentTypeExpression(Operator.EQ, attributeDefinition.getContentTypeId());
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(cTypeExpr);
        AmetysObjectIterable<Content> contents = _resolver.query(xpathQuery);
        
        for (Content content : contents)
        {
            AttributesImpl valueAttrs = new AttributesImpl();
            valueAttrs.addCDATAAttribute("id", content.getId());
            XMLUtils.createElement(contentHandler, "item", valueAttrs, content.getTitle());
        }
        
        XMLUtils.endElement(contentHandler, "items");
    }
    
}
