/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow.copy;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.workflow.copy.CreateContentByCopyFunction;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.workflow.copy.CopyODFContentClientSideElement.DuplicationMode;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue.Mode;

import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow function to create a program item by copy of another
 */
public abstract class AbstractCreateProgramItemByCopyFunction extends AbstractCreateODFContentByCopyFunction
{
    @Override
    protected Map<String, Object> getAdditionalCopyMap(Map transientVars, Content baseContent, String viewName, String fallbackViewName) throws WorkflowException
    {
        Map<String, Object> additionalCopyMap = new HashMap<>();
        additionalCopyMap.put(ProgramItem.CATALOG, null);
        
        @SuppressWarnings("unchecked")
        DuplicationMode duplicationMode = Optional.ofNullable(transientVars)
                .map(tv -> tv.get(CreateContentByCopyFunction.COPY_MAP_KEY))
                .map(copyMap -> (Map<String, Object>) copyMap)
                .map(copyMap -> copyMap.get(CopyODFContentClientSideElement.DUPLICATION_MODE_KEY))
                .map(String.class::cast)
                .map(DuplicationMode::valueOf)
                .orElse(DuplicationMode.SINGLE);
        
        Set<String> childrendReferencesNames = _getChildrenReferencesName();
        if (!childrendReferencesNames.isEmpty())
        {
            boolean isValidDuplicationMode = true;
            Map<String, Object> childReferenceMap = new HashMap<>();
            childReferenceMap.put(CopyODFContentClientSideElement.DUPLICATION_MODE_KEY, duplicationMode.toString());
            switch (duplicationMode)
            {
                case SINGLE:
                    childReferenceMap.put("$mode", "reference");
                    break;
                case STRUCTURE_ONLY:
                    if (baseContent instanceof CourseList)
                    {
                        childReferenceMap.put("$mode", "reference");
                    }
                    else
                    {
                        childReferenceMap.put("$mode", "create");
                    }
                    break;
                case FULL:
                    childReferenceMap.put("$mode", "create");
                    break;
                default:
                    isValidDuplicationMode = false;
                    break;
            }
            
            if (isValidDuplicationMode)
            {
                for (String childrendReferenceName : childrendReferencesNames)
                {
                    additionalCopyMap.put(childrendReferenceName, childReferenceMap);
                }
            }
        }
        
        return additionalCopyMap;
    }
    
    @Override
    protected SynchronizableValue _buildParentSynchronizableValue(Content parent)
    {
        SynchronizableValue value = new SynchronizableValue(List.of(parent.getId()));
        value.setMode(Mode.APPEND);
        return value;
    }
    
    @Override
    protected void _populateAdditionalData(Map transientVars, ModifiableContent content) throws WorkflowException
    {
        super._populateAdditionalData(transientVars, content);
        
        // Generate new code
        content.setValue(ProgramItem.CODE, org.ametys.core.util.StringUtils.generateKey().toUpperCase());
        
        // copy catalog
        // FIXME remove as soon as the canWrite returns really true at creation
        content.setValue(ProgramItem.CATALOG, getBaseContentForCopy(transientVars).getValue(ProgramItem.CATALOG));
    }
    
    /**
     * Get the attributes names holding the children relations between the content to create and its children
     * @return the attributes names
     */
    protected abstract Set<String> _getChildrenReferencesName();
}
