/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.enumeration;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;


/**
 * Component for certification labels
 *
 */
public class CertificationLabels implements Serviceable, Configurable, Component, PluginAware
{
    /** The Avalon role */
    public static final String ROLE = CertificationLabels.class.getName();
    
    private Map<String, I18nizableText> _entries = new HashMap<>();
    
    private String _pluginName;
    
    private SourceResolver _srcResolver;
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _srcResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Configuration[] children = configuration.getChildren("entry");
        
        if (children.length == 0)
        {
            try
            {
                Source pluginSource = _srcResolver.resolveURI("plugin:" + _pluginName + "://labels.xml");
                
                try (InputStream is = pluginSource.getInputStream())
                {
                    Configuration cfg = new DefaultConfigurationBuilder().build(is);
                    children = cfg.getChildren("entry");
                }
            }
            catch (SAXException | IOException e)
            {
                throw new ConfigurationException("Unable to configure the labels enumerator", e);
            }
        }
        
        for (Configuration child : children)
        {
            String value = child.getChild("value").getValue();
            I18nizableText label = I18nizableText.parseI18nizableText(child.getChild("label"), "plugin." + _pluginName);
            _entries.put(value, label);
        }
    }
        
    /**
     * Get the title of a label
     * @param value the code of label
     * @return the label's title
     */
    public I18nizableText getLabelTitle(String value)
    {
        return _entries.get(value);
    }
    
    /**
     * Get the labels
     * @return the labels
     */
    public Map<String, I18nizableText> getLabels()
    {
        return _entries;
    }
}
