/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.enumeration;

import java.util.Locale;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;

/**
 * This class represents a entry of ODF reference table
 *
 */
public class OdfReferenceTableEntry
{
    /** Attribute name for code */
    public static final String CODE = "code";
    /** Attribute name for CDM value */
    public static final String CDM_VALUE = "cdmValue";
    
    /** Attribute name for ordering */
    public static final String ORDER = "order";
    /** Attribute name for archiving */
    public static final String ARCHIVED = "archived";
    
    private Content _entry;
    
    /**
     * Constructor
     * @param content The content representing the entry
     */
    public OdfReferenceTableEntry(Content content)
    {
        _entry = content;
    }
    
    /**
     * Get the identifier
     * @return The identifier
     */
    public String getId()
    {
        return _entry.getId();
    }
    
    /**
     * Get the label
     * @param lang The language
     * @return The label
     */
    public String getLabel(String lang)
    {
        return _entry.getTitle(new Locale(lang));
    }
    
    /**
     * Get the code
     * @return the code
     */
    public String getCode()
    {
        return _entry.getValue(CODE, false, StringUtils.EMPTY);
    }
    
    /** 
     * Get the CDM value
     * @return the CDM value
     */
    public String getCdmValue()
    {
        return _entry.getValue(CDM_VALUE, false, StringUtils.EMPTY);
    }
    
    /**
     * Get the order
     * @return the order
     */
    public Long getOrder()
    {
        return _entry.getValue(ORDER, false, Long.MAX_VALUE);
    }
    
    /**
     * Get the archiving state
     * @return <code>true</code> if the entry is archived, <code>false</code> otherwise
     */
    public Boolean isArchived()
    {
        return _entry.getValue(ARCHIVED, false, false);
    }
    
    /**
     * Get the content
     * @return the content
     */
    public Content getContent()
    {
        return _entry;
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(_entry.getId());
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null)
        {
            return false;
        }
        if (getClass() != obj.getClass())
        {
            return false;
        }
        OdfReferenceTableEntry other = (OdfReferenceTableEntry) obj;
        return Objects.equals(_entry.getId(), other._entry.getId());
    }
}
