/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.person;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * DAO for manipulating {@link Person}
 */
public class PersonDAO implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = PersonDAO.class.getName();
    
    /** The Ametys Object Resolver */
    private AmetysObjectResolver _resolver;

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Gets the information about the given persons
     * @param personIds The person ids
     * @return A map of information
     */
    @Callable
    public Map<String, Object> getPersonsInfos (List<String> personIds)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> persons = new ArrayList<>();
        List<String> notFound = new ArrayList<>();
        
        for (String id : personIds)
        {
            persons.add(getPersonInfos(id));
        }
        
        result.put("persons", persons);
        result.put("personsNotFound", notFound);
        
        return result;
    }
    
    /**
     * Gets the information about the given person
     * @param personId The person id
     * @return A map of information
     */
    @Callable
    public Map<String, Object> getPersonInfos (String personId)
    {
        Person person = _resolver.resolveById(personId);
        return getPersonInfos(person);
    }

    /**
     * Gets the information about the given person
     * @param person The person
     * @return A map of information
     */
    public Map<String, Object> getPersonInfos(Person person)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", person.getId());
        infos.put("name", person.getName());
        infos.put("title", person.getTitle());
        infos.put("personTitle", person.getPersonTitle());
        infos.put("lastName", person.getLastName());
        infos.put("givenName", person.getGivenName());
        
        return infos;
    }
}
