/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for access controler on role attributes
 */
public class ODFRoleAccessControllerHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The avalon role */
    public static final String ROLE = ODFRoleAccessControllerHelper.class.getName();
    
    private static final String __CACHE_ID = ODFRoleAccessControllerHelper.class.getName() + "$cache";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    /** The ODF right heper */
    protected ODFRightHelper _odfRightHelper;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _cacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
        _odfRightHelper = (ODFRightHelper) smanager.lookup(ODFRightHelper.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        if (!_cacheManager.hasCache(__CACHE_ID))
        {
            _cacheManager.createRequestCache(__CACHE_ID,
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_ROLE_ACCESS_CONTROLLER_LABEL"),
                new I18nizableText("plugin.odf", "PLUGINS_ODF_CACHE_ROLE_ACCESS_CONTROLLER_DESCRIPTION"),
                false
            );
        }
    }
    
    /**
     * Determines if user has a ODF role on any program item
     * @param user The user
     * @param roleAttributeName the attribute name for role
     * @return true if the user has a ODF role on at least a program item
     */
    public boolean hasODFRoleOnAnyProgramItem(UserIdentity user, String roleAttributeName)
    {
        Cache<CacheKey, Boolean> cache = _getCache();
        
        CacheKey key = CacheKey.of(user, roleAttributeName);
        return cache.get(key, __ -> _hasODFRoleOnAnyProgramItem(user, roleAttributeName));
    }
    
    private boolean _hasODFRoleOnAnyProgramItem(UserIdentity user, String roleAttributeName)
    {
        AmetysObjectIterable<ProgramItem> programItems = _odfRightHelper.getProgramItemsWithUserAsRole(user, roleAttributeName);
        return programItems.getSize() != 0;
    }
    
    static class CacheKey extends AbstractCacheKey
    {
        CacheKey(UserIdentity userIdentity, String roleAttributePath)
        {
            super(userIdentity, roleAttributePath);
        }

        static CacheKey of(UserIdentity userIdentity, String roleAttributePath)
        {
            return new CacheKey(userIdentity, roleAttributePath);
        } 
    }
    
    private Cache<CacheKey, Boolean> _getCache()
    {
        return _cacheManager.get(__CACHE_ID);
    }
}
