/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.validator;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.holder.DataHolderDisableConditionsEvaluator;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.data.EducationalPath;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * This global validation checks if the values of educational path attributs of a repeater entries are unique.<br>
 * &lt;repeaterPath&gt;, &lt;educationalPathDataName&gt; and &lt;errorI18nKey&gt; configuration are expected.
 */
public class RepeaterWithEducationalPathValidator extends AbstractContentValidator implements Serviceable, Configurable
{
    private DisableConditionsEvaluator<ModelAwareDataHolder> _disableConditionsEvaluator;
    private String _repeaterPath;
    private String _educationalPathAttribute;
    private ODFHelper _odfHelper;
    private String _errorI18nKey;
    
    @SuppressWarnings("unchecked")
    public void service(ServiceManager manager) throws ServiceException
    {
        _disableConditionsEvaluator = (DisableConditionsEvaluator<ModelAwareDataHolder>) manager.lookup(DataHolderDisableConditionsEvaluator.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _repeaterPath = configuration.getChild("repeaterPath").getValue();
        _educationalPathAttribute = configuration.getChild("educationalPathAttribute").getValue();
        _errorI18nKey = configuration.getChild("errorI18nKey").getValue();
    }
    
    @Override
    public ValidationResult validate(Content content)
    {
        ValidationResult result = new ValidationResult();
        
        ModelItem repeaterDefinition = content.getDefinition(_repeaterPath);
        
        if (content.hasValue(_repeaterPath) && !_disableConditionsEvaluator.evaluateDisableConditions(repeaterDefinition, _repeaterPath, content))
        {
            Set<EducationalPath> paths = new HashSet<>();
            
            ModelAwareRepeater repeater = content.getRepeater(_repeaterPath);
            
            ModelItem contentDataDefinition = content.getDefinition(_repeaterPath + ModelItem.ITEM_PATH_SEPARATOR + _educationalPathAttribute);
            
            for (ModelAwareRepeaterEntry entry : repeater.getEntries())
            {
                String dataPath = _repeaterPath + "[" + entry.getPosition() + "]" + ModelItem.ITEM_PATH_SEPARATOR + _educationalPathAttribute;
                if (!_disableConditionsEvaluator.evaluateDisableConditions(contentDataDefinition, dataPath, content))
                {
                    EducationalPath path = entry.getValue(_educationalPathAttribute);
                    
                    if (path != null && !paths.add(path))
                    {
                        // Two entries or more have the same value for education path
                        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                        i18nParams.put("path", new I18nizableText(_odfHelper.getEducationalPathAsString(path)));
                        result.addError(new I18nizableText(StringUtils.substringBefore(_errorI18nKey, ':'), StringUtils.substringAfter(_errorI18nKey, ':'), i18nParams));
                        break; // stop iteration
                    }
                }
            }
        }
        
        return result;
    }
    
    @Override
    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        Object repeater = values.get(_repeaterPath);
        
        List<Map<String, Object>> repeaterEntries =  repeater instanceof SynchronizableRepeater synchronizableRepeater
                    ? synchronizableRepeater.getEntries()
                    : List.of(); // repeater can be an UntouchedValue, if it is disable or non-writable

        if (!repeaterEntries.isEmpty())
        {
            Set<EducationalPath> paths = new HashSet<>();
            ModelItem modelItem = content.getDefinition(_repeaterPath + ModelItem.ITEM_PATH_SEPARATOR + _educationalPathAttribute);
            
            for (int i = 0; i < repeaterEntries.size(); i++)
            {
                Map<String, Object> entry = repeaterEntries.get(i);
                int position = i + 1;
                
                Optional<String> oldDataPath = Optional.of(repeaterEntries)
                                                       .filter(SynchronizableRepeater.class::isInstance)
                                                       .map(SynchronizableRepeater.class::cast)
                                                       .flatMap(syncRepeater -> syncRepeater.getPreviousPosition(position))
                                                       .map(previousPosition -> _repeaterPath + "[" + previousPosition + "]" + ModelItem.ITEM_PATH_SEPARATOR + _educationalPathAttribute);
                
                EducationalPath path = Optional.ofNullable(_educationalPathAttribute)
                                                  .map(entry::get)
                                                  .map(value -> DataHolderHelper.getValueFromSynchronizableValue(value, content, modelItem, oldDataPath, SynchronizationContext.newInstance()))
                                                  .filter(EducationalPath.class::isInstance)
                                                  .map(EducationalPath.class::cast)
                                                  .orElse(null);   // value can be an UntouchedValue, if it is disabled or non-writable
                
                if (path != null && !paths.add(path))
                {
                    // Two entries or more have the same value for education path
                    Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                    i18nParams.put("path", new I18nizableText(_odfHelper.getEducationalPathAsString(path)));
                    result.addError(new I18nizableText(StringUtils.substringBefore(_errorI18nKey, ':'), StringUtils.substringAfter(_errorI18nKey, ':'), i18nParams));
                    break; // stop iteration
                }
            }
        }
        
        return result;
    }
    
}
