/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.time.LocalDate;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.File;
import org.ametys.cms.data.Geocode;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.MonthEnumerator;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.DataContext;

/**
 * base CDMfr extension for Ametys
 */
public class AmetysCDMfrExtension extends AbstractCDMfrExtension implements Serviceable
{
    private OdfReferenceTableHelper _refTableHelper;
    private CatalogsManager _catalogsManager;
    private SourceResolver _sourceResolver;
    private AmetysObjectResolver _resolver;
    private MonthEnumerator _monthEnumerator;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _monthEnumerator = (MonthEnumerator) manager.lookup(MonthEnumerator.ROLE);
    }
    
    @Override
    public void abstractProgram2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        
        // Certified
        // <ametys-cdm:certified/>
        if (program.isCertified())
        {
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.CERTIFIED);
        }
        
        // Passerelle et réorientation
        // <ametys-cdm:reorientation>...</ametys-cdm:reorientation>
        DataContext reorientationContext = RepositoryDataContext.newInstance()
                                                                .withObject(program)
                                                                .withDataPath(AbstractProgram.REORIENTATION);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.REORIENTATION, program.getReorientation(), reorientationContext, _sourceResolver, attrs);
        
        
        // Date de fin de la formation
        // <ametys-cdm:teachingEnd>...</ametys-cdm:teachingEnd>
        LocalDate teachingEnd = program.getTeachingEnd();
        if (teachingEnd != null)
        {
            CDMHelper.date2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + CDMFRTagsConstants.TAG_TEACHING_END, teachingEnd);
        }

        // Ouvert en stage
        // <ametys-cdm:internshipOpen/>
        if (program.isInternshipOpen())
        {
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.INTERNSHIP_OPEN);
        }
        
        // Stages
        // <ametys-cdm:training>
        //      <ametys-cdm:trainingInfo value="internship_mandatory">Obligatoire</ametys-cdm:trainingInfo>
        //      <ametys-cdm:trainingDuration>6 mois</ametys-cdm:trainingInfo>
        //      <ametys-cdm:trainingAbroadInfo value="internship_mandatory">Obligatoire</ametys-cdm:trainingAbroadInfo>
        //      <ametys-cdm:trainingAbroadDuration>6 mois</ametys-cdm:trainingAbroadDuration>
        //      <ametys-cdm:trainingDetails>
        //          <ametys-cdm:trainingDetail>
        //              <ametys-cdm:title>...</ametys-cmd:title>
        //              <ametys-cdm:duration>...</ametys-cmd:duration>
        //              <ametys-cdm:period>...</ametys-cmd:period>
        //              <ametys-cdm:kind>...</ametys-cmd:kind>
        //          </ametys-cdm:trainingDetail>
        //      </ametys-cdm:trainingDetails>
        // </ametys-cdm:training>
        saxTraining(contentHandler, program);
        
        // Etablissement partenaires
        // <ametys-cdm:partnerSchools/>
        saxPartnerSchools(contentHandler, program);
        
        // Laboratoires partenaires
        // <ametys-cdm:partnerLaboratories/>
        saxPartnerLaboratories(contentHandler, program);
        
        // Fichier attachés
        // <ametys-cdm:attachments/>
        saxAttachments(contentHandler, program);
        
        // Niveau RNCP
        // <ametys-cdm:rncpLevel/>
        if (program instanceof Program)
        {
            String[] rncpLevels = program.getRncpLevel();
            
            if (rncpLevels.length > 0)
            {
                for (String rncpLevel : rncpLevels)
                {
                    String rncpValue = _refTableHelper.getItemCDMfrValue(rncpLevel, true);
                    String rncpLabel = _refTableHelper.getItemLabel(rncpLevel, program.getLanguage());
                    
                    attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("value", rncpValue);
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.RNCP_LEVEL, attrs, rncpLabel);
                }
            }
            else
            {
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.RNCP_LEVEL);
            }
        }
        
        // RNCP code
        // <ametys-cdm:rncpCode>Code RNCP</ametys-cdm:rncpCode>
        for (String rncpCode : program.getRncpCode())
        {
            if (StringUtils.isNotEmpty(rncpCode))
            {
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "rncpCode", rncpCode);
            }
        }
        
        // Niveau(x) d'entrée
        // <ametys-cdm:educationEntryLevel code="code">Label</ametys-cdm:educationEntryLevel>
        _saxMultipleEnumeration(contentHandler, program.getEducationLevelEntry(), AbstractProgram.EDUCATION_ENTRY_LEVEL, program.getLanguage());
        
        // Campus
        // <ametys-cdm:campus code="code">Label</ametys-cdm:campus>
        _saxMultipleEnumeration(contentHandler, program.getCampus(), AbstractProgram.CAMPUS, program.getLanguage());
        
        // Lieu(x) à l'étranger
        // <ametys-cdm:foreignPlace code="code">Label</ametys-cdm:foreignPlace>
        _saxMultipleEnumeration(contentHandler, program.getForeignPlace(), AbstractProgram.FOREIGN_PLACE, program.getLanguage());
        
        // Champ(s) de formation
        // <ametys-cdm:programField code="code">Label</ametys-cdm:programField>
        _saxMultipleEnumeration(contentHandler, program.getProgramField(), AbstractProgram.PROGRAM_FIELD, program.getLanguage());
        
        // Disciplines
        // <ametys-cdm:disciplines code="code">Label</ametys-cdm:disciplines>
        _saxMultipleEnumeration(contentHandler, program.getDisciplines(), AbstractProgram.DISCIPLINES, program.getLanguage());
        
        // Compétences requises
        // <ametys-cdm:requiredskills code="code">Label</ametys-cdm:disciplines>
        _saxMultipleEnumeration(contentHandler, program.getRequiredSkills(), AbstractProgram.REQUIRED_SKILLS, program.getLanguage());
        
        // Poursuite d'études dans l'établissement
        // <ametys-cdm:furtherStudyPrograms code="cdm-code">Label</ametys-cdm:furtherStudyPrograms>
        for (String furtherStudyProgramId : program.getFurtherStudyPrograms())
        {
            AbstractProgram furtherStudyProgram  = _resolver.resolveById(furtherStudyProgramId);
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute("code", furtherStudyProgram.getCdmCode());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.FURTHER_STUDY_PROGRAMS, attrs, furtherStudyProgram.getTitle());
        }
        
        // Secteurs d'activité
        // <ametys-cdm:sectors code="code">Label</ametys-cdm:sectors>
        _saxMultipleEnumeration(contentHandler, program.getSectors(), AbstractProgram.SECTORS, program.getLanguage());
        
        // Ouvert en alternance
        // <ametys-cdm:apprenticeshipOpen />
        if (program.isApprenticeshipOpen())
        {
            XMLUtils.createElement(contentHandler, AbstractProgram.APPRENTICESHIP_OPEN);
        }
        
        // Rythme d'alternance
        // <ametys-cdm:apprenticeshipPeriod>...</ametys-cdm:apprenticeshipPeriod>
        DataContext apprenticeshipPeriodContext = RepositoryDataContext.newInstance()
                                                                       .withObject(program)
                                                                       .withDataPath(AbstractProgram.APPRENTICESHIP_PERIOD);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.APPRENTICESHIP_PERIOD, program.getApprenticeshipPeriod(), apprenticeshipPeriodContext, _sourceResolver);
        
        // Modalitités d'alternance
        // <ametys-cdm:apprenticeshipModalities>...</ametys-cdm:apprenticeshipModalities>
        DataContext apprenticeshipModalitiesContext = RepositoryDataContext.newInstance()
                                                                       .withObject(program)
                                                                       .withDataPath(AbstractProgram.APPRENTICESHIP_MODALITIES);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.APPRENTICESHIP_MODALITIES, program.getApprenticeshipModalities(), apprenticeshipModalitiesContext, _sourceResolver);
        
        // Type(s) de contrat
        // <ametys-cdm:apprenticeshipContract code="code">Label</ametys-cdm:apprenticeshipContract>
        _saxMultipleEnumeration(contentHandler, program.getApprenticeshipContract(), AbstractProgram.APPRENTICESHIP_CONTRACT, program.getLanguage());
        
        // Formation internationale
        // <ametys-cdm:internationalEducation code="code">Label</ametys-cdm:internationalEducation>
        _saxMultipleEnumeration(contentHandler, program.getInternationalEducation(), AbstractProgram.INTERNATIONAL_EDUCATION, program.getLanguage());
        
        // Dimension internationale
        // <ametys-cdm:internationalDimension>...</ametys-cdm:internationalDimension>
        DataContext internationalDimensionContext = RepositoryDataContext.newInstance()
                                                                         .withObject(program)
                                                                         .withDataPath(AbstractProgram.INTERNATIONAL_DIMENSION);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.INTERNATIONAL_DIMENSION, program.getInternationalDimension(), internationalDimensionContext, _sourceResolver);
        
        // Géolocalisation
        // <ametys-cdm:disciplines code="code">Label</ametys-cdm:disciplines>
        Geocode geocode = program.getGeocode();
        if (geocode != null)
        {
            Double latitude = geocode.getLatitude();
            Double longitude = geocode.getLongitude();
            if (latitude != null && longitude != null)
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute("latitude", String.valueOf(latitude));
                attrs.addCDATAAttribute("longitude", String.valueOf(longitude));
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.GEOCODE, attrs);
            }
        }
        
        // Autres structures partenaires
        DataContext otherPartnersContext = RepositoryDataContext.newInstance()
                                                                .withObject(program)
                                                                .withDataPath(AbstractProgram.OTHER_PARTNERS);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.OTHER_PARTNERS, program.getOtherPartners(), otherPartnersContext, _sourceResolver);
        
        // Certifications possibles
        // <ametys-cdm:disciplines code="code">Label</ametys-cdm:disciplines>
        _saxMultipleEnumeration(contentHandler, program.getAvailableCertification(), AbstractProgram.AVAILABLE_CERTIFICATION, program.getLanguage());
        
        // Autre contact
        // <ametys-cdm:otherContact>...</ametys-cdm:otherContact>
        RichText otherContact = program.getValue(AbstractProgram.OTHER_CONTACT, false, null);
        if (otherContact != null)
        {
            DataContext otherContactContext = RepositoryDataContext.newInstance()
                                                                   .withObject(program)
                                                                   .withDataPath(AbstractProgram.OTHER_CONTACT);
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.OTHER_CONTACT, otherContact, otherContactContext, _sourceResolver);
        }
    }
    
    /**
     * SAX a multiple metadata based on an reference table.
     * &lt;ametys-cdm:tagName code="code"&gt;Label&lt;/ametys-cdm:tagName&gt;
     * @param contentHandler The handler
     * @param values Values to SAX
     * @param tagName The tag name
     * @param lang The language to translate the label
     * @throws SAXException if an error occurs
     */
    private void _saxMultipleEnumeration(ContentHandler contentHandler, String[] values, String tagName, String lang) throws SAXException
    {
        for (String value : values)
        {
            _saxSingleEnumeration(contentHandler, value, tagName, lang);
        }
    }
    
    private void _saxSingleEnumeration(ContentHandler contentHandler, String value, String tagName, String lang) throws SAXException
    {
        String code = _refTableHelper.getItemCDMfrValue(value, true);
        String label = _refTableHelper.getItemLabel(value, lang);
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("code", code);
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + tagName, attrs, label);
    }
    
    /**
     * Sax training and internship informations.
     * @param contentHandler the receiving contentHandler.
     * @param program the abstract program (common for program and subprogram)
     * @throws SAXException if an error occurs during CDM processing.
     */
    protected void saxTraining(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "training");
        
        String internship = program.getInternship();
        if (!internship.isEmpty())
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("value", _refTableHelper.getItemCDMfrValue(internship, true));
            String internshipLabel = _refTableHelper.getItemLabel(internship, program.getLanguage());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingInfo", attrs, internshipLabel);
        }
        else
        {
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingInfo");
        }
        
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingDuration", program.getInternshipDuration());
        
        String internshipAbroad = program.getInternshipAbroad();
        if (!internshipAbroad.isEmpty())
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("value", _refTableHelper.getItemCDMfrValue(internshipAbroad, true));
            String internshipLabel = _refTableHelper.getItemLabel(internshipAbroad, program.getLanguage());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingAbroadInfo", attrs, internshipLabel);
        }
        else
        {
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingAbroadInfo");
        }
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingAbroadDuration", program.getAbroadInternshipDuration());

        if (program.hasValue(AbstractProgram.INTERNSHIP_DESCRIPTION))
        {
            ModelAwareRepeater internshipDescriptions = program.getRepeater(AbstractProgram.INTERNSHIP_DESCRIPTION);
            if (internshipDescriptions.getSize() > 0)
            {
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingDetails");
                for (ModelAwareRepeaterEntry internshipDescription : internshipDescriptions.getEntries())
                {
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingDetail");
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "title", internshipDescription.getValue(AbstractProgram.INTERNSHIP_DESCRIPTION_TITLE, false, StringUtils.EMPTY));
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "duration", internshipDescription.getValue(AbstractProgram.INTERNSHIP_DESCRIPTION_DURATION, false, StringUtils.EMPTY));

                    for (String value : internshipDescription.getValue(AbstractProgram.INTERNSHIP_DESCRIPTION_PERIOD, false, ArrayUtils.EMPTY_STRING_ARRAY))
                    {
                        AttributesImpl attrs = new AttributesImpl();
                        attrs.addCDATAAttribute("code", value);
                        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "period", attrs, _monthEnumerator.getLabel(value, program.getLanguage()));
                    }
                    
                    StringBuilder dataPath = new StringBuilder(AbstractProgram.INTERNSHIP_DESCRIPTION)
                            .append("[")
                            .append(internshipDescription.getPosition())
                            .append("]")
                            .append(ModelItem.ITEM_PATH_SEPARATOR)
                            .append(AbstractProgram.INTERNSHIP_DESCRIPTION_KIND);
                    DataContext internshipDescriptionKindContext = RepositoryDataContext.newInstance()
                                                                                        .withObject(program)
                                                                                        .withDataPath(dataPath.toString());
                    CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.INTERNSHIP_DESCRIPTION_KIND, internshipDescription.getValue(AbstractProgram.INTERNSHIP_DESCRIPTION_KIND), internshipDescriptionKindContext, _sourceResolver);
                    
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingDetail");
                }
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "trainingDetails");
            }
        }

        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "training");
    }
    
    /**
     * Sax partner schools (&lt;ametys-cdm:partnerSchools/&gt;).
     * @param contentHandler the receiving contentHandler.
     * @param program the abstract program (common for program and subprogram)
     * @throws SAXException if an error occurs during CDM processing.
     */
    protected void saxPartnerSchools(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        if (program.hasValue(AbstractProgram.PARTNER_SCHOOLS))
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.PARTNER_SCHOOLS);
            
            ModelAwareRepeater partnerSchools = program.getRepeater(AbstractProgram.PARTNER_SCHOOLS);
            for (ModelAwareRepeaterEntry partnerSchool : partnerSchools.getEntries())
            {
                String linkUrl = partnerSchool.getValue(AbstractProgram.PARTNER_SCHOOLS_LINK_URL, false, StringUtils.EMPTY);
                String linkLabel = partnerSchool.getValue(AbstractProgram.PARTNER_SCHOOLS_LINK_LABEL, false, StringUtils.EMPTY);
                
                if (!linkUrl.isEmpty() || !linkLabel.isEmpty())
                {
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "partnerSchool");
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "linkUrl", linkUrl);
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "linkLabel", linkLabel);
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "partnerSchool");
                }
            }
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.PARTNER_SCHOOLS);
        }
    }

    /**
     * Sax partner laboratories (&lt;ametys-cdm:partnerLaboratories/&gt;).
     * @param contentHandler the receiving contentHandler.
     * @param program the abstract program (common for program and subprogram)
     * @throws SAXException if an error occurs during CDM processing.
     */
    protected void saxPartnerLaboratories(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        if (program.hasValue(AbstractProgram.PARTNER_LABORATORIES))
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.PARTNER_LABORATORIES);
            
            ModelAwareRepeater partnerLaboratories = program.getRepeater(AbstractProgram.PARTNER_LABORATORIES);
            for (ModelAwareRepeaterEntry partnerLaboratory : partnerLaboratories.getEntries())
            {
                String linkUrl = partnerLaboratory.getValue(AbstractProgram.PARTNER_LABORATORIES_LINK_URL, false, StringUtils.EMPTY);
                String linkLabel = partnerLaboratory.getValue(AbstractProgram.PARTNER_LABORATORIES_LINK_LABEL, false, StringUtils.EMPTY);
                
                if (!linkUrl.isEmpty() || !linkLabel.isEmpty())
                {
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "partnerLaboratory");
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "linkUrl", linkUrl);
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "linkLabel", linkLabel);
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "partnerLaboratory");
                }
            }
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM +  AbstractProgram.PARTNER_LABORATORIES);
        }
    }

    /**
     * Sax attachments (&lt;ametys-cdm:attachments/&gt;).
     * @param contentHandler the receiving contentHandler.
     * @param program the abstract program (common for program and subprogram)
     * @throws SAXException if an error occurs during CDM processing.
     */
    protected void saxAttachments(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        if (program.hasValue(AbstractProgram.ATTACHMENTS))
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.ATTACHMENTS);
            
            ModelAwareRepeater attachments = program.getRepeater(AbstractProgram.ATTACHMENTS);
            for (ModelAwareRepeaterEntry entry: attachments.getEntries())
            {
                if (entry.hasValue(AbstractProgram.ATTACHMENTS_ATTACHMENT))
                {
                    String url = null;
                    if (ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID.equals(entry.getType(AbstractProgram.ATTACHMENTS_ATTACHMENT).getId()))
                    {
                        File file = entry.getValue(AbstractProgram.ATTACHMENTS_ATTACHMENT);
                        url = _getFileAbsoluteUrl(program, file, "attachments[" + entry.getPosition() + "]/attachment");
                    }
                    
                    String text = entry.getValue(AbstractProgram.ATTACHMENTS_ATTACHMENT_TEXT, false, StringUtils.EMPTY);
                    
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "attachment");
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "text", text);
                    if (url != null)
                    {
                        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "href", url);
                    }
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "attachment");
                }
            }
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + AbstractProgram.ATTACHMENTS);
        }
    }
    
    /**
     * Sax the acquired skills
     * @param contentHandler the receiving contentHandler.
     * @param course the course
     * @throws SAXException if an error occurs during CDM processing.
     */
    protected void saxAcquiredSkills(ContentHandler contentHandler, Course course) throws SAXException
    {
        if (course.hasValue(Course.ACQUIRED_SKILLS))
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + Course.ACQUIRED_SKILLS);
            
            ModelAwareRepeater repeater = course.getRepeater(Course.ACQUIRED_SKILLS);
            for (ModelAwareRepeaterEntry entry : repeater.getEntries())
            {
                ContentValue skillSet = entry.getValue(Course.ACQUIRED_SKILLS_SKILLSET, false, null);
                if (skillSet != null)
                {
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "acquiredSkill");
                    
                    _saxSingleEnumeration(contentHandler, skillSet.getContentId(), "skillSet", course.getLanguage());
                    
                    ModelAwareRepeater skillRep = entry.getRepeater(Course.ACQUIRED_SKILLS_SKILLS);
                    if (skillRep != null)
                    {
                        for (ModelAwareRepeaterEntry skillEntry : skillRep.getEntries())
                        {
                            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "skill");
                            
                            ContentValue skill = skillEntry.getValue(Course.ACQUIRED_SKILLS_SKILLS_SKILL, false, null);
                            if (skill != null)
                            {
                                _saxSingleEnumeration(contentHandler, skill.getContentId(), "skill", course.getLanguage());
                            }
                            ContentValue acquisitionLevel = skillEntry.getValue(Course.ACQUIRED_SKILLS_SKILLS_ACQUISITION_LEVEL, false, null);
                            if (acquisitionLevel != null)
                            {
                                _saxSingleEnumeration(contentHandler, acquisitionLevel.getContentId(), "acquisitionLevel", course.getLanguage());
                            }
                            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "skill");
                        }
                    }
                    
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "acquiredSkill");
                    
                }
            }
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + Course.ACQUIRED_SKILLS);
        }
    }
    
    @Override
    public void program2CDM(ContentHandler contentHandler, Program program, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        // Catalogue de formation
        String catalogCode = program.getCatalog();
        if (StringUtils.isNotEmpty(catalogCode))
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("value", catalogCode);
            
            Catalog catalog = _catalogsManager.getCatalog(catalogCode);
            String catalogLabel = catalog != null ? catalog.getTitle() : catalogCode;
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "catalog", attrs, catalogLabel);
        }
    }
    
    public void subProgram2CDM(ContentHandler contentHandler, SubProgram subProgram, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        // nothing specific to subprograms in the default Ametys model
    }

    @Override
    public void course2CDM(ContentHandler contentHandler, Course course, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        // Catalogue de formation
        String catalogCode = course.getCatalog();
        if (StringUtils.isNotEmpty(catalogCode))
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("value", catalogCode);
            
            Catalog catalog = _catalogsManager.getCatalog(catalogCode);
            String catalogLabel = catalog != null ? catalog.getTitle() : catalogCode;
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "catalog", attrs, catalogLabel);
        }
        
        // Ouvert aux étudiants en échange
        // <ametys-cdm:openToExchangeStudents />
        if (course.isOpenToExchangeStudents())
        {
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + Course.OPEN_TO_EXCHANGE_STUDENTS);
        }
        
        // <ametys-cdm:requiredSkills/>
        _saxMultipleEnumeration(contentHandler, course.getRequiredSkills(), Course.REQUIRED_SKILLS, course.getLanguage());
        
        // Compétence acquises
        saxAcquiredSkills(contentHandler, course);
        
        // Course parts
        for (CoursePart coursePart : course.getCourseParts())
        {
            _coursePart2CDM(contentHandler, coursePart);
        }
    }
    
    /**
     * Sax a {@link CoursePart} with the following structure :
     * &lt;ametys-cdm:coursePart @ametys-cdm:code="code"&gt;
     *     &lt;ametys-cdm:title&gt;Title&lt;/ametys-cdm:title&gt;
     *     &lt;ametys-cdm:nature&gt;Nature CDM-fr value or code&lt;/ametys-cdm:nature&gt;
     *     &lt;ametys-cdm:nbHours&gt;Number of hours&lt;/ametys-cdm:nbHours&gt;
     *     &lt;ametys-cdm:courseHolder&gt;CDM ID of the course holder&lt;/ametys-cdm:courseHolder&gt;
     * &lt;/ametys-cdm:coursePart&gt;
     * @param contentHandler The content handler
     * @param coursePart The {@link CoursePart} to sax
     * @throws SAXException if an error occurs
     */
    protected void _coursePart2CDM(ContentHandler contentHandler, CoursePart coursePart) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("code", coursePart.getCode());
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "coursePart", attrs);
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + Content.ATTRIBUTE_TITLE, coursePart.getTitle());
        String nature = _refTableHelper.getItemCDMfrValue(coursePart.getNature(), true);
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + CoursePart.NATURE, nature);
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + CoursePart.NB_HOURS, String.valueOf(coursePart.getNumberOfHours()));
        
        Course courseHolder = coursePart.getCourseHolder();
        if (courseHolder != null)
        {
            // it is not a normal case, but sometimes, the courseHolder references a not existing course, and we don't want to fail here
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + CoursePart.COURSE_HOLDER, courseHolder.getCDMId());
        }
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "coursePart");
    }

    @Override
    public void orgunit2CDM(ContentHandler contentHandler, OrgUnit orgunit) throws SAXException
    {
        // nothing
    }

    @Override
    public void person2CDM(ContentHandler contentHandler, Person person) throws SAXException
    {
        // Titre du contenu (à ne pas confondre avec personTitle)
        String title = person.getTitle();
        if (StringUtils.isNotEmpty(title))
        {
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "title", title);
        }
    }

    @Override
    public void container2CDM(ContentHandler contentHandler, Container container, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        // Période
        // <ametys-cdm:period code="code">Label</ametys-cdm:period>
        String period = container.getPeriod();
        if (StringUtils.isNotEmpty(period))
        {
            String code = _refTableHelper.getItemCDMfrValue(period, true);
            String label = _refTableHelper.getItemLabel(period, container.getLanguage());
    
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("code", code);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + Container.PERIOD, attrs, label);
        }
    }
}
