/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.clientsideelement;

import java.time.LocalDate;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.course.ShareableCourseStatusHelper;
import org.ametys.odf.course.ShareableCourseStatusHelper.ShareableStatus;

/**
 * Client side element for shareable course status button
 */
public class ShareableCourseStatusButtonClientSideElement extends ShareableCourseContentClientSideElement
{
    /** The shareable course status helper */
    protected ShareableCourseStatusHelper _shareableCourseStatusHelper; 
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _shareableCourseStatusHelper = (ShareableCourseStatusHelper) manager.lookup(ShareableCourseStatusHelper.ROLE);
    }
    
    /**
     * True if the button is enable
     * @param buttonShareableStatus the button status
     * @param contentShareableStatus the content status
     * @return true if the button is enable
     */
    public boolean canChangeStatus(ShareableStatus buttonShareableStatus, ShareableStatus contentShareableStatus)
    {
        switch (buttonShareableStatus)
        {
            case NONE:
                return contentShareableStatus != ShareableStatus.NONE && contentShareableStatus != ShareableStatus.REFUSED;
            case PROPOSED:
                return contentShareableStatus == ShareableStatus.NONE || contentShareableStatus == ShareableStatus.REFUSED;
            case VALIDATED:
                return contentShareableStatus != ShareableStatus.VALIDATED;
            default:
                break;
        }
        
        return false;
    }
    
    /**
     * Set the shareable course status to the contents
     * @param contentIds the list of content ids
     * @param status the shareable course status
     * @param comment the comment. Can be null
     */
    @Callable
    public void setShareableCourseStatus(List<String> contentIds, String status, String comment)
    {
        for (String contentId : contentIds)
        {
            ShareableStatus buttonShareableStatus = StringUtils.isNotBlank(status) ? ShareableStatus.valueOf(status.toUpperCase()) : ShareableStatus.NONE;

            ModifiableContent content = _resolver.resolveById(contentId);
            ShareableStatus contentShareableStatus = _shareableCourseStatusHelper.getShareableStatus(content);
            
            if (canChangeStatus(buttonShareableStatus, contentShareableStatus))
            {
                UserIdentity user = _currentUserProvider.getUser();
                _shareableCourseStatusHelper.setWorkflowStateAttribute(content, LocalDate.now(), user, buttonShareableStatus, comment, false);
            }
        }
    }
}
