/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.contenttype;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.type.EducationalPathRepositoryElementType;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Manager for odf content type attribute
 */
public class ODFContentTypeAttributeManager extends AbstractLogEnabled implements Component, Serviceable
{
    /** The avalon role */
    public static final String ROLE = ODFContentTypeAttributeManager.class.getName();
    
    private ODFHelper _odfHelper;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    /**
     * Determine whether an attribute can be read at this time.
     * Method called by {@link RestrictedModelItem#canRead(Object)} to do some other checks, depending on the content type
     * @param definition the attribute definition
     * @param content The content where attribute is to be read on. Can be null, on content creation. 
     * @return <code>true</code> if the current user is allowed to read the attribute of this content.
     * @throws AmetysRepositoryException if an error occurs while accessing the content.
     */
    public boolean canRead(Content content, RestrictedModelItem<Content> definition) throws AmetysRepositoryException
    {
        if (content instanceof Course course && definition instanceof RepeaterDefinition repeaterDefinition && _containsEducationalPathAttribute(repeaterDefinition))
        {
            // A repeater those entries depend on a educational path attribute is editable only if the course is shared
            return _odfHelper.isShared(course);
        }
        
        return true;
    }
    
    /**
     * Determine whether an attribute can be written at this time.
     * Method called by {@link RestrictedModelItem#canWrite(Object)} to do some other checks, depending on the content type
     * @param definition the attribute definition
     * @param content The content where attribute is to be written on. Can be null, on content creation. 
     * @return <code>true</code> if the current user is allowed to write the attribute of this content.
     * @throws AmetysRepositoryException if an error occurs while accessing the content.
     */
    public boolean canWrite(Content content, RestrictedModelItem<Content> definition) throws AmetysRepositoryException
    {
        return canRead(content, definition);
    }
    
    private boolean _containsEducationalPathAttribute(RepeaterDefinition repeaterDefinition)
    {
        for (ModelItem childItem : repeaterDefinition.getModelItems())
        {
            if (EducationalPathRepositoryElementType.EDUCATIONAL_PATH_ELEMENT_TYPE_ID.equals(childItem.getType().getId()))
            {
                return true;
            }
        }
        
        return false;
    }
}
