/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.course.search;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.MetadataType;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.NotQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.cms.search.ui.model.impl.AbstractCustomSearchUICriterion;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.ShareableCourseConstants;
import org.ametys.odf.course.ShareableCourseHelper;
import org.ametys.odf.course.ShareableCourseStatusHelper.ShareableStatus;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.ModelItem;

/**
 * UI criteria for shareable course
 */
public class ShareableCourseSearchUICriteria extends AbstractCustomSearchUICriterion
{
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The shareable course helper */
    protected ShareableCourseHelper _shareableCourseHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _shareableCourseHelper = (ShareableCourseHelper) manager.lookup(ShareableCourseHelper.ROLE);
    }
    
    public String getFieldId()
    {
        return "shareable.course.field";
    }

    public Operator getOperator()
    {
        // Not suppose to be used
        return Operator.EQ;
    }
    
    @Override
    public MetadataType getType()
    {
        // Not suppose to be used
        return MetadataType.STRING;
    }

    public Query getQuery(Object value, Operator customOperator, Map<String, Object> allValues, String language, Map<String, Object> contextualParameters)
    {
        if (!_shareableCourseHelper.handleShareableCourse())
        {
            return null;
        }
        
        return _getShareableQuery(value, customOperator, allValues, language, contextualParameters);
    }

    /**
     * Get the {@link Query} associated to shareable fields
     * @param value The user-submitted value (or the default value if not set) for this criterion.
     * @param customOperator In advanced search mode, the operator chosen by the user. <code>null</code> to use the criterion-defined operator (simple search mode).
     * @param allValues All the user-submitted values.
     * @param language The current search language.
     * @param contextualParameters the search contextual parameters.
     * @return {@link Query} associated to shareable fields
     */
    protected Query _getShareableQuery(Object value, Operator customOperator, Map<String, Object> allValues, String language, Map<String, Object> contextualParameters)
    {
        List<Query> queries = new ArrayList<>();
        
        // Shareable course must be validated
        queries.add(_getValidatedShareableCourseQuery());
        
        if (contextualParameters.containsKey("courseListId"))
        {
            String courseListId = (String) contextualParameters.get("courseListId");
            if (StringUtils.isNotBlank(courseListId))
            {
                ProgramItem courseList = _resolver.resolveById(courseListId);
                queries.addAll(_getShareableFieldQueries(courseList));
            }
        }
        
        return queries.stream()
                .filter(q -> q != null)
                .collect(AndQuery.collector());
    }
    
    /**
     * Get the query to get only validated shareable course
     * @return the query
     */
    protected Query _getValidatedShareableCourseQuery()
    {
        return new StringQuery(ShareableCourseConstants.SHAREABLE_COURSE_COMPOSITE_METADATA + ModelItem.ITEM_PATH_SEPARATOR + ShareableCourseConstants.SHAREABLE_COURSE_STATUS_METADATA, ShareableStatus.VALIDATED.name());
    }
    
    /**
     * Get query form each shareable field
     * @param programItem the program item
     * @return the list of query
     */
    protected List<Query> _getShareableFieldQueries(ProgramItem programItem)
    {
        List<Query> queries = new ArrayList<>();
        
        Set<Program> programs = _odfHelper.getParentPrograms(programItem);
        Set<Container> containers = _odfHelper.getParentContainers(programItem);
        
        // Query for programs field
        Set<String> programValues = _shareableCourseHelper.getProgramIds(programs);
        queries.add(_getFieldQuery(programValues, ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME));
        
        // Query for periods field
        Set<String> periodValues = _shareableCourseHelper.getPeriods(containers);
        queries.add(_getFieldQuery(periodValues, ShareableCourseConstants.PERIODS_FIELD_ATTRIBUTE_NAME));
        
        // Query for degree field
        Set<String> degreeValues = _shareableCourseHelper.getDegrees(programs);
        queries.add(_getFieldQuery(degreeValues, ShareableCourseConstants.DEGREES_FIELD_ATTRIBUTE_NAME));
        
        // Query for orgunits field
        Set<String> orgunitValues = _shareableCourseHelper.getOrgUnits(programs);
        queries.add(_getFieldQuery(orgunitValues, ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME));
        
        return queries;
    }
    
    /**
     * Get query for the shareable course field. Query is "not(field) or field = values"
     * @param values the values
     * @param metadataName the metadata name of the field
     * @return the fitler query
     */
    protected Query _getFieldQuery(Set<String> values, String metadataName)
    {
        if (values.isEmpty())
        {
            return null;
        }
        
        Query programValuesQuery = values.stream()
                .map(v -> new StringQuery(metadataName, v))
                .collect(OrQuery.collector());
        
        Query noProgramValueQuery = new NotQuery(new StringQuery(metadataName));
        
        return new OrQuery(noProgramValueQuery, programValuesQuery);
    }
}
