/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.data.type;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.common.SolrInputDocument;

import org.ametys.cms.data.type.indexing.IndexableDataContext;
import org.ametys.cms.data.type.indexing.IndexableElementType;
import org.ametys.core.model.type.AbstractElementType;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.data.EducationalPath;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.ametysobject.DataAwareAmetysObject;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.type.ComplexRepositoryElementType;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Class for educationalPath type of elements stored in the repository.
 * Education path is stored as multiple string with contents' id composing the path
 */
public class EducationalPathRepositoryElementType extends AbstractElementType<EducationalPath> implements ComplexRepositoryElementType<EducationalPath>, IndexableElementType<EducationalPath>
{
    /** JCR type for geocodes. */
    public static final String EDUCATIONAL_PATH_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":educationalPath";
    /** Constant for educational-path element type */
    public static final String EDUCATIONAL_PATH_ELEMENT_TYPE_ID = "educational-path";
    /** The ODFHelper instance */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public EducationalPath convertValue(Object value) throws BadItemTypeException
    {
        if (value instanceof String)
        {
            String strValue = (String) value;
            if (StringUtils.isEmpty(strValue))
            {
                return null;
            }
            
            String[] pathSegments = strValue.split(EducationalPath.PATH_SEGMENT_SEPARATOR);
            
            return EducationalPath.of(pathSegments);
        }
        else if (value instanceof String[])
        {
            return EducationalPath.of((String[]) value);
        }
        
        return super.convertValue(value);
    }
    
    @Override
    public String toString(EducationalPath value)
    {
        if (value != null)
        {
            return value.toString();
        }
        else
        {
            return null;
        }
    }
    
    @SuppressWarnings({"unchecked"})
    @Override
    protected EducationalPath _singleValueFromJSON(Object json, DataContext context) throws BadItemTypeException
    {
        if (json == null)
        {
            return null;
        }
        else if (json instanceof String jsonStr)
        {
            return StringUtils.isEmpty(jsonStr) ? null : EducationalPath.of(jsonStr.split(EducationalPath.PATH_SEGMENT_SEPARATOR));
        }
        else if (json instanceof List jsonList)
        {
            return Optional.ofNullable(jsonList)
                .filter(List::isEmpty)
                .map(s -> EducationalPath.of(((List<String>) jsonList).toArray(String[]::new)))
                .orElse(null);
        }
        else
        {
            throw new BadItemTypeException("Unable to convert the given json value '" + json + "' into a " + getManagedClass().getName());
        }
    }
    
    @Override
    protected Object _singleTypedValueToJSON(EducationalPath value, DataContext context)
    {
        return value.getProgramItemIds();
    }
    
    @Override
    public Object valueToJSONForClient(Object value, DataContext context) throws BadItemTypeException
    {
        return _valueToJSON(value, context, this::_singleValueToJSONForClient);
    }
    
    /**
     * Convert the single value into a JSON object for client rendering
     * @param value the value to convert
     * @param context The context of the data to convert
     * @return The value as JSON
     * @throws BadItemTypeException if the given value is not compatible is the current type
     */
    private Object _singleValueToJSONForClient(Object value, DataContext context)
    {
        if (value instanceof EducationalPath educationPath)
        {
            if (_checkValidty(educationPath, context))
            {
                return Map.of(
                        "id", educationPath.toString(),
                        "title", _odfHelper.getEducationalPathAsString(educationPath)
                    );
            }
            else
            {
                getLogger().warn("The educational path '{}' does not match any existing ODF path. Value null is returned.", educationPath);
            }
        }
        
        return null;
    }
    
    private boolean _checkValidty(EducationalPath educationPath, DataContext context)
    {
        if (context instanceof RepositoryDataContext repoContext)
        {
            DataAwareAmetysObject content = repoContext.getObject().orElse(null);
            if (content != null && content instanceof ProgramItem programItem)
            {
                return _odfHelper.isValid(educationPath, programItem, false);
            }
        }
        
        // Program item not found, check validity of the path regardless of the object from which the value is computed
        return _odfHelper.isValid(educationPath);
    }
    
    @Override
    protected boolean _useJSONForEdition()
    {
        return true;
    }
    
    public boolean isSimple()
    {
        return false;
    }
    
    public String getRepositoryDataType()
    {
        return EDUCATIONAL_PATH_NODETYPE;
    }

    public void indexSingleValue(SolrInputDocument document, SolrInputDocument rootObjectDocument, String fieldName, EducationalPath value, IndexableDataContext context)
    {
        EducationalPath singleValue = getSingleValueToIndex(value);
        List<String> programItemIds = singleValue.getProgramItemIds();
        
        for (String programItemId : programItemIds)
        {
            document.addField(fieldName + getIndexingFieldSuffix(context), programItemId);
            // Facets
            document.addField(fieldName + "_s_dv", programItemId);
        }
    }

    public void indexSingleValueForFullTextField(SolrInputDocument document, SolrInputDocument rootObjectDocument, EducationalPath value, IndexableDataContext context)
    {
        // Empty
    }
    
    public String getIndexingFieldSuffix(DataContext context)
    {
        return "_s";
    }

    public String getSchemaType()
    {
        return "string";
    }

    public boolean isSingleValueEmpty(RepositoryData pathData)
    {
        return !pathData.hasValue(EducationalPath.PATH_SEGMENTS_IDENTIFIER) || pathData.getStrings(EducationalPath.PATH_SEGMENTS_IDENTIFIER).length == 0;
    }

    public EducationalPath readSingleValue(RepositoryData pathData)
    {
        String[] pathSegments = pathData.getStrings(EducationalPath.PATH_SEGMENTS_IDENTIFIER);
        if (pathSegments.length > 0)
        {
            return EducationalPath.of(pathSegments);
        }
        return null;
    }

    public void writeSingleValue(ModifiableRepositoryData parentData, String name, EducationalPath value)
    {
        ModifiableRepositoryData pathData = parentData.addRepositoryData(name, getRepositoryDataType());
        
        if (value != null)
        {
            pathData.setValues(EducationalPath.PATH_SEGMENTS_IDENTIFIER, value.getProgramItemIds().toArray(String[]::new));
        }
    }
}
