/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.orgunit;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * Enumerator for {@link OrgUnit}s using UAI code as unique identifier
 *
 */
public class OrgUnitUAICodeEnumerator implements Enumerator<String>, Serviceable, org.ametys.runtime.parameter.Enumerator
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The OrgUnit root provider */
    protected RootOrgUnitProvider _orgUnitProvider;
    
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _orgUnitProvider = (RootOrgUnitProvider) smanager.lookup(RootOrgUnitProvider.ROLE);
    }

    @Override
    public Map<Object, I18nizableText> getEntries() throws Exception
    {
        return (Map<Object, I18nizableText>) (Object) getTypedEntries();
    }

    @Override
    public Map<String, I18nizableText> getTypedEntries() throws Exception
    {
        Map<String, I18nizableText> entries = new HashMap<>();
        
        OrgUnit orgUnit = _orgUnitProvider.getRoot();
        entries.putAll(_getSubOrgUnits(orgUnit));
        
        return entries;
    }
    
    private Map<String, I18nizableText> _getSubOrgUnits (OrgUnit orgUnit)
    {
        Map<String, I18nizableText> entries = new HashMap<>();
        
        List<String> ids = orgUnit.getSubOrgUnits();
        for (String id : ids)
        {
            try
            {
                OrgUnit child = _resolver.resolveById(id);
                String title = child.getTitle();
                
                entries.put(child.getUAICode(), new I18nizableText(title));
                
                entries.putAll(_getSubOrgUnits (child));
            }
            catch (UnknownAmetysObjectException e)
            {
                // Ignore.
            }
        }
        
        return entries;
    }

    @Override
    public I18nizableText getEntry(String value) throws Exception
    {
        I18nizableText entry = null;
        
        if (StringUtils.isEmpty(value))
        {
            entry = new I18nizableText("plugin.odf", "PLUGINS_ODF_ORGUNIT_ENUMERATOR_VALUE_ALL");
        }
        else
        {
            try
            {
                OrgUnit orgunit = _resolver.resolveById(value);
                entry = new I18nizableText(orgunit.getTitle());
            }
            catch (UnknownAmetysObjectException e)
            {
                // Ignore.
            }
        }
        
        return entry;
    }
    
    @Override
    // TODO NEWATTRIBUTEAPI: remove this method when org.ametys.runtime.parameter.Enumerator will be removed
    public Map<String, Object> getConfiguration()
    {
        return Collections.EMPTY_MAP;
    }
}
