/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessController.Permission.PermissionType;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.rights.ODFRightHelper.PermissionContext;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.core.impl.right.WorkspaceAccessController;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Abstract class for access controller based of a ODF role attribute
 *
 */
public abstract class AbstractODFRoleAccessController implements AccessController, Serviceable, PluginAware
{
    private static final String __CMS_RIGHT_CONTEXT = "/cms";
    
    /** The rights profile DAO */
    protected RightProfilesDAO _rightProfileDAO;
    /** The ODF contents tree helper */
    protected ODFContentsTreeHelper _odfContentsTreeHelper;
    /** The ODF right helper */
    protected ODFRightHelper _odfRightHelper;
    /** The ametys resolver */
    protected AmetysObjectResolver _resolver;
    /** The role access helper */
    protected ODFRoleAccessControllerHelper _roleAccessControllerHelper;

    private String _id;

    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _rightProfileDAO = (RightProfilesDAO) smanager.lookup(RightProfilesDAO.ROLE);
        _odfContentsTreeHelper = (ODFContentsTreeHelper) smanager.lookup(ODFContentsTreeHelper.ROLE);
        _odfRightHelper = (ODFRightHelper) smanager.lookup(org.ametys.odf.rights.ODFRightHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _roleAccessControllerHelper = (ODFRoleAccessControllerHelper) smanager.lookup(ODFRoleAccessControllerHelper.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public String getId()
    {
        return _id;
    }
    
    public boolean isSupported(Object object)
    {
        return object instanceof ProgramItem || object instanceof OrgUnit || object instanceof String && ((String) object).startsWith(__CMS_RIGHT_CONTEXT);
    }
    
    /**
     * Get the parents of the content for rights purpose
     * @param content the content
     * @param permissionCtx the permission context
     * @return the parents of content
     */
    protected Set<Content> _getParents(Content content, PermissionContext permissionCtx)
    {
        return _odfRightHelper.getParents(content, permissionCtx).stream()
                .filter(Content.class::isInstance)
                .map (Content.class::cast)
                .collect(Collectors.toSet());
    }
    
    /**
     * Get the permission context
     * @param initialContent the initial content
     * @return the permission context.
     */
    protected PermissionContext _getPermissionContext(Content initialContent)
    {
        return new PermissionContext(initialContent);
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (object instanceof String)
        {
            if (_roleAccessControllerHelper.hasODFRoleOnAnyProgramItem(user, _getRoleAttributePath()))
            {
                // allow user on CMS context if he has permission on at least a program item
                return _getRightsInTargetProfile().contains(rightId) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
            }
            else
            {
                return AccessResult.UNKNOWN;
            }
        }
        else
        {
            return _getPermission(user, userGroups, rightId, (Content) object, _getPermissionContext((Content) object));
        }
    }
    
    private AccessResult _getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Content object, PermissionContext permissionCtx)
    {
        List<String> rights = _getRightsInTargetProfile();
        if (rights.contains(rightId))
        {
            Set<UserIdentity> allowedUsers = _getLocalAllowedUsers(object);
            if (allowedUsers.contains(user))
            {
                return AccessResult.USER_ALLOWED;
            }
        }
        
        AccessResult permission = AccessResult.UNKNOWN;
        
        Set<Content> parents = _getParents(object, permissionCtx);
        if (parents != null)
        {
            for (Content parent : parents)
            {
                AccessResult parentResult = _getPermission(user, userGroups, rightId, parent, permissionCtx);
                permission = AccessResult.merge(permission, parentResult);
            }
        }
        
        return permission;
    }
    
    /**
     * Get the rights hold by target profile
     * @return the rights hold by target profile
     */
    protected synchronized List<String> _getRightsInTargetProfile()
    {
        String profileId = _getTargetProfileId();
        return StringUtils.isNotBlank(profileId) ? _rightProfileDAO.getRights(profileId) : List.of();
    }
    
    /**
     * Get the id of target profile
     * @return the id of target profile
     */
    protected abstract String _getTargetProfileId();
    
    /**
     * Get the allowed users for this content taking into account the content itself and its parents
     * @param content the ODF content (program item or orgunit)
     * @param permissionCtx the permission context
     * @return the allowed users. Empty if no user is allowed on this content
     */
    protected Set<UserIdentity> _getAllowedUsers(Content content, PermissionContext permissionCtx)
    {
        Set<UserIdentity> allowedUsers = _getLocalAllowedUsers(content);
        
        Set<Content> parents = _getParents(content, permissionCtx);
        if (parents != null)
        {
            for (Content parent : parents)
            {
                allowedUsers.addAll(_getAllowedUsers(parent, permissionCtx));
            }
        }
        
        return allowedUsers;
    }
    
    /**
     * Get the local allowed users for this content
     * @param content the ODF content (program item or orgunit)
     * @return the allowed users. Empty if no user is allowed on this content
     */
    protected abstract Set<UserIdentity> _getLocalAllowedUsers(Content content);

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        if (object instanceof String)
        {
            if (_roleAccessControllerHelper.hasODFRoleOnAnyProgramItem(user, _getRoleAttributePath()))
            {
                return _getRightsInTargetProfile().stream()
                        .collect(Collectors.toMap(r -> r, r -> AccessResult.USER_ALLOWED));
            }
        }
        else
        {
            Set<UserIdentity> allowedUsers = _getAllowedUsers((Content) object, _getPermissionContext((Content) object));
            if (allowedUsers.contains(user))
            {
                return _getRightsInTargetProfile().stream()
                    .collect(Collectors.toMap(r -> r, r -> AccessResult.USER_ALLOWED));
            }
        }
        return Map.of();
    }
    
    /**
     * Get the attribute path for role
     * @return the attribute path for role
     */
    protected abstract String _getRoleAttributePath();

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        if (object instanceof Content && _getRightsInTargetProfile().contains(rightId))
        {
            Set<UserIdentity> allowedUsers = _getAllowedUsers((Content) object, _getPermissionContext((Content) object));
            if (allowedUsers != null)
            {
                return allowedUsers.stream()
                    .collect(Collectors.toMap(user -> user, user -> AccessResult.USER_ALLOWED));
            }
        }
        return Map.of();
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return Map.of();
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Map.of();
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Map.of();
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        boolean supportWorkspaceCtx = workspacesContexts.stream()
            .filter(String.class::isInstance)
            .map(String.class::cast)
            .anyMatch(ctx -> ctx.startsWith(__CMS_RIGHT_CONTEXT));
        
        if (supportWorkspaceCtx && _roleAccessControllerHelper.hasODFRoleOnAnyProgramItem(user, _getRoleAttributePath()))
        {
            // allow BO access if user has permission on at least a program item
            return _getRightsInTargetProfile().contains(rightId);
        }
        return false;
    }
    
    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }
    
    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    public AccessExplanation explainPermission(UserIdentity user, Set<GroupIdentity> groups, String rightId, Object object)
    {
        if (_getRightsInTargetProfile().contains(rightId))
        {
            return _explainPermissionForRole(user, groups, object, true);
        }
        else
        {
            return getStandardAccessExplanation(AccessResult.UNKNOWN, object);
        }
    }
    
    private AccessExplanation _explainPermissionForRole(UserIdentity user, Set<GroupIdentity> groups, Object object, boolean withHierarchy)
    {
        if (object instanceof String)
        {
            if (_roleAccessControllerHelper.hasODFRoleOnAnyProgramItem(user, _getRoleAttributePath()))
            {
                // allow user on CMS context if he has permission on at least a program item
                return new AccessExplanation(getId(), AccessResult.USER_ALLOWED,
                                new I18nizableText("plugin.odf", "PLUGINS_ODF_ROLE_ACCESS_CONTROLLER_GENERAL_EXPLANATION",
                                        Map.of("role", _getRoleLabel())));
            }
            else
            {
                return AccessController.getDefaultAccessExplanation(getId(), AccessResult.UNKNOWN);
            }
        }
        else
        {
            PermissionDetails details = _getPermissionDetails(user, groups, (Content) object, _getPermissionContext((Content) object), withHierarchy);
            return _buildExplanation(details);
        }
    }

    private PermissionDetails _getPermissionDetails(UserIdentity user, Set<GroupIdentity> groups, Content content, PermissionContext permissionCtx, boolean withHierarchy)
    {
        Set<UserIdentity> allowedUsers = _getLocalAllowedUsers(content);
        if (allowedUsers.contains(user))
        {
            return new PermissionDetails(AccessResult.USER_ALLOWED, content, false);
        }
        
        PermissionDetails details = new PermissionDetails(AccessResult.UNKNOWN, content, false);
        if (withHierarchy)
        {
            Set<Content> parents = _getParents(content, permissionCtx);
            if (parents != null)
            {
                for (Content parent : parents)
                {
                    PermissionDetails parentDetails = _getPermissionDetails(user, groups, parent, permissionCtx, true);
                    
                    AccessResult parentResult = parentDetails.result();
                    if (parentResult != AccessResult.UNKNOWN && AccessResult.merge(parentResult, details.result()) == parentResult)
                    {
                        // FIXME here we arbitrarily keep the last explanation but we should merge instead
                        // Build a new explanation only if the actual not inherited
                        details = parentDetails.inherited() ? parentDetails : new PermissionDetails(parentResult, parentDetails.object(), true);
                    }
                }
            }
        }
        
        return details;
    }

    private AccessExplanation _buildExplanation(PermissionDetails details)
    {
        AccessResult result = details.result();
        if (AccessResult.UNKNOWN.equals(result))
        {
            return AccessController.getDefaultAccessExplanation(getId(), AccessResult.UNKNOWN);
        }
        
        Map<String, I18nizableTextParameter> params = Map.of(
                "title", new I18nizableText(details.object().getTitle()),
                "role", _getRoleLabel()
                );
        I18nizableText label;
        if (details.inherited())
        {
            label = new I18nizableText("plugin.odf", "PLUGINS_ODF_ROLE_ACCESS_CONTROLLER_INHERITED_EXPLANATION", params);
        }
        else
        {
            label = new I18nizableText("plugin.odf", "PLUGINS_ODF_ROLE_ACCESS_CONTROLLER_EXPLANATION", params);
        }
        return new AccessExplanation(getId(), details.result(), label);
    }

    /**
     * Get the label to insert in the explanation to describe the role.
     * The label should start with a lower case.
     * @return the label
     */
    protected abstract I18nizableText _getRoleLabel();
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups)
    {
        Map<ExplanationObject, Map<Permission, AccessExplanation>> result = new HashMap<>();
        try (AmetysObjectIterable<ProgramItem> programItemsWithUserAsRole = _odfRightHelper.getProgramItemsWithUserAsRole(identity, _getRoleAttributePath()))
        {
            for (ProgramItem item : programItemsWithUserAsRole)
            {
                AccessExplanation explanation = _explainPermissionForRole(identity, groups, item, false);
                if (explanation.accessResult() != AccessResult.UNKNOWN)
                {
                    Map<Permission, AccessExplanation> contextPermission = new HashMap<>();
                    contextPermission.put(new Permission(PermissionType.PROFILE, _getTargetProfileId()), explanation);
                    
                    result.put(getExplanationObject(item), contextPermission);
                }
            }
        }
        
        String generalContext = __CMS_RIGHT_CONTEXT;
        AccessExplanation explanation = _explainPermissionForRole(identity, groups, generalContext, false);
        if (explanation.accessResult() != AccessResult.UNKNOWN)
        {
            Map<Permission, AccessExplanation> contextPermission = new HashMap<>();
            contextPermission.put(new Permission(PermissionType.PROFILE, _getTargetProfileId()), explanation);
            
            result.put(getExplanationObject(generalContext), contextPermission);
        }
        
        return result;
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof String)
        {
            return WorkspaceAccessController.GENERAL_CONTEXT_CATEGORY;
        }
        else if (object instanceof Content content)
        {
            return ODFContentHierarchicalAccessController.getContentObjectLabel(content, _odfContentsTreeHelper);
        }
        throw new RightsException("Unsupported object: " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        if (object instanceof String)
        {
            return WorkspaceAccessController.GENERAL_CONTEXT_CATEGORY;
        }
        else
        {
            return ODFContentHierarchicalAccessController.ODF_CONTEXT_CATEGORY;
        }
    }
    
    private record PermissionDetails(AccessResult result, Content object, boolean inherited) { }
}
