/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * This class controls a ribbon button that allows to set the publishable state of a content.<br/>
 * The button is toggle if content is currently unpublishable.
 * @private
 */
Ext.define('Ametys.plugins.odf.content.controller.PublishableODFContentController', {
	extend: 'Ametys.plugins.cms.content.controller.SmartContentController',
		
	/**
	 * @private
	 * @property {String[]} _notPublishableContentIds The ids of not publishable contents among the current selection
	 */
	/**
	 * @private
	 * @property {String[]} _publishableContentIds The ids of publishable contents among the current selection
	 */
	
	/**
	 * @cfg {String} [icon-glyph] The CSS class for button's icon when the content is publishable
	 */
    /**
     * @cfg {String} [not-publishable-icon-glyph] The CSS class for button's icon when the content is not publishable
     */
	
	constructor: function(config)
	{
		this.callParent(arguments);
		
		Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
	},
	
	/**
	 * Listener handler for modified messages
	 * @param {Ametys.message.Message} message the message
	 */
	 _onModified: function(message)
	 { 
		 if (this.updateTargetsInCurrentSelectionTargets (message))
		 {
			 this.refresh();
		 }
	 },
	 
	/**
	 * Returns the ids of publishable contents among the current selection
	 * @return {String[]} The contents' id
	 */
	getNotPublishableContentIds: function ()
	{
		return this._notPublishableContentIds;
	},
	
	/**
	 * Returns the ids of not publishable contents among the current selection
	 * @return {String[]} The contents' id
	 */
	getPublishableContentIds: function ()
	{
		return this._publishableContentIds;
	},
	
	/**
	 * @private
	 * Get the status
     * @param targets The content targets
	 */
	_getStatus: function (targets)
	{
		this._contentIds = [];
		this._notPublishableContentIds = [];
		this._publishableContentIds = [];
		
        this.disable();
        this.callParent(arguments);
	},

    _calculateStatus: function(targets)
    {
        var calculateStatus = this.callParent(arguments);
        
        calculateStatus["not-publishable-contents"] = [];
        calculateStatus["publishable-contents"] = [];

        Ext.Array.each(targets, function(contentTarget)
            {
                var parameters = contentTarget.getParameters();
                if (parameters && parameters.content)
                {
                    var content = parameters.content;
                    if (!parameters.additionalData.isPublishable)
                    {
                        var i18nStr = this.getConfig("not-publishable-contents-description");
                        var description = Ext.String.format(i18nStr, content.getTitle());
                        var contentParam = this._getContentDefaultParameters(content);
                        contentParam["description"] = description;
                        calculateStatus["not-publishable-contents"].push(contentParam);
                    }
                    else
                    {
                        var i18nStr = this.getConfig("publishable-contents-description");
                        var description = Ext.String.format(i18nStr, content.getTitle());
                        var contentParam = this._getContentDefaultParameters(content);
                        contentParam["description"] = description;
                        calculateStatus["publishable-contents"].push(contentParam);
                    }
                    
                }
            }, this);
        return calculateStatus;
    },
	
	_getStatusCb: function(params)
	{
        this.callParent(arguments);
        
		var me = this;
        Ext.Array.each (params['not-publishable-contents'], function (content) {
            me._notPublishableContentIds.push(content.id);
		});
		
        Ext.Array.each (params['publishable-contents'], function (content) {
            me._publishableContentIds.push(content.id);
		});

        this.toggle(this._notPublishableContentIds.length > 0);
		this._updateIcons(params);
	},
	
	_updateTooltipDescription: function (description, params)
	{
        this.callParent(arguments);
        
        var parentDescription = this.description;
        
		parentDescription = this._handlingMultiple (parentDescription, 'publishable-contents', params['publishable-contents']);
		parentDescription = this._handlingMultiple(parentDescription, "not-publishable-contents", params['not-publishable-contents']);
		
		this.setDescription (parentDescription);
	},
	
	/**
     * @private
	 * Update the icons of the button
	 * @param {Object} params the server's parameters
	 */
	_updateIcons: function(params)
	{
		var nbPublishableContents = params['publishable-contents'].length;
		var nbNotPublishableContents = params['not-publishable-contents'].length;
		
		var iconGlyph = this.getInitialConfig()['icon-glyph'];
		
		if (nbPublishableContents > 0 && nbNotPublishableContents == 0)
		{
            iconGlyph = this.getInitialConfig()['icon-glyph'];
		}
		
		if (nbNotPublishableContents > 0)
		{
			iconGlyph = this.getInitialConfig()['not-publishable-icon-glyph'];
		}
        
        this.setGlyphIcon(iconGlyph);
	}
    
});