/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.odf.CallWSHelper;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Send CDM-fr to a distant server
 *
 */
public class SendCDMFRFunction extends AbstractContentWorkflowComponent implements EnhancedFunction, Initializable
{
    private boolean _isActive;
    private SourceResolver _sourceResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue("odf.publish.server");
    }

    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        if (!_isActive)
        {
            return;
        }
        
        // Retrieve current content
        WorkflowAwareContent content = getContent(transientVars);
        
        try
        {
            // Generate CDM-fr file
            Source cdmfrSource = _sourceResolver.resolveURI(getExportUri() + "?id=" + content.getId() + "&" + ExportCDMfrHelper.REQUEST_PARAM_VALID_LABEL + "=true&" + ExportCDMfrHelper.REQUEST_PARAM_EXPORT_FOR_AMETYS + "=true");

            // Send to remote server
            try (InputStream is = cdmfrSource.getInputStream())
            {
                Set<String> failedUrl = CallWSHelper.callWS("_odf-sync/upload-cdm", is, _logger);
                if (failedUrl == null || failedUrl.size() > 0)
                {
                    List<String> params = new ArrayList<>();
                    params.add(content.getTitle());
                    params.add(StringUtils.join(failedUrl, ", "));
                    addWorkflowWarning(transientVars, new I18nizableText("plugin.odf", "PLUGINS_ODF_PUBLISH_PROGRAM_PORTAL_ERROR", params));
                    
                    _logger.error("The program " + content.getId() + " can't be synchronized with portals" + (failedUrl != null ? " " + StringUtils.join(failedUrl, ", ") : ""));
                }
            }
        }
        catch (IOException e)
        {
            addWorkflowError(transientVars, new I18nizableText("plugin.odf", "PLUGINS_ODF_PUBLISH_PROGRAM_REMOTE_ERROR", Collections.singletonList(content.getTitle())));
            _logger.error("Unable to publish CDM-fr on distant server", e);
        }
    }
    
    /**
     * Get the URI to use to export CDMfr
     * @return The uri location
     */
    protected String getExportUri ()
    {
        return "cocoon://_plugins/odf/export-cdmfr.xml";
    }
    
    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_SEND_CDMFR_FUNCTION_LABEL");
    }
}
