/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.contenttype;

import java.util.Arrays;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.contenttype.DefaultContentType;
import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.ShareableCourseHelper;
import org.ametys.odf.course.ShareableCourseStatusHelper;
import org.ametys.odf.course.ShareableCourseStatusHelper.ShareableStatus;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.version.VersionableAmetysObject;

/**
 * The ODF content type
 */
public class ODFContentType extends DefaultContentType
{
    /** The ODF content type metadata manager */
    protected ODFContentTypeAttributeManager _odfContentTypeAttributeManager;
    
    /** The shareable course status helper */
    protected ShareableCourseStatusHelper _shareableCourseStatusHelper;
    
    /** The shareable course helper */
    protected ShareableCourseHelper _shareableCourseHelper;
    
    /** The ODF skills helper */
    protected ODFSkillsHelper _odfSkillsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfContentTypeAttributeManager = (ODFContentTypeAttributeManager) manager.lookup(ODFContentTypeAttributeManager.ROLE);
        _shareableCourseStatusHelper = (ShareableCourseStatusHelper) manager.lookup(ShareableCourseStatusHelper.ROLE);
        _shareableCourseHelper = (ShareableCourseHelper) manager.lookup(ShareableCourseHelper.ROLE);
        _odfSkillsHelper = (ODFSkillsHelper) manager.lookup(ODFSkillsHelper.ROLE);
    }
    
    public boolean canRead(Content content, RestrictedModelItem<Content> definition) throws AmetysRepositoryException
    {
        return super.canRead(content, definition) && _odfContentTypeAttributeManager.canRead(content, definition);
    }
    
    public boolean canWrite(Content content, RestrictedModelItem<Content> definition) throws AmetysRepositoryException
    {
        return super.canWrite(content, definition) && _odfContentTypeAttributeManager.canWrite(content, definition);
    }
    
    @Override
    public Map<String, Object> getAdditionalData(Content content)
    {
        Map<String, Object> additionalData = super.getAdditionalData(content);
        
        if (_shareableCourseHelper.handleShareableCourse())
        {
            if (content instanceof Course)
            {
                ShareableStatus shareableStatus = _shareableCourseStatusHelper.getShareableStatus(content);
                additionalData.put("shareableCourseStatus", shareableStatus.name());
            }
        }
        
        if (content instanceof ProgramItem programItem)
        {
            additionalData.put("excludedFromSkills", _odfSkillsHelper.isExcluded(programItem));
            if (content instanceof VersionableAmetysObject)
            {
                String[] allLabels = ((VersionableAmetysObject) content).getAllLabels();
                additionalData.put("hasLiveVersion", Arrays.asList(allLabels).contains(CmsConstants.LIVE_LABEL));
            }
            
            additionalData.put("isPublishable", programItem.isPublishable());
        }
        
        return additionalData;
    }
}
