/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation;

import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Observer to update course part title on course (only on course parts holded by the course) or course part modification.
 */
public class CoursePartTitleObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The ODF reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        if (event.getId().equals(ObservationConstants.EVENT_CONTENT_MODIFIED))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return content instanceof CoursePart;
        }
        
        return false;
    }

    @Override
    public int getPriority(Event event)
    {
        return 0;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        CoursePart coursePart = (CoursePart) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        String coursePartTitle =
            Optional.of(coursePart)
                .map(CoursePart::getNature)
                .map(nature -> _odfRefTableHelper.getItemCode(nature))
                .orElse("N/D");
        
        coursePart.setTitle(coursePartTitle);
        coursePart.saveChanges();
    }
}
