/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.ShareableCourseConstants;
import org.ametys.odf.course.ShareableCourseHelper;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue.Mode;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.model.View;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Observer to unlink the shareable field of a {@link Course} on {@link Content} deletion.
 */
public class ShareableFieldCourseObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    private ContentWorkflowHelper _contentWorkflowHelper;
    private ShareableCourseHelper _shareableCourseHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _shareableCourseHelper = (ShareableCourseHelper) manager.lookup(ShareableCourseHelper.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        if (event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return _isProgramContent(content)
                || _isOrgUnitContent(content)
                || _isDegreeContent(content)
                || _isPeriodContent(content);
        }
        
        return false;
    }

    @Override
    public int getPriority(Event event)
    {
        return 0;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        String contentId = content.getId();
        
        if (_isProgramContent(content))
        {
            _editShareableField(_shareableCourseHelper.getShareableCourses(contentId, null, null, null), contentId, ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME);
        }
        else if (_isDegreeContent(content))
        {
            _editShareableField(_shareableCourseHelper.getShareableCourses(null, contentId, null, null), contentId, ShareableCourseConstants.DEGREES_FIELD_ATTRIBUTE_NAME);
        }
        else if (_isPeriodContent(content))
        {
            _editShareableField(_shareableCourseHelper.getShareableCourses(null, null, contentId, null), contentId, ShareableCourseConstants.PERIODS_FIELD_ATTRIBUTE_NAME);
        }
        else if (_isOrgUnitContent(content))
        {
            _editShareableField(_shareableCourseHelper.getShareableCourses(null, null, null, contentId), contentId, ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME);
        }
    }

    /**
     * Edit shareable fields
     * @param courses the list of courses to edit shareable fields
     * @param contentId the content id to remove from sharebale fields
     * @param attributeNameToRemove the attribute name of the shareable field
     * @throws WorkflowException if an error occurred
     */
    protected void _editShareableField(AmetysObjectIterable<Course> courses, String contentId, String attributeNameToRemove) throws WorkflowException
    {
        for (Course course : courses)
        {
            Map<String, Object> paramsEdit = new HashMap<>();
            paramsEdit.put(CheckRightsCondition.FORCE, true); // Ignore the right condition
            
            Map<String, Object> contextParameters = new HashMap<>();
            
            contextParameters.put(EditContentFunction.QUIT, true);
            
            SynchronizableValue value = new SynchronizableValue(new String[] {contentId});
            value.setMode(Mode.REMOVE);
            contextParameters.put(EditContentFunction.VALUES_KEY, Map.of(attributeNameToRemove, value));
            
            View view = View.of(course.getModel(), attributeNameToRemove);
            contextParameters.put(EditContentFunction.VIEW, view);
            
            paramsEdit.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, contextParameters);
            _contentWorkflowHelper.doAction(course, 225, paramsEdit);
        }
    }
    
    private boolean _isProgramContent(Content content)
    {
        return content instanceof Program;
    }
    
    private boolean _isOrgUnitContent(Content content)
    {
        return content instanceof OrgUnit;
    }
    
    private boolean _isDegreeContent(Content content)
    {
        return ArrayUtils.contains(content.getTypes(), OdfReferenceTableHelper.DEGREE);
    }
    
    private boolean _isPeriodContent(Content content)
    {
        return ArrayUtils.contains(content.getTypes(), OdfReferenceTableHelper.PERIOD);
    }
    
}
