/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.orgunit;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * DAO for manipulating OrgUnits
 */
public class OrgUnitDAO implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = OrgUnitDAO.class.getName();
    
    /** The Ametys Object Resolver */
    private AmetysObjectResolver _resolver;

    /** The ODF helper */
    private ODFHelper _odfHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    /**
     * Gets the information about the given orgunits
     * @param orgUnitIds The orgunits ids
     * @return A map of information
     */
    @Callable
    public Map<String, Object> getOrgUnitsInfos (List<String> orgUnitIds)
    {
        return getOrgUnitsInfos(orgUnitIds, null);
    }
    
    /**
     * Gets the information about the given orgunits
     * @param orgUnitIds The orgunits ids
     * @param rootOrgUnitId the root orgUnitId
     * @return A map of information
     */
    @Callable
    public Map<String, Object> getOrgUnitsInfos (List<String> orgUnitIds, String rootOrgUnitId)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> orgUnits = new ArrayList<>();
        List<String> notFound = new ArrayList<>();
        
        for (String id : orgUnitIds)
        {
            orgUnits.add(getOrgUnitInfos(id, rootOrgUnitId));
        }
        
        result.put("orgUnits", orgUnits);
        result.put("orgUnitsNotFound", notFound);
        
        return result;
    }
    
    /**
     * Gets the information about the given orgunit
     * @param orgUnitId The orgunit id
     * @return A map of information
     */
    @Callable
    public Map<String, Object> getOrgUnitInfos (String orgUnitId)
    {
        return getOrgUnitInfos(orgUnitId, null);
    }
    
    /**
     * Gets the information about the given orgunit
     * @param orgUnitId The orgunit id
     * @param rootOrgUnitId the root orgUnit id
     * @return A map of information
     */
    @Callable
    public Map<String, Object> getOrgUnitInfos (String orgUnitId, String rootOrgUnitId)
    {
        OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
        return getOrgUnitInfos(orgUnit, rootOrgUnitId);
    }

    /**
     * Gets the information about the given orgunit
     * @param orgUnit The orgunit
     * @param rootOrgUnitId the root orgUnit id
     * @return A map of information
     */
    public Map<String, Object> getOrgUnitInfos(OrgUnit orgUnit, String rootOrgUnitId)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", orgUnit.getId());
        infos.put("name", orgUnit.getName());
        infos.put("title", orgUnit.getTitle());
        infos.put("path", _odfHelper.getOrgUnitPath(orgUnit.getId(), rootOrgUnitId));
        
        return infos;
    }
}
