/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program.properties;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.model.properties.AbstractContentProperty;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnitFactory;

/**
 * Property to get all the orgunits linked to a program item (hold by it self and its parents)
 */
public class SelfAndParentOrgUnitsProperty extends AbstractContentProperty<Content>
{
    /** The ODF Helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    protected Set<? extends ModifiableContent> _getLinkedContents(Content content)
    {
        if (content instanceof ProgramItem programItem)
        {
            return _getSelfAndParentOrgUnits(programItem);
        }
        return null;
    }
    
    private Set<ModifiableContent> _getSelfAndParentOrgUnits(ProgramItem programItem)
    {
        Set<ModifiableContent> orgUnits = Arrays.stream(((Content) programItem).getValue(ProgramItem.ORG_UNITS_REFERENCES, false, new ContentValue[0]))
            .map(ContentValue::getContentIfExists)
            .flatMap(Optional::stream)
            .collect(Collectors.toSet());
        
        List<ProgramItem> parents = _odfHelper.getParentProgramItems(programItem);
        for (ProgramItem parent : parents)
        {
            orgUnits.addAll(_getSelfAndParentOrgUnits(parent));
        }
        
        return orgUnits;
    }
    
    @Override
    public String getCriterionWidget()
    {
        return "edition.select-orgunit";
    }
    
    @Override
    public boolean isMultiple()
    {
        return true;
    }

    public String getContentTypeId()
    {
        return OrgUnitFactory.ORGUNIT_CONTENT_TYPE;
    }
}
