/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.tree;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.tree.ODFTreeIndicator.IndicatorData;
import org.ametys.plugins.contentstree.ContentsTreeHelper;

/**
 * Helper ODF contents tree adding indicators
 */
public class ODFContentsTreeHelper extends ContentsTreeHelper
{
    /** The Avalon role */
    @SuppressWarnings("hiding")
    public static final String ROLE = ODFContentsTreeHelper.class.getName();

    /** Extension point for indicators */
    protected ODFTreeIndicatorExtensionPoint _odfTreeIndicatorEP;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfTreeIndicatorEP = (ODFTreeIndicatorExtensionPoint) smanager.lookup(ODFTreeIndicatorExtensionPoint.ROLE);
    }
    
    @Override
    protected Map<String, Object> content2Json(Content content)
    {
        Map<String, Object> infos = super.content2Json(content);
        infos.putAll(getIndicatorData(content));
        return infos;
    }
    
    /**
     * Get the indicator data to JSON format
     * @param content the contet
     * @return the indicator data to JSON format
     */
    protected Map<String, Object> getIndicatorData(Content content)
    {
        Map<String, Object> indicatorsData = new HashMap<>();
        
        for (String indicatorId : _odfTreeIndicatorEP.getExtensionsIds())
        {
            ODFTreeIndicator indicator = _odfTreeIndicatorEP.getExtension(indicatorId);
            IndicatorData indicatorData = indicator.getIndicatorData(content);
            if (indicatorData != null)
            {
                indicatorsData.put(indicator.getId(), _indicatorData2json(indicatorData));
            }
        }
        
        return indicatorsData;
    }
    
    private Map<String, Object> _indicatorData2json(IndicatorData indicatorData)
    {
        Map<String, Object> data = new HashMap<>();
        data.put("tooltip", indicatorData.tooltip());
        data.put("cssClass", indicatorData.cssClass());
        data.put("text", indicatorData.text());
        
        Map<String, Object> additionalData = indicatorData.additionalData();
        if (additionalData != null)
        {
            data.putAll(additionalData);
        }
        return data;
    }
    
    /**
     * Get the ProgramItem code we want to display.
     * @param <T> The type of the program element, should be a subclasse of {@link ProgramItem} and {@link Content}
     * @param programItem The program item
     * @return The code to display
     */
    public <T extends ProgramItem & Content> String getProgramItemDisplayCode(T programItem)
    {
        return programItem.getCode();
    }
    
    @Override
    protected boolean isContentMatching(Content content, String value)
    {
        boolean matchTitle = super.isContentMatching(content, value);
        if (!matchTitle)
        {
            if (content instanceof ProgramItem)
            {
                // Try with code
                String code =  StringUtils.stripAccents(((ProgramItem) content).getCode().toLowerCase());
                return code.contains(value);
            }
            else if (content instanceof OrgUnit)
            {
                // Try with code
                String code =  StringUtils.stripAccents(((OrgUnit) content).getUAICode().toLowerCase());
                return code.contains(value);
            }
        }
        
        return matchTitle;
    }
}
