/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * This class controls a ribbon button that allows to set the skills exclusion.<br/>
 * The button is toggle if odf content is currently excluded.
 * @private
 */
Ext.define('Ametys.plugins.odf.skills.SkillsExclusionController', {
	extend: 'Ametys.plugins.cms.content.controller.SmartContentController',
		
	/**
	 * @private
	 * @property {String[]} _programItemIds The id the currently selected program items
	 */
	/**
	 * @private
	 * @property {String[]} _includedProgramItemIds The ids of included program items among the current selection
	 */
	/**
	 * @private
	 * @property {String[]} _excludedProgramItemIds The ids of excluded program items among the current selection
	 */
	
	/**
	 * @cfg {String} [icon-glyph] The CSS class for button's icon when the program item is included
	 */
    /**
     * @cfg {String} [excluded-icon-glyph] The CSS class for button's icon when the program item is excluded
     */
	
	constructor: function(config)
	{
		this.callParent(arguments);
		
		Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
	},
	
	/**
	 * Listener handler for modified messages
	 * @param {Ametys.message.Message} message the message
	 */
	 _onModified: function(message)
	 { 
		 if (this.updateTargetsInCurrentSelectionTargets (message))
		 {
			 this.refresh();
		 }
	 },
	 
	/**
	 * Returns the ids of program items currently included among the current selection
	 * @return {String[]} The program item ids
	 */
	getIncludedProgramItemIds: function ()
	{
		return this._includedProgramItemIds;
	},
	
	/**
	 * Returns the ids of program items currently excluded among the current selection
	 * @return {String[]} The program item ids
	 */
	getExcludedProgramItemIds: function ()
	{
		return this._excludedProgramItemIds;
	},
	
	_getStatus: function (targets)
	{
		this._programItemIds = [];
		this._excludedProgramItemIds = [];
		this._includedProgramItemIds = [];
		
		this.callParent(arguments);
	},

    _calculateStatus: function(targets)
    {
        var result = this.callParent(arguments);
        result["excluded-program-items"] = [];
        result["included-program-items"] = [];

        Ext.Array.each(targets, function(programItemTarget)
            {
                var parameters = programItemTarget.getParameters();
                if (parameters.additionalData.excludedFromSkills)
                {
                    var i18nStr = this.getConfig("program-item-excluded-description");
                    var description = Ext.String.format(i18nStr, parameters.title);
                    var programItemParam = this._getProgramItemDefaultParameters(parameters);
                    programItemParam["description"] = description;
                    result["excluded-program-items"].push(programItemParam);
                }
                else
                {
                    var i18nStr = this.getConfig("program-item-included-description");
                    var description = Ext.String.format(i18nStr, parameters.title);
                    var programItemParam = this._getProgramItemDefaultParameters(parameters);
                    programItemParam["description"] = description;
                    result["included-program-items"].push(programItemParam);
                }
            }, this);
        return result;
    },
	
	_getStatusCb: function(params)
	{
        this.callParent(arguments);
        
		var me = this;
		Ext.Array.each (params['excluded-program-items'], function (content) {
			me._programItemIds.push(content.id);
			me._excludedProgramItemIds.push(content.id);
		});
		
		Ext.Array.each (params['included-program-items'], function (content) {
			me._programItemIds.push(content.id);
			me._includedProgramItemIds.push(content.id);
		});
		
		this._updateIcons(params);
		this.toggle(this._excludedProgramItemIds.length > 0);
	},
	
	_updateTooltipDescription: function (description, params)
	{
        this.callParent(arguments);
        var description = this._description;
        
		description = this._handlingMultiple(description, 'program-item-included', params['included-program-items']);
		description = this._handlingMultiple(description, "program-item-excluded", params['excluded-program-items']);
		
		this.setDescription(description);
	},
	
	/**
     * @private
	 * Update the icons of the button
	 * @param {Object} params the server's parameters
	 */
	_updateIcons: function(params)
	{
		var nbIncludedProgramItems = params['included-program-items'].length;
		var nbExcludedProgramItems = params['excluded-program-items'].length;
		
		var iconGlyph = this.getInitialConfig()['icon-glyph'];
		
		if (nbIncludedProgramItems > 0 && nbExcludedProgramItems == 0)
		{
            iconGlyph = this.getInitialConfig()['icon-glyph'];
		}
		else if (nbExcludedProgramItems)
		{
			iconGlyph = this.getInitialConfig()['excluded-icon-glyph'];
		}
        
        this.setGlyphIcon(iconGlyph);
	},
    
    /**
     * Create a json object representing a program item
     * @private
     * @param {Object} params The params of the program item
     * @return {Object} containing id and title
     */
    _getProgramItemDefaultParameters : function(params)
    {
        var defaultParams = {};
        defaultParams["id"] = params.id;
        defaultParams["title"] = params.title;
        return defaultParams;
    }
});