/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.odf.catalog;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang.StringUtils;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.SchedulerException;

import org.ametys.cms.repository.Content;
import org.ametys.core.schedule.Runnable;
import org.ametys.core.schedule.Runnable.FireProcess;
import org.ametys.core.schedule.Runnable.MisfirePolicy;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.I18nUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Program;
import org.ametys.odf.schedulable.CopyCatalogSchedulable;
import org.ametys.odf.schedulable.DeleteCatalogSchedulable;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * DAO for manipulating catalogs.
 *
 */
public class CatalogDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = CatalogDAO.class.getName();
    
    /** The catalog manager */
    protected CatalogsManager _catalogsManager;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The scheduler */
    protected Scheduler _scheduler;
    
    /** The I18N utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _scheduler = (Scheduler) manager.lookup(Scheduler.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    /**
     * Creates a new ODF catalog.
     * @param title The title of the catalog
     * @param name The code of the catalog
     * @param catalogNameToCopy The catalog name to copy or null
     * @return The id and the title of the created catalog, or an error
     * @throws ProcessingException if creation failed
     */
    @Callable
    public Map<String, String> createCatalog (String title, String name, String catalogNameToCopy) throws ProcessingException
    {
        Map<String, String> result = new HashMap<>();
        
        // FIXME CMS-5758 FilterNameHelper.filterName do not authorized name with number (so name is computed from JS)
        
        Catalog catalog = _catalogsManager.getCatalog(name);
        if (catalog != null)
        {
            result.put("message", "already-exist");
            return result;
        }
        
        Catalog newCatalog = _catalogsManager.createCatalog(name, title);
        
        if (StringUtils.isNotEmpty(catalogNameToCopy))
        {
            Catalog catalogToCopy = _catalogsManager.getCatalog(catalogNameToCopy);
            
            if (catalogToCopy == null)
            {
                result.put("message", "not-found");
                return result;
            }
            
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("srcCatalog", new I18nizableText(catalogToCopy.getTitle()));
            i18nParams.put("destCatalog", new I18nizableText(newCatalog.getTitle()));
            
            Map<String, Object> params = new HashMap<>();
            params.put(CopyCatalogSchedulable.JOBDATAMAP_SRC_CATALOG_KEY, catalogToCopy.getName());
            params.put(CopyCatalogSchedulable.JOBDATAMAP_DEST_CATALOG_KEY, newCatalog.getName());
            
            Runnable runnable = new DefaultRunnable(CopyCatalogSchedulable.SCHEDULABLE_ID + "$" + newCatalog.getName(), 
                new I18nizableText(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_LABEL_WITH_DETAILS", i18nParams))),
                new I18nizableText(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_COPY_CATALOG_DESCRIPTION_WITH_DETAILS", i18nParams))),
                FireProcess.NOW, 
                null /* cron*/, 
                CopyCatalogSchedulable.SCHEDULABLE_ID, 
                false /* removable */, 
                false /* modifiable */, 
                false /* deactivatable */, 
                MisfirePolicy.FIRE_ONCE, 
                false /* isVolatile */, 
                _currentUserProvider.getUser(), 
                params
            );
            
            try
            {
                JobKey jobKey = new JobKey(runnable.getId(), Scheduler.JOB_GROUP);
                if (_scheduler.getScheduler().checkExists(jobKey))
                {
                    _scheduler.getScheduler().deleteJob(jobKey);
                }
                _scheduler.scheduleJob(runnable);
            }
            catch (SchedulerException e)
            {
                getLogger().error("An error occured when trying to schedule the copy of the catalog '{}' to '{}'", catalogToCopy.getTitle(), newCatalog.getTitle(), e);
            }
        }
        
        result.put("id", newCatalog.getId());
        result.put("title", newCatalog.getTitle());
        
        return result;
    }
    
    /**
     * Edits an ODF catalog.
     * @param id The id of the catalog to edit
     * @param title The title of the catalog
     * @return The id and the title of the edited catalog, or an error
     */
    @Callable
    public Map<String, String> editCatalog (String id, String title)
    {
        Map<String, String> result = new HashMap<>();
        
        try
        {
            Catalog catalog = _resolver.resolveById(id);
            
            catalog.setTitle(title);
            catalog.saveChanges();
            
            result.put("id", catalog.getId());
        }
        catch (UnknownAmetysObjectException e)
        {
            result.put("message", "not-found");
        }
        
        return result;
    }
    
    /**
     * Set a catalog as default catalog
     * @param id The id of catalog
     * @return The id and the title of the edited catalog, or an error
     */
    @Callable
    public synchronized Map<String, String> setDefaultCatalog(String id)
    {
        Map<String, String> result = new HashMap<>();
        
        try
        {
            Catalog catalog = _resolver.resolveById(id);
            
            Catalog defaultCatalog = _catalogsManager.getDefaultCatalog();
            if (defaultCatalog != null)
            {
                defaultCatalog.setDefault(false);
                defaultCatalog.saveChanges();
            }
            catalog.setDefault(true);
            catalog.saveChanges();
            
            _catalogsManager.updateDefaultCatalog();
            
            result.put("id", catalog.getId());
        }
        catch (UnknownAmetysObjectException e)
        {
            result.put("message", "not-found");
        }
        
        return result;
    }
    
    /**
     * Removes an ODF catalog.
     * @param catalogId the catalog's id
     * @param forceDeletion if true, will not check if the catalog is referenced by {@code ProgramItem}s before deleting the catalog
     * @return The id of the deleted catalog, or an error
     */
    @Callable
    public Map<String, Object> removeCatalog (String catalogId, boolean forceDeletion)
    {
        Map<String, Object> result = new HashMap<>();
        result.put("id", catalogId);
        Catalog catalog = null;
        try
        {
            catalog = _resolver.resolveById(catalogId);
        }
        catch (UnknownAmetysObjectException e)
        {
            result.put("error", "unknown-catalog");
            return result;
        }
        
        if (!forceDeletion)
        {
            List<Content> contents = _catalogsManager.getContents(catalog.getName());
            if (!contents.isEmpty())
            {
                // Still has programItems
                // We provide a summary of the remaining items
                result.put("error", "remaining-items");
                result.put("remainingItems", _summarizeRemainingItems(contents));
                return result;
            }
        }
        String runnableId = _doRemoveCatalog(catalog);
        if (runnableId != null)
        {
            result.put("jobId", runnableId);
        }
        else
        {
            result.put("error", "job-start-failed");
        }
        return result;
    }
    
    private String _doRemoveCatalog(Catalog catalog)
    {
        String runnableId = DeleteCatalogSchedulable.SCHEDULABLE_ID + "$" + catalog.getId();
        try
        {
            JobKey jobKey = new JobKey(runnableId, Scheduler.JOB_GROUP);
            // here we want to check if a deletion of the catalog is not already running or finished
            // we rely on the job key which is not ideal, would be better to rely on the job param of the schedulable
            if (_scheduler.getScheduler().checkExists(jobKey))
            {
                Map<String, Object> runningMap = _scheduler.isRunning(jobKey.getName());
                boolean running = (boolean) runningMap.get("running");
                if (running)
                {
                    // A removal of the catalog is already ongoing so we do nothing
                    return runnableId;
                }
                else
                {
                    JobDetail jobDetail = _scheduler.getScheduler().getJobDetail(jobKey);
                    boolean success = jobDetail.getJobDataMap().getBoolean("success");
                    if (success)
                    {
                        // The catalog was already removed with success we might better do nothing
                        return null;
                    }
                    else
                    {
                        // The job failed, so we restart it
                        _scheduler.remove(jobKey.getName());
                    }
                }
            }
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("catalogName", new I18nizableText(catalog.getTitle()));
            i18nParams.put("catalogId", new I18nizableText(catalog.getId()));
    
            Map<String, Object> params = new HashMap<>();
            params.put(DeleteCatalogSchedulable.JOBDATAMAP_CATALOG_NAME_KEY, catalog.getName());
            
            Runnable runnable = new DefaultRunnable(runnableId, 
                new I18nizableText(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_LABEL_WITH_DETAILS", i18nParams))),
                new I18nizableText(_i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SCHEDULABLE_DELETE_CATALOG_DESCRIPTION_WITH_DETAILS", i18nParams))),
                FireProcess.NOW,
                null /* cron*/, 
                DeleteCatalogSchedulable.SCHEDULABLE_ID, 
                true /* removable */, 
                false /* modifiable */, 
                false /* deactivatable */, 
                MisfirePolicy.FIRE_ONCE, 
                true /* isVolatile */, 
                _currentUserProvider.getUser(), 
                params
            );
            
            _scheduler.scheduleJob(runnable);
            return runnableId;
        }
        catch (SchedulerException e)
        {
            getLogger().error("An error occured when trying to schedule the deletion of the catalog '{}'", catalog.getTitle(), e);
            return null;
        }
    }
    
    private Map<String, AtomicInteger> _summarizeRemainingItems(List<Content> contents)
    {
        Map<String, AtomicInteger> summary = new HashMap<>();
        for (Content content : contents)
        {
            String key = content instanceof ProgramItem || content instanceof CoursePart
                    ? StringUtils.substringAfterLast(content.getTypes()[0], ".")
                    : "other";
            summary.computeIfAbsent(key, __ -> new AtomicInteger()).incrementAndGet();
        }
        return summary;
    }
    
    /**
     * Gets the properties of a catalog.
     * @param id The catalog id
     * @return The properties of the catalog in a map
     */
    @Callable
    public Map<String, Object> getCatalogProperties(String id)
    {
        Catalog catalog = _resolver.resolveById(id);
        return getCatalogProperties(catalog);
    }
    
    /**
     * Gets the properties of a set of catalogs.
     * @param ids The catalogs' id
     * @return The properties of the catalogs
     */
    @Callable
    public Map<String, Object> getCatalogsProperties(List<String> ids)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> catalogs = new LinkedList<>();
        Set<String> unknownCatalogs = new HashSet<>();
        
        for (String id : ids)
        {
            try
            {
                Catalog catalog = _resolver.resolveById(id);
                catalogs.add(getCatalogProperties(catalog));
            }
            catch (UnknownAmetysObjectException e)
            {
                unknownCatalogs.add(id);
            }
        }
        
        result.put("catalogs", catalogs);
        result.put("unknownCatalogs", unknownCatalogs);
        
        return result;
    }
    
    /**
     * Get the properties of a catalog as a Map
     * @param catalog The catalog
     * @return The properties into a map object
     */
    public Map<String, Object> getCatalogProperties(Catalog catalog)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("id", catalog.getId());
        result.put("title", catalog.getTitle());
        result.put("isDefault", catalog.isDefault());
        result.put("code", catalog.getName());
        
        AmetysObjectIterable<Program> programs = _catalogsManager.getPrograms(catalog.getName());
        result.put("nbPrograms", programs.getSize());
        
        return result;
    }
}
