/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.skill.ODFSkillsHelper;

/**
 * SAX the acquired skills hold by child courses of a {@link ProgramItem}
 *
 */
public class ProgramItemSkillsGenerator extends ServiceableGenerator
{
    private ODFSkillsHelper _skillsHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _skillsHelper = (ODFSkillsHelper) smanager.lookup(ODFSkillsHelper.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Content content = (Content) request.getAttribute(Content.class.getName());

        String viewName = request.getParameter("viewName");
        
        contentHandler.startDocument();
        
        if ("main".equals(viewName) || StringUtils.isEmpty(viewName))
        {
            if (content instanceof ProgramItem)
            {
                Map<Content, Map<Content, Map<Content, Content>>> skillsDistribution = _skillsHelper.getSkillsDistribution((ProgramItem) content, 2);
                saxSkillsDistribution(skillsDistribution);
            }
            else
            {
                getLogger().warn("Cannot get the skills of a non program item '" + content.getId() + "'");
            }
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * SAX the skills distribution
     * @param distribution the skills distribution
     * @throws IOException if an I/O error occurred
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxSkillsDistribution(Map<Content, Map<Content, Map<Content, Content>>> distribution) throws IOException, SAXException
    {
        XMLUtils.startElement(contentHandler, "skills");
        
        // Iterate on skill sets
        for (Map.Entry<Content, Map<Content, Map<Content, Content>>> skillSetEntry : distribution.entrySet())
        {
            Content skillSet = skillSetEntry.getKey();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", skillSet.getId());
            attrs.addCDATAAttribute("title", skillSet.getTitle());
            attrs.addCDATAAttribute("code", skillSet.getValue(OdfReferenceTableEntry.CODE, false, ""));
            
            XMLUtils.startElement(contentHandler, "skillSet", attrs);
            
            // Iterate on skills
            for (Map.Entry<Content, Map<Content, Content>> skillEntry : skillSetEntry.getValue().entrySet())
            {
                Content skill = skillEntry.getKey();
                
                attrs.clear();
                attrs.addCDATAAttribute("id", skill.getId());
                attrs.addCDATAAttribute("title", skill.getTitle());
                attrs.addCDATAAttribute("code", skill.getValue(OdfReferenceTableEntry.CODE, false, StringUtils.EMPTY));
                
                XMLUtils.startElement(contentHandler, "skill", attrs);
                
                // Iterate on courses
                for (Map.Entry<Content, Content> courseEntry : skillEntry.getValue().entrySet())
                {
                    Content course = courseEntry.getKey();
                    
                    attrs.clear();
                    attrs.addCDATAAttribute("id", course.getId());
                    attrs.addCDATAAttribute("title", course.getTitle());
                    attrs.addCDATAAttribute("code", ((Course) course).getCode());
                    
                    XMLUtils.startElement(contentHandler, "course", attrs);
                    
                    // Acquision level can be null
                    Content acquisitionLevel = courseEntry.getValue();
                    if (acquisitionLevel != null)
                    {
                        attrs.clear();
                        attrs.addCDATAAttribute("id", acquisitionLevel.getId());
                        attrs.addCDATAAttribute("title", acquisitionLevel.getTitle());
                        attrs.addCDATAAttribute("code", acquisitionLevel.getValue(OdfReferenceTableEntry.CODE, false, ""));
                        attrs.addCDATAAttribute("order", String.valueOf(acquisitionLevel.getValue(OdfReferenceTableEntry.ORDER, false, 1)));
                        XMLUtils.createElement(contentHandler, "acquisitionLevel", attrs);
                    }
                    
                    XMLUtils.endElement(contentHandler, "course");
                }
                
                XMLUtils.endElement(contentHandler, "skill");
            }
            
            XMLUtils.endElement(contentHandler, "skillSet");
        }
        
        XMLUtils.endElement(contentHandler, "skills");
    }
}
