/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.course;

import java.io.IOException;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder.UserInput;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareComposite;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import jakarta.mail.MessagingException;

/**
 * Helper for shareable course status
 */
public class ShareableCourseStatusHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role. */
    public static final String ROLE = ShareableCourseStatusHelper.class.getName();
    
    /** The metadata name for the date of the "proposed" state */
    private static final String __PROPOSED_DATE_METADATA = "proposed_date";

    /** The metadata name for the author of the "proposed" state */
    private static final String __PROPOSED_AUTHOR_METADATA = "proposed_author";

    /** The metadata name for the date of the "validated" state */
    private static final String __VALIDATED_DATE_METADATA = "validated_date";

    /** The metadata name for the author of the "validated" state */
    private static final String __VALIDATED_AUTHOR_METADATA = "validated_author";

    /** The metadata name for the date of the "refused" state */
    private static final String __REFUSED_DATE_METADATA = "refused_date";

    /** The metadata name for the author of the "refused" state */
    private static final String __REFUSED_AUTHOR_METADATA = "refused_author";
    
    /** The metadata name for the comment of the "refused" state */
    private static final String __REFUSED_COMMENT_METADATA = "refused_comment";
    
    /** The notification right id */
    private static final String __NOTIFICATION_RIGHT_ID = "ODF_Rights_Shareable_Course_Receive_Notification";
    
    /** The propose right id */
    private static final String __PROPOSE_RIGHT_ID = "ODF_Rights_Shareable_Course_Propose";
    
    /** The validate right id */
    private static final String __VALIDATE_RIGHT_ID = "ODF_Rights_Shareable_Course_Validate";
    
    /** The refusal right id */
    private static final String __REFUSE_RIGHT_ID = "ODF_Rights_Shareable_Course_Refuse";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The observation manager */
    protected ObservationManager _observationManager;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /**
     * Enumeration for the shareable course status
     */
    public enum ShareableStatus
    {
        /** Aucun */
        NONE,
        /** Proposé */
        PROPOSED,
        /** Validé */
        VALIDATED,
        /** Refusé */
        REFUSED
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the shareable status of the course
     * @param content the content
     * @return the shareable course status
     */
    public ShareableStatus getShareableStatus(Content content)
    {
        String metadataPath = ShareableCourseConstants.SHAREABLE_COURSE_COMPOSITE_METADATA + ModelItem.ITEM_PATH_SEPARATOR + ShareableCourseConstants.SHAREABLE_COURSE_STATUS_METADATA;
        String status = content.getValue(metadataPath, false, ShareableStatus.NONE.name());
        return ShareableStatus.valueOf(status.toUpperCase());
    }
    
    /**
     * Set the workflow state attribute (date, login) to the content 
     * @param content the content
     * @param validationDate the validation date
     * @param user the user
     * @param status the shareable course status
     * @param comment the comment. Can be null
     * @param ignoreRights true to ignore user rights
     */
    public void setWorkflowStateAttribute(ModifiableContent content, LocalDate validationDate, UserIdentity user, ShareableStatus status, String comment, boolean ignoreRights)
    {
        boolean hasChanges = false;
        switch (status)
        {
            case NONE :
                if (ignoreRights || _rightManager.hasRight(user, __REFUSE_RIGHT_ID, content).equals(RightResult.RIGHT_ALLOW))
                {
                    UserIdentity proposedStateAuthor = getProposedStateAuthor(content);
                    hasChanges = setRefusedStateAttribute(content, validationDate, user, comment);
                    
                    // Send mail to users who propose the course as shareable
                    _sendNotificationMail(
                            content, 
                            proposedStateAuthor != null ? Collections.singleton(proposedStateAuthor) : new HashSet<>(),
                            comment,
                            "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_SUBJECT_ACTION_REFUSE", 
                            "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_ACTION_REFUSE"
                    );
                }
                break;
            case PROPOSED :
                if (ignoreRights || _rightManager.hasRight(user, __PROPOSE_RIGHT_ID, content).equals(RightResult.RIGHT_ALLOW))
                {
                    hasChanges = setProposedStateAttribute(content, validationDate, user);
                    
                    // Send mail to users who have the right "ODF_Rights_Shareable_Course_Receive_Notification"
                    Set<UserIdentity> users = _rightManager.getAllowedUsers(__NOTIFICATION_RIGHT_ID, content).resolveAllowedUsers(Config.getInstance().getValue("runtime.mail.massive.sending"));
                    _sendNotificationMail(
                            content, 
                            users, 
                            comment,
                            "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_SUBJECT_ACTION_PROPOSE", 
                            "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_ACTION_PROPOSE"
                    );
                }
                break;
            case VALIDATED :
                if (ignoreRights || _rightManager.hasRight(user, __VALIDATE_RIGHT_ID, content).equals(RightResult.RIGHT_ALLOW))
                {
                    hasChanges = setValidatedStateAttribute(content, validationDate, user);
                    
                    // Send mail to users who propose the course as shareable
                    UserIdentity proposedStateAuthor = getProposedStateAuthor(content);
                    if (proposedStateAuthor != null)
                    {
                        _sendNotificationMail(
                                content, 
                                Collections.singleton(proposedStateAuthor), 
                                comment,
                                "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_SUBJECT_ACTION_VALIDATE", 
                                "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_ACTION_VALIDATE"
                        );
                    }
                }
                break;
            default :
                getLogger().error("{} is an unknown shareable course status", status);
        }
        
        if (hasChanges)
        {
            _notifyShareableCourseWorkflowModification(content);
        }
    }
    
    /**
     * Remove the shareable course workflow metadata.
     * @param content The content to clean
     * @return <code>true</code> if the content has changed
     */
    public boolean removeShareableWorkflow(ModifiableContent content)
    {
        content.removeValue(ShareableCourseConstants.SHAREABLE_COURSE_COMPOSITE_METADATA);
        
        if (content.needsSave())
        {
            content.saveChanges();
            return true;
        }
        
        return false;
    }
    
    /**
     * Set the attribute for 'proposed' state
     * @param content the content
     * @param proposeDate the proposed date
     * @param user the user
     * @return <code>true</code> if the content has changed.
     */
    public boolean setProposedStateAttribute(ModifiableContent content, LocalDate proposeDate, UserIdentity user)
    {
        ModifiableModelAwareComposite composite = content.getComposite(ShareableCourseConstants.SHAREABLE_COURSE_COMPOSITE_METADATA, true);
        
        composite.setValue(__PROPOSED_DATE_METADATA, proposeDate);
        composite.setValue(__PROPOSED_AUTHOR_METADATA, user);
        
        composite.setValue(ShareableCourseConstants.SHAREABLE_COURSE_STATUS_METADATA, ShareableStatus.PROPOSED.name());

        if (content.needsSave())
        {
            content.saveChanges();
            return true;
        }
        
        return false;
    }
    
    /**
     * Set the attribute for 'validated' state
     * @param content the content
     * @param validationDate the validation date
     * @param user the login
     * @return <code>true</code> if the content has changed.
     */
    public boolean setValidatedStateAttribute(ModifiableContent content, LocalDate validationDate, UserIdentity user)
    {
        ModifiableModelAwareComposite composite = content.getComposite(ShareableCourseConstants.SHAREABLE_COURSE_COMPOSITE_METADATA, true);
        
        composite.setValue(__VALIDATED_DATE_METADATA, validationDate);
        composite.setValue(__VALIDATED_AUTHOR_METADATA, user);
        
        composite.setValue(ShareableCourseConstants.SHAREABLE_COURSE_STATUS_METADATA, ShareableStatus.VALIDATED.name());

        if (content.needsSave())
        {
            content.saveChanges();
            return true;
        }
        
        return false;
    }
    
    /**
     * Set the attribute for 'validated' state
     * @param content the content
     * @param validationDate the validation date
     * @param user the login
     * @param comment the comment. Can be null
     * @return <code>true</code> if the content has changed.
     */
    public boolean setRefusedStateAttribute(ModifiableContent content, LocalDate validationDate, UserIdentity user, String comment)
    {
        ModifiableModelAwareComposite composite = content.getComposite(ShareableCourseConstants.SHAREABLE_COURSE_COMPOSITE_METADATA, true);
        
        composite.setValue(__REFUSED_DATE_METADATA, validationDate);
        composite.setValue(__REFUSED_AUTHOR_METADATA, user);
        if (StringUtils.isNotBlank(comment))
        {
            composite.setValue(__REFUSED_COMMENT_METADATA, comment);
        }
        
        composite.setValue(ShareableCourseConstants.SHAREABLE_COURSE_STATUS_METADATA, ShareableStatus.REFUSED.name());

        if (content.needsSave())
        {
            content.saveChanges();
            return true;
        }
        
        return false;
    }
    
    /**
     * Get 'proposed' state author
     * @param content the content
     * @return the 'proposed' state author
     */
    public UserIdentity getProposedStateAuthor(Content content)
    {
        return content.getValue(ShareableCourseConstants.SHAREABLE_COURSE_COMPOSITE_METADATA + ModelItem.ITEM_PATH_SEPARATOR + __PROPOSED_AUTHOR_METADATA, false, null);
    }
    
    /**
     * Send a notification with the content modified event.
     * @param content The content to notify on
     */
    protected void _notifyShareableCourseWorkflowModification(Content content)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_MODIFIED, _currentUserProvider.getUser(), eventParams));
    }

    /**
     * Send mail to users
     * @param content the course modified
     * @param usersNotified the list of user to notify
     * @param comment the comment. Can be null
     * @param mailSubjectKey the mail subject i18n key
     * @param mailBodyKey the mail body i18n key
     */
    protected void _sendNotificationMail(Content content, Set<UserIdentity> usersNotified, String comment, String mailSubjectKey, String mailBodyKey)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();

        List<String> recipients = usersNotified.stream()
                                               .map(_userManager::getUser)
                                               .filter(Objects::nonNull)
                                               .map(User::getEmail)
                                               .filter(StringUtils::isNotBlank)
                                               .collect(Collectors.toList());
        
        try
        {
            String mailSubject = _getMailSubject(mailSubjectKey, content);
            String htmlBody = _getMailBody(mailSubjectKey, mailBodyKey, _userManager.getUser(currentUser), content, comment);
            
            SendMailHelper.newMail()
                          .withSubject(mailSubject)
                          .withHTMLBody(htmlBody)
                          .withRecipients(recipients)
                          .withAsync(true)
                          .sendMail();
        }
        catch (MessagingException | IOException e)
        {
            getLogger().warn("Could not send a notification mail to " + recipients, e);
        }
    }
    
    /**
     * Get the subject of mail
     * @param subjectI18nKey the i18n key to use for subject
     * @param content the content
     * @return the subject
     */
    protected String _getMailSubject (String subjectI18nKey, Content content)
    {
        I18nizableText subjectKey = new I18nizableText("plugin.odf", subjectI18nKey, _getSubjectI18nParams(content));
        return _i18nUtils.translate(subjectKey, content.getLanguage());
    }
    
    /**
     * Get the i18n parameters of mail subject
     * @param content the content
     * @return the i18n parameters
     */
    protected List<String> _getSubjectI18nParams (Content content)
    {
        List<String> params = new ArrayList<>();
        params.add(_contentHelper.getTitle(content));
        return params;
    }
    
    /**
     * Get the text body of mail
     * @param subjectI18nKey the i18n key to use for body
     * @param bodyI18nKey the i18n key to use for body
     * @param user the caller
     * @param content the content
     * @param comment the comment. Can be null
     * @return the text body
     * @throws IOException if failed to build HTML bodu
     */
    protected String _getMailBody (String subjectI18nKey, String bodyI18nKey, User user, Content content, String comment) throws IOException
    {
        String contentUri = _getContentUri(content);
        
        MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
            .withLanguage(content.getLanguage())
            .withTitle(_getMailSubject(subjectI18nKey, content))
            .addMessage(new I18nizableText("plugin.odf", bodyI18nKey, _getBodyI18nParams(user, content)))
            .addMessage(new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_LINK", List.of(contentUri)))
            .withLink(contentUri, new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_LINK_TITLE"))
            .withDetails(new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_FILTERS_MSG"), _getFiltersText(content), false);
                    
                    
        if (StringUtils.isNotEmpty(comment))
        {
            bodyBuilder.withUserInput(new UserInput(user, ZonedDateTime.now(), comment), new I18nizableText("plugin.cms", "WORKFLOW_MAIL_BODY_USER_COMMENT"));
        }
        
        return bodyBuilder.build();
    }
    
    /**
     * Get the i18n parameters of mail body text
     * @param user the caller
     * @param content the content
     * @return the i18n parameters
     */
    protected List<String> _getBodyI18nParams (User user, Content content)
    {
        List<String> params = new ArrayList<>();
        
        params.add(user.getFullName()); // {0}
        params.add(content.getTitle()); // {1}
        params.add(_getContentUri(content)); // {2}
        
        return params;
    }
    
    /**
     * Get the i18n parameters of mail footer text
     * @param user the caller
     * @param content the content
     * @return the i18n parameters
     */
    protected List<String> _getFooterI18nParams (User user, Content content)
    {
        List<String> params = new ArrayList<>();
        
        params.add(_getContentUri(content)); // {0}
        
        return params;
    }
    
    /**
     * Get the list of filter as text for the mail body
     * @param content the content
     * @return the list of filter as text
     */
    protected String _getFiltersText(Content content)
    {
        Map<I18nizableText, List<String>> filters = new HashMap<>();
        
        List<String> filteredPrograms = _getFilterList(content, ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME);
        if (!filteredPrograms.isEmpty())
        {
            filters.put(new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_FILTER_PROGRAMS"), filteredPrograms);
        }
        
        List<String> filteredDegrees = _getFilterList(content, ShareableCourseConstants.DEGREES_FIELD_ATTRIBUTE_NAME);
        if (!filteredDegrees.isEmpty())
        {
            filters.put(new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_FILTER_DEGREES"), filteredDegrees);
        }
        
        List<String> filteredOUs = _getFilterList(content, ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME);
        if (!filteredOUs.isEmpty())
        {
            filters.put(new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_FILTER_ORGUNITS"), filteredOUs);
        }
        
        List<String> filteredPeriods = _getFilterList(content, ShareableCourseConstants.PERIODS_FIELD_ATTRIBUTE_NAME);
        if (!filteredPeriods.isEmpty())
        {
            filters.put(new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_FILTER_PERIODS"), filteredPeriods);
        }
        
        if (filters.isEmpty())
        {
            return _i18nUtils.translate(new I18nizableText("plugin.odf", "PLUGINS_ODF_SHAREABLE_COURSE_WORKFLOW_MAIL_BODY_NO_FILTERS_MSG"), content.getLanguage());
        }
        else
        {
            StringBuilder sb = new StringBuilder();
            sb.append("<ul>");
            
            for (Entry<I18nizableText, List<String>> filter: filters.entrySet())
            {
                sb.append("<li>");
                sb.append(_i18nUtils.translate(filter.getKey(), content.getLanguage()));
                sb.append("<ul>");
                filter.getValue().stream().forEach(e -> sb.append("<li>").append(e).append("</li>"));
                sb.append("</ul>");
                sb.append("</li>");
            }
            sb.append("</ul>");
            return sb.toString();
        }
    }
    
    private List<String> _getFilterList(Content content, String attribute)
    {
        ContentValue[] values = content.getValue(attribute, false, new ContentValue[0]);
        
        return Stream.of(values)
            .map(ContentValue::getContent)
            .map(Content::getTitle)
            .collect(Collectors.toList());
    }
    
    /**
     * Get the content uri
     * @param content the content
     * @return the content uri
     */
    protected String _getContentUri(Content content)
    {
        return _contentHelper.getContentBOUrl(content, Map.of());
    }
}
