/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program.properties;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.model.properties.AbstractProperty;
import org.ametys.cms.repository.Content;
import org.ametys.core.model.type.ModelItemTypeHelper;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Property to get the primary ODF content type as single value.
 * Property's value is the content type's id prefixed by a priority integer so that values that can be sorted with following order: program, subprogram, container then course
 */
public class ODFPrimaryContentTypeProperty extends AbstractProperty<String, Content> implements Serviceable
{
    private ContentTypeExtensionPoint _cTypeEP;

    public void service(ServiceManager manager) throws ServiceException
    {
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    @Override
    public boolean isMultiple()
    {
        return false;
    }

    public Object getValue(Content content)
    {
        String cTypeId = content.getTypes()[0];
        return _getPrefixedValue(cTypeId);
    }
    
    public Object valueToJSON(Content content, DataContext context)
    {
        String cTypeId = content.getTypes()[0];
        if (cTypeId != null)
        {
            ContentType cType = _cTypeEP.getExtension(cTypeId);
            if (cType != null)
            {
                return Map.of("id", _getPrefixedValue(cTypeId), "label", cType.getLabel());
            }
        }
        return null;
    }
    
    private String _getPrefixedValue(String cTypeId)
    {
        return _getPriority(cTypeId) + "-" + cTypeId;
    }
    
    private int _getPriority(String cTypeId)
    {
        if (ProgramFactory.PROGRAM_CONTENT_TYPE.equals(cTypeId))
        {
            return 0;
        }
        else if (SubProgramFactory.SUBPROGRAM_CONTENT_TYPE.equals(cTypeId))
        {
            return 1;
        }
        else if (ContainerFactory.CONTAINER_CONTENT_TYPE.equals(cTypeId))
        {
            return 2;
        }
        else if (CourseFactory.COURSE_CONTENT_TYPE.equals(cTypeId))
        {
            return 3;
        }
        
        return -1;
    }
    
    public void valueToSAX(ContentHandler contentHandler, Content content, DataContext context) throws SAXException
    {
        String cTypeId = content.getTypes()[0];
        if (cTypeId != null)
        {
            ContentType cType = _cTypeEP.getExtension(cTypeId);
            if (cType != null)
            {
                AttributesImpl attr = ModelItemTypeHelper.getXMLAttributesFromDataContext(context);
                attr.addCDATAAttribute("id", cTypeId);
                XMLUtils.startElement(contentHandler, getName(), attr);
                cType.getLabel().toSAX(contentHandler);
                XMLUtils.endElement(contentHandler, getName());
            }
        }
        
    }
    
    @Override
    protected String _getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
}
