/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.content.indexing.solr.SolrIndexer;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.cms.search.solr.SolrClientProvider;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;

/**
 * Scheduler to compute the acquired skills on subprogram/programs from skills hold by their courses.
 */
public class ComputeProgramSkillsSchedulable extends AbstractStaticSchedulable
{
    /** The key for the catalog */
    public static final String JOBDATAMAP_CATALOG_KEY = "catalog";
    
    /** The catalog manager */
    protected CatalogsManager _catalogsManager;
    /** The solr indexer */
    protected SolrIndexer _solrIndexer;
    /** The provider for solr client */
    protected SolrClientProvider _solrClientProvider;
    /** The content searcher */
    protected ContentSearcherFactory _contentSearcherFactory;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _solrIndexer = (SolrIndexer) manager.lookup(SolrIndexer.ROLE);
        _solrClientProvider = (SolrClientProvider) manager.lookup(SolrClientProvider.ROLE);
        _contentSearcherFactory = (ContentSearcherFactory) manager.lookup(ContentSearcherFactory.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        doReindex(context, RepositoryConstants.DEFAULT_WORKSPACE);
    }
    
    /**
     * Find and reindex contents for a given workspace and catalog
     * @param context The execution context
     * @param workspaceName the workspace's name
     */
    protected void doReindex(JobExecutionContext context, String workspaceName)
    {
        long time_0 = System.currentTimeMillis();
        
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String catalogName = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_CATALOG_KEY);
        
        Set<String> contentIds = getContentIds(catalogName, workspaceName);
        indexContents(contentIds, workspaceName);
        
        getLogger().debug("Successfully reindex {} programs and subprograms for workspace '{}' and catalog '{}'  in {} ms", contentIds.size(), workspaceName, catalogName, System.currentTimeMillis() - time_0);
    }
    
    /**
     * Get the id of contents to re-index
     * @param catalogName The catalog's name. Can be empty or null to get contents of default catalog
     * @param workspaceName the workspace's name
     * @return the id of contents to re-index
     */
    protected Set<String> getContentIds(String catalogName, String workspaceName)
    {
        Request request = ContextHelper.getRequest(_context);
        String currentWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, workspaceName);
            
            String searchCatalogName = catalogName;
            if (StringUtils.isEmpty(searchCatalogName))
            {
                searchCatalogName = _catalogsManager.getDefaultCatalogName();
            }
            
            Set<String> contentIds = _contentSearcherFactory
                    .create(ProgramFactory.PROGRAM_CONTENT_TYPE, SubProgramFactory.SUBPROGRAM_CONTENT_TYPE)
                    .setCheckRights(false)
                    .search(new StringQuery(ProgramItem.CATALOG, Operator.EQ, searchCatalogName, null))
                    .stream()
                    .map(Content::getId)
                    .collect(Collectors.toSet());
            
            getLogger().debug("Found {} contents to reindex for workspace '{}' and catalog '{}'", contentIds.size(), workspaceName, searchCatalogName);
            
            return contentIds;
        }
        catch (Exception e)
        {
            getLogger().error("Unable to get content ids to reindex for updating skills", e);
            return Collections.EMPTY_SET;
        }
        finally
        {
            // Restore current workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWorkspace);
        }
    }
    
    /**
     * Index the given contents for a given workspace
     * @param contentIds the id of contents to index
     * @param workspaceName the workspace's name
     */
    protected void indexContents(Set<String> contentIds, String workspaceName)
    {
        SolrClient solrClient = _solrClientProvider.getUpdateClient(workspaceName, false);
        
        contentIds.stream().forEach(id -> 
        {
            try
            {
                _solrIndexer.indexContent(id, workspaceName, false, solrClient);
            }
            catch (Exception e)
            {
                getLogger().error("Fail to re-index content with id '{}' after computing skills", id);
            }
        });
        
        try
        {
            // Commit all uncommited changes
            _solrIndexer.commit();
        }
        catch (IOException | SolrServerException e)
        {
            getLogger().error("Unable to commit changes into Solr index after computing skills", e);
        }
    }
}
