/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.skill;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.supercsv.io.CsvListReader;
import org.supercsv.io.ICsvListReader;
import org.supercsv.prefs.CsvPreference;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.indexing.solr.SolrIndexHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * ODF skills helper
 */
public class ODFSkillsHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** The avalon role. */
    public static final String ROLE = ODFSkillsHelper.class.getName();
    
    /** The skills other names attribute name */
    public static final String SKILL_OTHER_NAMES_ATTRIBUTE_NAME = "otherNames";
    
    /** The internal attribute name to excluded from skills */
    public static final String SKILLS_EXCLUDED_INTERNAL_ATTRIBUTE_NAME = "excluded";
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;

    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;

    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The observation manager */
    protected ObservationManager _observationManager;

    /** The Solr index helper */
    protected SolrIndexHelper _solrIndexHelper;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }

    /**
     * Exclude or include the program items from skills display
     * @param programItemIds the list of program item ids
     * @param excluded <code>true</code> if the program items need to be excluded.
     * @return the map of changed program items properties
     */
    @Callable
    public Map<String, Object> setProgramItemsExclusion(List<String> programItemIds, boolean excluded)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("allright-program-items", new ArrayList<>());
        
        for (String programItemId : programItemIds)
        {
            ProgramItem programItem = _resolver.resolveById(programItemId);
            if (programItem instanceof AbstractProgram || programItem instanceof Container)
            {
                ((Content) programItem).getInternalDataHolder().setValue(SKILLS_EXCLUDED_INTERNAL_ATTRIBUTE_NAME, excluded);
                ((ModifiableAmetysObject) programItem).saveChanges();

                Map<String, Object> programItem2Json = new HashMap<>();
                programItem2Json.put("id", programItem.getId());
                programItem2Json.put("title", ((Content) programItem).getTitle());
                
                @SuppressWarnings("unchecked")
                List<Map<String, Object>> allRightProgramItems = (List<Map<String, Object>>) results.get("allright-program-items");
                allRightProgramItems.add(programItem2Json);
                
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(ObservationConstants.ARGS_CONTENT, programItem);
                eventParams.put(ObservationConstants.ARGS_CONTENT_ID, programItem.getId());
                eventParams.put(org.ametys.odf.observation.OdfObservationConstants.ODF_CONTENT_SKILLS_EXCLUSION_ARG, excluded);
                _observationManager.notify(new Event(org.ametys.odf.observation.OdfObservationConstants.ODF_CONTENT_SKILLS_EXCLUSION_CHANGED, _currentUserProvider.getUser(), eventParams));
            }
        }
        
        return results;
        
    }
    
    /**
     * <code>true</code> if the program item is excluded from skills display
     * @param programItem the program item
     * @return <code>true</code> if the program item is excluded from skills display
     */
    public boolean isExcluded(ProgramItem programItem)
    {
        if (programItem instanceof AbstractProgram || programItem instanceof Container)
        {
            return ((Content) programItem).getInternalDataHolder().getValue(SKILLS_EXCLUDED_INTERNAL_ATTRIBUTE_NAME, false);
        }
        
        return false;
    }
    
    /**
     * Get the computed skill values for a given {@link AbstractProgram}
     * @param abstractProgram the abstract program
     * @param maxDepth the max depth of courses. For example, set to 1 to compute skills over UE only, set 2 to compute skills over UE and EC, ...
     * @return the skill values computed from the attached courses
     */
    public Set<ContentValue> getComputedSkills(AbstractProgram abstractProgram, int maxDepth)
    {
        return _computeSkills(abstractProgram, 1, maxDepth);
    }
    
    private Set<ContentValue> _computeSkills(ProgramItem programItem, int depth, int maxDepth)
    {
        Set<ContentValue> skills = new HashSet<>();
        
        if (programItem instanceof Course)
        {
            // Get skills of current course
            ModifiableModelAwareRepeater repeaterSkills = ((Course) programItem).getRepeater(Course.ACQUIRED_SKILLS);
            if (repeaterSkills != null)
            {
                List< ? extends ModifiableModelAwareRepeaterEntry> entries = repeaterSkills.getEntries();
                for (ModifiableModelAwareRepeaterEntry entry : entries)
                {
                    ModifiableModelAwareRepeater repeater = entry.getRepeater(Course.ACQUIRED_SKILLS_SKILLS);
                    if (repeater != null)
                    {
                        skills.addAll(repeater.getEntries().stream()
                            .map(e -> (ContentValue) e.getValue(Course.ACQUIRED_SKILLS_SKILLS_SKILL, false, null))
                            .filter(Objects::nonNull)
                            .collect(Collectors.toSet()));
                    }
                }
            }
            
            if (depth < maxDepth)
            {
                ((Course) programItem).getCourseLists()
                    .stream()
                    .forEach(cl -> 
                    {
                        cl.getCourses()
                            .stream().forEach(c -> 
                            {
                                skills.addAll(_computeSkills(c, depth + 1, maxDepth));
                            });
                    });
            }
        }
        else 
        {
            List<ProgramItem> childProgramItems = _odfHelper.getChildProgramItems(programItem);
            for (ProgramItem childProgramItem : childProgramItems)
            {
                skills.addAll(_computeSkills(childProgramItem, depth, maxDepth));
            }
        }
        
        return skills;
    }
    
    /**
     * Get the skills distribution by courses over a {@link ProgramItem}
     * Distribution is computed over the course of first level only
     * @param programItem the program item
     * @return the skills distribution
     */
    public Map<Content, Map<Content, Map<Content, Content>>> getSkillsDistribution(ProgramItem programItem)
    {
        return getSkillsDistribution(programItem, 1);
    }
    
    /**
     * Get the skills distribution by courses over a {@link ProgramItem}
     * @param programItem the program item
     * @param maxDepth the max depth of courses. For example, set to 1 to compute distribution over UE only, set 2 to compute distribution over UE and EC, ...
     * @return the skills distribution as Map&lt;SkillSet, Map&lt;Skill, Map&lt;Course, AcquisitionLevel&gt;&gt;&gt;
     */
    public Map<Content, Map<Content, Map<Content, Content>>> getSkillsDistribution(ProgramItem programItem, int maxDepth)
    {
        // Map<SkillSet, Map<Skill, Map<Course, AcquisitionLevel>>>
        Map<Content, Map<Content, Map<Content, Content>>> skillsDistribution = new LinkedHashMap<>();
        
        if (!isExcluded(programItem))
        {
            _buildSkillsDistribution(programItem, skillsDistribution, maxDepth);
        }
        
        return skillsDistribution;
    }
    
    
    
    private void _buildSkillsDistribution(ProgramItem programItem, Map<Content, Map<Content, Map<Content, Content>>> skillsDistribution, int maxDepth)
    {
        if (programItem instanceof Course)
        {
            _buildSkillsDistribution((Course) programItem, (Course) programItem, skillsDistribution, 1, maxDepth);
        }
        else 
        {
            List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem)
                .stream()
                .filter(Predicate.not(this::isExcluded))
                .collect(Collectors.toList());
            for (ProgramItem childProgramItem : children)
            {
                _buildSkillsDistribution(childProgramItem, skillsDistribution, maxDepth);
            }
        }
    }
    
    private void _buildSkillsDistribution(Course course, Course parentCourse, Map<Content, Map<Content, Map<Content, Content>>> skillsDistribution, int depth, int maxDepth)
    {
        List<? extends ModifiableModelAwareRepeaterEntry> acquiredSkillEntries = Optional.of(course)
                .map(e -> e.getRepeater(Course.ACQUIRED_SKILLS))
                .map(ModifiableModelAwareRepeater::getEntries)
                .orElse(List.of());
        
        for (ModifiableModelAwareRepeaterEntry acquiredSkillEntry : acquiredSkillEntries)
        {
            Optional.of(acquiredSkillEntry)
                .map(e -> e.<ContentValue>getValue(Course.ACQUIRED_SKILLS_SKILLSET))
                .flatMap(ContentValue::getContentIfExists)
                .ifPresent(
                    skillSet ->
                    {
                        Map<Content, Map<Content, Content>> skills = skillsDistribution.computeIfAbsent(skillSet, __ -> new LinkedHashMap<>());
                        
                        List<? extends ModifiableModelAwareRepeaterEntry> skillEntries = Optional.of(acquiredSkillEntry)
                            .map(e -> e.getRepeater(Course.ACQUIRED_SKILLS_SKILLS))
                            .map(ModifiableModelAwareRepeater::getEntries)
                            .orElse(List.of());
                        
                        for (ModifiableModelAwareRepeaterEntry skillEntry : skillEntries)
                        {
                            Content skill = Optional.of(skillEntry)
                                .map(entry -> entry.<ContentValue>getValue(Course.ACQUIRED_SKILLS_SKILLS_SKILL))
                                .flatMap(ContentValue::getContentIfExists)
                                .orElse(null);
                            
                            if (skill != null)
                            {
                                Content acquisitionLevel = 
                                        Optional.of(skillEntry)
                                        .map(entry -> entry.<ContentValue>getValue(Course.ACQUIRED_SKILLS_SKILLS_ACQUISITION_LEVEL))
                                        .flatMap(ContentValue::getContentIfExists)
                                        .orElse(null);
                                
                                Map<Content, Content> courses = skills.computeIfAbsent(skill, s -> new LinkedHashMap<>());
                                courses.put(parentCourse, _getMaxAcquisitionLevel(acquisitionLevel, courses.get(parentCourse)));
                            }
                        }
                    }
                );
        }
        
        if (depth < maxDepth)
        {
            // Get skills distribution over child courses
            course.getCourseLists()
                .stream()
                .forEach(cl -> 
                {
                    cl.getCourses()
                        .stream().forEach(c -> 
                        {
                            _buildSkillsDistribution(c, parentCourse, skillsDistribution, depth + 1, maxDepth);
                        });
                });
        }
    }
    
    private Content _getMaxAcquisitionLevel(Content level1, Content level2)
    {
        if (level1 == null)
        {
            return level2;
        }
        
        if (level2 == null)
        {
            return level1;
        }
        
        long order1 = level1.getValue(OdfReferenceTableEntry.ORDER, false, -1L);
        long order2 = level2.getValue(OdfReferenceTableEntry.ORDER, false, -1L);
        
        if (order1 >= order2)
        {
            return level1;
        }
        else
        {
            return level2;
        }
    }
    
    /**
     * Create all skills from ESCO file
     * @param skillsCSVFilePath the skills CSV file path
     */
    public void createSkillsFromESCOFileCSV(String skillsCSVFilePath)
    {
        String[] handledEvents = new String[] {
            ObservationConstants.EVENT_CONTENT_ADDED, 
            ObservationConstants.EVENT_CONTENT_MODIFIED,  
            ObservationConstants.EVENT_CONTENT_WORKFLOW_CHANGED,
            ObservationConstants.EVENT_CONTENT_TAGGED,
            ObservationConstants.EVENT_CONTENT_DELETED,
        };
        
        try
        {
            _solrIndexHelper.pauseSolrCommitForEvents(handledEvents);
            for (Skill skill : _getSkillsFromCSVFile(skillsCSVFilePath))
            {
                try
                {
                    _createSkillTableRef(skill);
                }
                catch (AmetysRepositoryException | WorkflowException e) 
                {
                    getLogger().warn("An error occurred creating skill with label {}", skill.getLabel(), e);
                }
            }
        }
        finally
        {
            _solrIndexHelper.restartSolrCommitForEvents(handledEvents);
        }
    }
    
    /**
     * Get the list of skills from the csv file
     * @param skillsCSVFilePath the skills CSV file path
     * @return the list of skills
     */
    protected List<Skill> _getSkillsFromCSVFile(String skillsCSVFilePath)
    {
        List<Skill> skills = new ArrayList<>();
        try (BufferedReader reader = Files.newBufferedReader(Paths.get(skillsCSVFilePath), StandardCharsets.UTF_8);
             ICsvListReader listReader = new CsvListReader(reader, CsvPreference.STANDARD_PREFERENCE))
        {
            listReader.getHeader(true); //Skip header

            List<String> read = listReader.read();
            while (read != null)
            {
                String conceptUri = read.get(1); // Uri
                String label = read.get(4); // Get label
                if (StringUtils.isNotBlank(label))
                {
                    String otherNamesAsString = read.get(5); // Get other names
                    String[] otherNames = StringUtils.isNotBlank(otherNamesAsString) ? StringUtils.split(otherNamesAsString, "\n") : ArrayUtils.EMPTY_STRING_ARRAY;
                    skills.add(new Skill(label, otherNames, conceptUri));
                }
                read = listReader.read();
            }
        }
        catch (IOException e)
        {
            getLogger().warn("An error occurred parsing file {}", skillsCSVFilePath, e);
        }
        
        getLogger().info("Find {} skills into file {}", skills.size(), skillsCSVFilePath);
        
        return skills;
    }
    
    /**
     * Create a skill table ref content from the skill object
     * @param skill the skill object
     * @throws AmetysRepositoryException if a repository error occurred
     * @throws WorkflowException if a workflow error occurred
     */
    protected void _createSkillTableRef(Skill skill) throws AmetysRepositoryException, WorkflowException
    {
        String uri = skill.getConceptUri();
        String titleFR = skill.getLabel();
        String[] otherNames = skill.getOtherNames();
        
        ContentTypeExpression cTypeExpr = new ContentTypeExpression(Operator.EQ, OdfReferenceTableHelper.SKILL);
        StringExpression codeExpr = new StringExpression(OdfReferenceTableEntry.CODE, Operator.EQ, uri);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, codeExpr));
        AmetysObjectIterable<ModifiableContent> contents = _resolver.query(xpathQuery);
        AmetysObjectIterator<ModifiableContent> it = contents.iterator();
        
        if (!it.hasNext())
        {
            Map<String, String> titleVariants = new HashMap<>();
            titleVariants.put("fr", titleFR);
            
            Map<String, Object> result = _contentWorkflowHelper.createContent("reference-table", 1, titleFR, titleVariants, new String[] {OdfReferenceTableHelper.SKILL}, new String[0]);
            ModifiableContent content = (ModifiableContent) result.get(AbstractContentWorkflowComponent.CONTENT_KEY);
            
            content.setValue(OdfReferenceTableEntry.CODE, uri);
            
            if (otherNames.length > 0)
            {
                content.setValue(SKILL_OTHER_NAMES_ATTRIBUTE_NAME, otherNames);
            }
            
            content.saveChanges();
            _contentWorkflowHelper.doAction((WorkflowAwareContent) content, 22);
            
            getLogger().info("Skill's content \"{}\" ({}) was successfully created", titleFR, content.getId());
        }
    }
    
    private static class Skill
    {
        private String _label;
        private String[] _otherNames;
        private String _conceptUri;
        
        public Skill(String label, String[] otherNames, String conceptUri)
        {
            _label = label;
            _otherNames = otherNames;
            _conceptUri = conceptUri;
        }
        
        public String getLabel()
        {
            return _label;
        }

        public String[] getOtherNames()
        {
            return _otherNames;
        }
        
        public String getConceptUri()
        {
            return _conceptUri;
        }
    }
}
