/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.jcr.NameHelper.NameComputationMode;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow function for creating a {@link CoursePart} content
 */
public class CreateCoursePartFunction extends AbstractCreateODFContentFunction
{
    /** Content name prefix for programs */
    public static final String CONTENT_NAME_PREFIX = "coursepart-";
    
    /** Constant for storing the catalog name to use into the transient variables map. */
    public static final String COURSE_HOLDER_KEY = AbstractCreateODFContentFunction.class.getName() + "$courseHolder";

    @Override
    protected void _populateAdditionalData(Map transientVars, ModifiableContent content) throws WorkflowException
    {
        super._populateAdditionalData(transientVars, content);
        
        if (content instanceof CoursePart coursePart)
        {
            /* Set the catalog */

            // Try to get the catalog from transient vars
            _getValueFromTransientVars(transientVars, CONTENT_CATALOG_KEY)
                // Used to get the catalog if the course part item is created by java code, like for the csv import.
                .or(() -> _getValueFromInitialValueSupplier(transientVars, List.of(ProgramItem.CATALOG)))
                // Set the catalog if value is present
                .ifPresent(coursePart::setCatalog);
            
            /* Set the course holder */

            // Try to get the course holder from transient vars
            _getValueFromTransientVars(transientVars, COURSE_HOLDER_KEY)
                // Used to get the courseHolder if the course part item is created by java code, like for the csv import.
                .or(() -> _getCourseHolderFromInitialValueSupplier(transientVars))
                // Used to get the courseHolder if the course part item is created by java code from another content, like for the csv import.
                .or(() -> _getValueFromTransientVars(transientVars, CreateContentFunction.PARENT_CONTEXT_VALUE))
                // Set the course holder if value is present
                .ifPresent(ch -> coursePart.setValue(CoursePart.COURSE_HOLDER, ch));
            
            /* Set the code */
    
            // Used to get the code if the course part is created by java code, like for the csv import.
            _getValueFromInitialValueSupplier(transientVars, List.of(ProgramItem.CODE))
                // Set the code if value is present
                .ifPresent(coursePart::setCode);
            
            // Generate a code if empty
            String code = coursePart.getCode();
            if (StringUtils.isEmpty(code))
            {
                coursePart.setCode(org.ametys.core.util.StringUtils.generateKey().toUpperCase());
            }
        }
    }
    
    @Override
    protected String _getContentNamePrefix()
    {
        return CONTENT_NAME_PREFIX;
    }
    
    @Override
    protected String _getObjectType(Map transientVars, Map args)
    {
        return CoursePartFactory.COURSE_PART_NODETYPE;
    }
    
    @Override
    protected NameComputationMode _getDefaultNameComputationMode()
    {
        return NameComputationMode.GENERATED_KEY;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_CREATE_COURSE_PART_CONTENT_FUNCTION_LABEL");
    }
    
    private Optional<String> _getValueFromTransientVars(Map transientVars, String attributeName)
    {
        return Optional.ofNullable(transientVars.get(attributeName))
            .map(String.class::cast)
            .filter(StringUtils::isNotEmpty);
    }
    
    @SuppressWarnings("unchecked")
    private Optional<String> _getValueFromInitialValueSupplier(Map transientVars, List<String> attributePath)
    {
        return Optional.ofNullable(transientVars.get(CreateContentFunction.INITIAL_VALUE_SUPPLIER))
            .map(Function.class::cast)
            .map(function -> function.apply(attributePath))
            .map(String.class::cast)
            .filter(StringUtils::isNotEmpty);
    }
    
    private Optional<String> _getCourseHolderFromInitialValueSupplier(Map transientVars)
    {
        return _getValueFromInitialValueSupplier(transientVars, List.of(CoursePart.PARENT_COURSES, ProgramItem.CODE))
            .map(this::_searchCourseByCode);
    }
    
    private String _searchCourseByCode(String code)
    {
        Expression idExpression = new StringExpression(ProgramItem.CODE, Operator.EQ, code);
        
        Expression expression = new AndExpression(
                new ContentTypeExpression(Operator.EQ, CourseFactory.COURSE_CONTENT_TYPE),
                idExpression);

        String xPathQuery = ContentQueryHelper.getContentXPathQuery(expression);
        AmetysObjectIterable<ModifiableDefaultContent> matchingContents = _resolver.query(xPathQuery);
        if (matchingContents.getSize() == 1)
        {
            ModifiableDefaultContent course = matchingContents.iterator().next();
            return course.getId();
        }
        return null;
    }
}
