/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.person.PersonFactory;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.version.VersionAwareAmetysObject;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.AbstractWorkflowComponent.ConditionFailure;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.Step;

/**
 * Helper for ODF contents on their workflow
 *
 */
public class ODFWorkflowHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role. */
    public static final String ROLE = ODFWorkflowHelper.class.getName();
    
    /** The validate step id */
    public static final int VALIDATED_STEP_ID = 3;
    
    /** The action id of global validation */
    public static final int VALIDATE_ACTION_ID = 4;
    
    /** The action id of global unpublishment */
    public static final int UNPUBLISH_ACTION_ID = 10;
    
    /** Constant for storing the result map into the transient variables map. */
    protected static final String CONTENTS_IN_ERROR_KEY = "contentsInError";
    
    /** Constant for storing the content with right error into the transient variables map. */
    protected static final String CONTENTS_WITH_RIGHT_ERROR_KEY = "contentsWithRightError";
    
    /** Constant for storing the result map into the transient variables map. */
    protected static final String VALIDATED_CONTENTS_KEY = "validatedContents";
    
    /** Constant for storing the unpublish result map into the transient variables map. */
    protected static final String UNPUBLISHED_CONTENTS_KEY = "unpublishedContents";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    /** The workflow helper for contents */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    private enum WorkflowActionStatus 
    {
        SUCCESS,
        RIGHT_ERROR,
        OTHER_ERROR
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    /**
     * Check if the contents has referenced contents that are not already validated (children excluded)
     * @param contentIds The id of contents to check
     * @return A map with success key to true if referenced contents are validated. A map with the invalidated contents otherwise.
     */
    @Callable
    public Map<String, Object> checkReferences(List<String> contentIds)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> contentsInError = new ArrayList<>();
        
        for (String contentId : contentIds)
        {
            Set<Content> invalidatedContents = new TreeSet<>(new ContentTypeComparator());
            
            WorkflowAwareContent content = _resolver.resolveById(contentId);
            _checkValidateStep(content, invalidatedContents, false);
            
            // Remove initial content from invalidated contents
            invalidatedContents.remove(content);
            
            if (!invalidatedContents.isEmpty())
            {
                List<Map<String, Object>> invalidatedContentsAsJson = invalidatedContents.stream()
                        .map(c -> _content2Json(c))
                        .collect(Collectors.toList());
                
                Map<String, Object> contentInError = new HashMap<>();
                contentInError.put("id", content.getId());
                contentInError.put("code", ((ProgramItem) content).getCode());
                contentInError.put("title", content.getTitle());
                contentInError.put("invalidatedContents", invalidatedContentsAsJson);
                contentsInError.add(contentInError);
            }
        }
        
        result.put("contentsInError", contentsInError);
        result.put("success", contentsInError.isEmpty());
        return result;
    }
    
    /**
     * Get the global validation status of a content
     * @param contentId the id of content
     * @return the result
     */
    @Callable
    public Map<String, Object> getGlobalValidationStatus(String contentId)
    {
        Map<String, Object> result = new HashMap<>();
        
        WorkflowAwareContent waContent = _resolver.resolveById(contentId);
        
        // Order invalidated contents by types
        Set<Content> invalidatedContents = getInvalidatedContents(waContent);
        
        List<Map<String, Object>> invalidatedContentsAsJson = invalidatedContents.stream()
                .map(c -> _content2Json(c))
                .collect(Collectors.toList());
        
        result.put("invalidatedContents", invalidatedContentsAsJson);
        result.put("globalValidated", invalidatedContents.isEmpty());
        
        return result;
    }
    
    /**
     * Get the invalidated contents referenced by a ODF content
     * @param content the initial ODF content
     * @return the set of referenced invalidated contents
     */
    public Set<Content> getInvalidatedContents(WorkflowAwareContent content)
    {
        Set<Content> invalidatedContents = new TreeSet<>(new ContentTypeComparator());
        
        _checkValidateStep(content, invalidatedContents, true);
        
        return invalidatedContents;
    }
    
    /**
     * Determines if a content is already in validated step
     * @param content The content to test
     * @return true if the content is already validated
     */
    public boolean isInValidatedStep (WorkflowAwareContent content)
    {
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(content);
        long workflowId = content.getWorkflowId();
        
        List<Step> steps = workflow.getCurrentSteps(workflowId);
        for (Step step : steps)
        {
            if (step.getStepId() == VALIDATED_STEP_ID)
            {
                return true;
            }
        }
        
        return false;
    }
    
    private void _checkValidateStep (WorkflowAwareContent content, Set<Content> invalidatedContents, boolean checkChildren)
    {
        if (!isInValidatedStep(content))
        {
            invalidatedContents.add(content);
        }
        
        if (checkChildren && content instanceof ProgramItem)
        {
            // Check the structure recursively
            List<ProgramItem> children = _odfHelper.getChildProgramItems((ProgramItem) content)
                    .stream()
                    .filter(ProgramItem::isPublishable)
                    .toList();
            for (ProgramItem child : children)
            {
                WorkflowAwareContent waChild = (WorkflowAwareContent) child;
                _checkValidateStep(waChild, invalidatedContents, checkChildren);
            }
        }
        
        // Validate others referenced contents
        if (content instanceof AbstractProgram)
        {
            _checkReferencedContents(((AbstractProgram) content).getOrgUnits(), invalidatedContents, checkChildren);
            _checkReferencedContents(((AbstractProgram) content).getContacts(), invalidatedContents, checkChildren);
        }
        else if (content instanceof Course)
        {
            _checkReferencedContents(((Course) content).getOrgUnits(), invalidatedContents, checkChildren);
            _checkReferencedContents(((Course) content).getContacts(), invalidatedContents, checkChildren);
        }
        else if (content instanceof OrgUnit)
        {
            _checkReferencedContents(((OrgUnit) content).getContacts(), invalidatedContents, checkChildren);
        }
    }
    
    private void _checkReferencedContents(Collection<String> refContentIds, Set<Content> invalidatedContents, boolean recursively)
    {
        for (String id : refContentIds)
        {
            try
            {
                if (StringUtils.isNotEmpty(id))
                {
                    WorkflowAwareContent refContent = _resolver.resolveById(id);
                    if (recursively)
                    {
                        _checkValidateStep(refContent, invalidatedContents, recursively);
                    }
                    else if (!isInValidatedStep(refContent))
                    {
                        invalidatedContents.add(refContent);
                    }
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // Nothing
            }
        }
    }
    
    /**
     * Global validation on a contents.
     * Validate the contents with their whole structure and the others referenced contacts and orgunits.
     * @param contentIds the id of contents to validation recursively
     * @return the result for each initial contents
     */
    @Callable
    public Map<String, Object> globalValidate(List<String> contentIds)
    {
        Map<String, Object> result = new HashMap<>();
        
        for (String contentId : contentIds)
        {
            Map<String, Object> contentResult = new HashMap<>();
            
            contentResult.put(CONTENTS_IN_ERROR_KEY, new HashSet<>());
            contentResult.put(CONTENTS_WITH_RIGHT_ERROR_KEY, new HashSet<>());
            contentResult.put(VALIDATED_CONTENTS_KEY, new HashSet<>());
            
            ProgramItem programItem = _resolver.resolveById(contentId);
            if (programItem.isPublishable())
            {
                _validateRecursively((WorkflowAwareContent) programItem, contentResult);
            }
            
            @SuppressWarnings("unchecked")
            Set<Content> contentsInError = (Set<Content>) contentResult.get(CONTENTS_IN_ERROR_KEY);
            List<Map<String, Object>> contentsInErrorAsJson = contentsInError.stream()
                    .map(c -> _content2Json(c))
                    .toList();
            
            contentResult.put(CONTENTS_IN_ERROR_KEY, contentsInErrorAsJson);
            
            @SuppressWarnings("unchecked")
            Set<Content> contentsWithRightError = (Set<Content>) contentResult.get(CONTENTS_WITH_RIGHT_ERROR_KEY);
            List<Map<String, Object>> contentsWithRightErrorAsJson = contentsWithRightError.stream()
                    .map(c -> _content2Json(c))
                    .toList();
            
            contentResult.put(CONTENTS_WITH_RIGHT_ERROR_KEY, contentsWithRightErrorAsJson);
            
            
            result.put(contentId, contentResult);
        }
        
        return result;
    }
    
    /**
     * Get the JSON representation of the content
     * @param content the content
     * @return the content properties
     */
    protected Map<String, Object> _content2Json(Content content)
    {
        Map<String, Object> content2json = new HashMap<>();
        content2json.put("title", content.getTitle());
        content2json.put("id", content.getId());
        
        if (content instanceof ProgramItem)
        {
            content2json.put("code", ((ProgramItem) content).getCode());
        }
        else if (content instanceof OrgUnit)
        {
            content2json.put("code", ((OrgUnit) content).getUAICode());
        }
        
        return content2json;
    }
    
    /**
     * Validate the referenced contents recursively
     * @param content The validated content
     * @param result the result object to fill during process
     */
    protected void _validateRecursively (WorkflowAwareContent content, Map<String, Object> result)
    {
        @SuppressWarnings("unchecked")
        Set<String> validatedContentIds = (Set<String>) result.get(VALIDATED_CONTENTS_KEY);
        @SuppressWarnings("unchecked")
        Set<Content> contentsInError = (Set<Content>) result.get(CONTENTS_IN_ERROR_KEY);
        @SuppressWarnings("unchecked")
        Set<Content> contentsWithRightError = (Set<Content>) result.get(CONTENTS_WITH_RIGHT_ERROR_KEY);
        
        if (!isInValidatedStep(content))
        {
            // Validate content itself
            WorkflowActionStatus status = _doValidateWorkflowAction (content, VALIDATE_ACTION_ID);
            if (status == WorkflowActionStatus.RIGHT_ERROR)
            {
                contentsWithRightError.add(content);
            }
            else if (status == WorkflowActionStatus.OTHER_ERROR)
            {
                contentsInError.add(content);
            }
            else
            {
                validatedContentIds.add(content.getId());
            }
        }
        
        if (content instanceof ProgramItem)
        {
            // Validate the structure recursively
            List<ProgramItem> children = _odfHelper.getChildProgramItems((ProgramItem) content)
                    .stream()
                    .filter(ProgramItem::isPublishable)
                    .toList();
            for (ProgramItem child : children)
            {
                _validateRecursively((WorkflowAwareContent) child, result);
            }
        }
        
        // Validate others referenced contents
        if (content instanceof AbstractProgram)
        {
            _validateReferencedContents(((AbstractProgram) content).getOrgUnits(), result);
            _validateReferencedContents(((AbstractProgram) content).getContacts(), result);
        }
        else if (content instanceof Course)
        {
            _validateReferencedContents(((Course) content).getOrgUnits(), result);
            _validateReferencedContents(((Course) content).getContacts(), result);
        }
        else if (content instanceof OrgUnit)
        {
            _validateReferencedContents(((OrgUnit) content).getContacts(), result);
        }
    }
    
    /**
     * Validate the list of referenced contents
     * @param refContentIds The id of contents to validate
     * @param result the result object to fill during process
     */
    protected void _validateReferencedContents (Collection<String> refContentIds, Map<String, Object> result)
    {
        @SuppressWarnings("unchecked")
        Set<String> validatedContentIds = (Set<String>) result.get(VALIDATED_CONTENTS_KEY);
        @SuppressWarnings("unchecked")
        Set<Content> contentsInError = (Set<Content>) result.get(CONTENTS_IN_ERROR_KEY);
        @SuppressWarnings("unchecked")
        Set<Content> contentsWithRightError = (Set<Content>) result.get(CONTENTS_WITH_RIGHT_ERROR_KEY);
        
        for (String id : refContentIds)
        {
            try
            {
                if (StringUtils.isNotEmpty(id))
                {
                    WorkflowAwareContent content = _resolver.resolveById(id);
                    if (!isInValidatedStep(content))
                    {
                        WorkflowActionStatus status = _doValidateWorkflowAction (content, VALIDATE_ACTION_ID);
                        if (status == WorkflowActionStatus.RIGHT_ERROR)
                        {
                            contentsWithRightError.add(content);
                        }
                        else if (status == WorkflowActionStatus.OTHER_ERROR)
                        {
                            contentsInError.add(content);
                        }
                        else
                        {
                            validatedContentIds.add(content.getId());
                        }
                    }
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // Nothing
            }
        }
    }
    
    /**
     * Validate a content
     * @param content The content to validate
     * @param actionId The id of validate action
     * @return the validate workflow status
     */
    protected WorkflowActionStatus _doValidateWorkflowAction (WorkflowAwareContent content, int actionId)
    {
        Map<String, Object> inputs = new HashMap<>();
        try
        {
            _contentWorkflowHelper.doAction(content, actionId, inputs, false);
            return WorkflowActionStatus.SUCCESS;
        }
        catch (InvalidActionException | WorkflowException e)
        {
            String failureString = _getActionFailuresAsString(inputs);
            if (e instanceof InvalidActionException)
            {
                getLogger().warn("Unable to validate content \"{}\" ({}): mandatory metadata are probably missing or the content is locked{}", content.getTitle(), content.getId(), failureString, e);
            }
            else
            {
                getLogger().warn("Failed to validate content \"{}\" ({}){}", content.getTitle(), content.getId(), failureString, e);
            }
            return _isRightConditionFailure(inputs) ? WorkflowActionStatus.RIGHT_ERROR : WorkflowActionStatus.OTHER_ERROR;
        }
    }
    
    private boolean _isRightConditionFailure(Map<String, Object> actionInputs)
    {
        if (actionInputs.containsKey(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY))
        {
            @SuppressWarnings("unchecked")
            List<ConditionFailure> failures = (List<ConditionFailure>) actionInputs.getOrDefault(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY, new ArrayList<>());
            return failures.stream()
                        .filter(c -> c.type().equals(CheckRightsCondition.class.getName()))
                        .findFirst()
                        .isPresent();
        }
        
        return false;
    }
    
    private String _getActionFailuresAsString(Map<String, Object> actionInputs)
    {
        String failuresAsString = "";
        if (actionInputs.containsKey(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY))
        {
            @SuppressWarnings("unchecked")
            List<ConditionFailure> failures = (List<ConditionFailure>) actionInputs.getOrDefault(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY, new ArrayList<>());
            if (!failures.isEmpty())
            {
                failuresAsString = ", due to the following error(s):\n" + String.join("\n", failures.stream().map(ConditionFailure::text).toList());
            }
        }
        
        return failuresAsString;
    }
    
    /**
     * Set the publishable state of contents
     * @param contentIds The id of contents
     * @param isPublishable <code>true</code> to set content as publishable, <code>false</code> otherwise
     * @return The result map
     */
    @Callable
    public Map<String, Object> setPublishableState (List<String> contentIds, boolean isPublishable)
    {
        Map<String, Object> result = new HashMap<>();
        
        for (String id : contentIds)
        {
            Map<String, Object> contentResult = new HashMap<>();
            Set<Content> contentsInError = new HashSet<>();
            contentResult.put(CONTENTS_IN_ERROR_KEY, contentsInError);
            Set<Content> contentsWithRightError = new HashSet<>();
            contentResult.put(CONTENTS_WITH_RIGHT_ERROR_KEY, contentsWithRightError);
            contentResult.put(UNPUBLISHED_CONTENTS_KEY, new HashSet<>());
            
            WorkflowAwareContent content = _resolver.resolveById(id);
            if (content instanceof ProgramItem programItem)
            {
                try
                {
                    programItem.setPublishable(isPublishable);
                    content.saveChanges();

                    if (!isPublishable)
                    {
                        _unpublishRecursively(programItem, contentResult);
                    }
                }
                catch (Exception e)
                {
                    getLogger().error("Unable to set publishable property for content '{}' with id '{}'", content.getTitle(), content.getId(), e);
                    contentsInError.add(content);
                }

                List<Map<String, Object>> contentsInErrorAsJson = contentsInError.stream()
                        .map(c -> _content2Json(c))
                        .collect(Collectors.toList());
                contentResult.put(CONTENTS_IN_ERROR_KEY, contentsInErrorAsJson);
                
                List<Map<String, Object>> contentsWithRightErrorAsJson = contentsWithRightError.stream()
                        .map(c -> _content2Json(c))
                        .collect(Collectors.toList());
                contentResult.put(CONTENTS_WITH_RIGHT_ERROR_KEY, contentsWithRightErrorAsJson);
                
                result.put(id, contentResult);
            }
        } 
        
        return result;
    }
    
    /**
     * Unpublish the referenced contents recursively
     * @param programItem The content to unpublish
     * @param result the result object to fill during process
     */
    protected void _unpublishRecursively (ProgramItem programItem, Map<String, Object> result)
    {
        @SuppressWarnings("unchecked")
        Set<String> unpublishedContentIds = (Set<String>) result.get(UNPUBLISHED_CONTENTS_KEY);
        @SuppressWarnings("unchecked")
        Set<Content> contentsInError = (Set<Content>) result.get(CONTENTS_IN_ERROR_KEY);
        @SuppressWarnings("unchecked")
        Set<Content> contentsWithRightError = (Set<Content>) result.get(CONTENTS_WITH_RIGHT_ERROR_KEY);
        
        if (_isPublished(programItem))
        {
            // Unpublish content itself
            WorkflowActionStatus status = _doUnpublishWorkflowAction((WorkflowAwareContent) programItem, UNPUBLISH_ACTION_ID);
            if (status == WorkflowActionStatus.RIGHT_ERROR)
            {
                contentsWithRightError.add((Content) programItem);
            }
            else if (status == WorkflowActionStatus.OTHER_ERROR)
            {
                contentsInError.add((Content) programItem);
            }
            else
            {
                unpublishedContentIds.add(programItem.getId());
            }
        }

        // Unpublish the structure recursively
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem child : children)
        {
            boolean hasOtherPublishedParent = _odfHelper.getParentProgramItems(child)
                .stream()
                .filter(p -> !p.getId().equals(programItem.getId()))
                .filter(this::_isPublished)
                .findFirst()
                .isPresent();
                
            // Don't unpublish the content if it has an other published parent
            if (!hasOtherPublishedParent)
            {
                _unpublishRecursively(child, result);
            }
        }
    }
    
    /**
     * Unpublish a content
     * @param content The content to unpublish
     * @param actionId The id of unpublish action
     * @return the workflow action status
     */
    protected WorkflowActionStatus _doUnpublishWorkflowAction (WorkflowAwareContent content, int actionId)
    {
        Map<String, Object> inputs = new HashMap<>();
        try
        {
            _contentWorkflowHelper.doAction(content, actionId, inputs, false);
            return WorkflowActionStatus.SUCCESS;
        }
        catch (Exception e)
        {
            String failureString = _getActionFailuresAsString(inputs);
            getLogger().warn("Failed to unpublish content \"{}\" ({}){}", content.getTitle(), content.getId(), failureString, e);
            return _isRightConditionFailure(inputs) ? WorkflowActionStatus.RIGHT_ERROR : WorkflowActionStatus.OTHER_ERROR;
        }
    }
    
    /**
     * <code>true</code> if the parent is publishable
     * @param content the content
     * @return <code>true</code> if the parent is publishable
     */
    public boolean isParentPublishable(ProgramItem content)
    {
        List<ProgramItem> parents = _odfHelper.getParentProgramItems(content);
        if (parents.isEmpty())
        {
            return true;
        }
        
        return parents.stream()
                .filter(c -> c.isPublishable() && isParentPublishable(c))
                .findAny()
                .isPresent();
    }
    
    private boolean _isPublished(ProgramItem content)
    {
        return content instanceof VersionAwareAmetysObject versionAAO ? ArrayUtils.contains(versionAAO.getAllLabels(), CmsConstants.LIVE_LABEL) : false;
    }
    
    class ContentTypeComparator implements Comparator<Content>
    {
        String[] _orderedContentTypes = new String[] {
            ProgramFactory.PROGRAM_CONTENT_TYPE,
            SubProgramFactory.SUBPROGRAM_CONTENT_TYPE,
            ContainerFactory.CONTAINER_CONTENT_TYPE,
            CourseListFactory.COURSE_LIST_CONTENT_TYPE,
            CourseFactory.COURSE_CONTENT_TYPE,
            OrgUnitFactory.ORGUNIT_CONTENT_TYPE,
            PersonFactory.PERSON_CONTENT_TYPE
        };
                
        @Override
        public int compare(Content c1, Content c2)
        {
            if (c1 == c2)
            {
                return 0;
            }
            
            String cTypeId1 = c1.getTypes()[0];
            String cTypeId2 = c2.getTypes()[0];
            
            int i1 = ArrayUtils.indexOf(_orderedContentTypes, cTypeId1);
            int i2 = ArrayUtils.indexOf(_orderedContentTypes, cTypeId2);
            
            if (i1 == i2)
            {
                // order by title for content of same type
                int compareTo = c1.getTitle().compareTo(c2.getTitle());
                if (compareTo == 0)
                {
                    // for content of same title, order by id to do not return 0 to add it in TreeSet
                    // Indeed, in a TreeSet implementation two elements that are equal by the method compareTo are, from the standpoint of the set, equal 
                    return c1.getId().compareTo(c2.getId());
                }
                else
                {
                    return compareTo;
                }
            }
            
            return i1 != -1 && i1 < i2 ? -1 : 1;
        }
    }
    
}
