/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/*
 * This script prepares the migration plugin.odf.jcr "v2020-1130T0010-ODF-2790".
 * If the first run of the CMS from the add of this migration is in safe mode and
 * the user has to update general configuration, old config values
 * "odf.programs.degree.license", "odf.programs.degree.licensepro" and
 * "odf.programs.degree.master" will be lost. Then it saves the config file before
 * the user update it.
 */

/* Imports */
const File = Java.type("java.io.File");
const IOUtils = Java.type("org.apache.commons.io.IOUtils");
const FileUtils = Java.type("org.apache.commons.io.FileUtils");

/* Components */
const xPathProcessor = Ametys.serviceManager.lookup(org.apache.excalibur.xml.xpath.XPathProcessor.ROLE);
const domParser = Ametys.serviceManager.lookup(org.apache.excalibur.xml.dom.DOMParser.ROLE);

/* Get Ametys config file */
let configFolder = org.ametys.runtime.util.AmetysHomeHelper.getAmetysHomeConfig();
let configFilename = org.ametys.runtime.servlet.RuntimeServlet.CONFIG_FILE_NAME;

let destFile = new File(configFolder, configFilename + ".ODF-2790");
if (destFile.exists())
{
    logger.warn("The file '" + destFile.getName() + "' already exists.");
}
else
{
    let srcFile = new File(configFolder, configFilename);
    
    if (srcFile.exists())
    {
        let fis = new java.io.FileInputStream(srcFile);
        try
        {
            let doc = domParser.parseDocument(new org.xml.sax.InputSource(fis));
            let configNode = doc.getFirstChild();
            
            let somethingToMigrate = xPathProcessor.selectSingleNode(configNode, "odf.programs.degree.license") != null;
            somethingToMigrate |= xPathProcessor.selectSingleNode(configNode, "odf.programs.degree.licensepro") != null;
            somethingToMigrate |= xPathProcessor.selectSingleNode(configNode, "odf.programs.degree.master") != null;
            
            if (somethingToMigrate)
            {
                // If something to migrate : config.xml file exists and contains one of the required config attribute
                FileUtils.copyFile(srcFile, destFile);
                logger.debug("The file '" + srcFile.getName() + "' has been correctly copied to '" + destFile.getName() + "'");
            }
        }
        catch (e)
        {
            logger.error("An error has occured during copying the file '" + srcFile.getName() + "' to '" + destFile.getName() + "'", e);
            throw e;
        }
        finally
        {
            IOUtils.closeQuietly(fis);
        }
    }
}
