/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.xslt.AmetysXSLTHelper;
import org.ametys.core.util.dom.AmetysNodeList;
import org.ametys.core.util.dom.StringElement;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.xslt.OdfReferenceTableElement;
import org.ametys.odf.xslt.ProgramElement;
import org.ametys.odf.xslt.SubProgramElement;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.runtime.config.Config;

/**
 * Helper component to be used from XSL stylesheets.
 */
public class OdfXSLTHelper implements Serviceable
{
    /** The ODF helper */
    protected static ODFHelper _odfHelper;
    /** The ODF reference helper */
    protected static OdfReferenceTableHelper _odfRefTableHelper;
    /** The Ametys resolver */
    protected static AmetysObjectResolver _ametysObjectResolver;
    /** The orgunit root provider */
    protected static RootOrgUnitProvider _rootOrgUnitProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) smanager.lookup(OdfReferenceTableHelper.ROLE);
        _ametysObjectResolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _rootOrgUnitProvider = (RootOrgUnitProvider) smanager.lookup(RootOrgUnitProvider.ROLE);
    }
    
    /**
     * Get the label associated with the degree key
     * @param cdmValue The code of degree
     * @return The label of degree or code if not found
     */
    public static String degreeLabel (String cdmValue)
    {
        return degreeLabel(cdmValue, Config.getInstance().getValue("odf.programs.lang"));
    }
    
    /**
     * Get the code associated with the given reference table's entry
     * @param tableRefEntryId The id of entry
     * @return the code or <code>null</code> if not found
     */
    public static String getCode (String tableRefEntryId)
    {
        try
        {
            Content content = _ametysObjectResolver.resolveById(tableRefEntryId);
            return content.getValue(OdfReferenceTableEntry.CODE);
        }
        catch (AmetysRepositoryException e)
        {
            return null;
        }
    }
    
    /**
     * Get the id of reference table's entry
     * @param tableRefId The id of content type
     * @param code The code
     * @return the id or <code>null</code> if not found
     */
    public static String getEntryId (String tableRefId, String code)
    {
        OdfReferenceTableEntry entry = _odfRefTableHelper.getItemFromCode(tableRefId, code);
        if (entry != null)
        {
            return entry.getId();
        }
        return null;
    }
    
    /**
     * Get the label associated with the degree key
     * @param cdmValue The cdm value of degree
     * @param lang The language
     * @return The label of degree or empty string if not found
     */
    public static String degreeLabel (String cdmValue, String lang)
    {
        return Optional
            .ofNullable(_odfRefTableHelper.getItemFromCDM(OdfReferenceTableHelper.DEGREE, cdmValue))
            .map(degree -> degree.getLabel(lang))
            .orElse(StringUtils.EMPTY);
    }
    
    /**
     * Get the whole structure of a subprogram, including the structure of child subprograms
     * @param subprogramId The id of subprogram
     * @return Node with the subprogram structure
     */
    public static Node getSubProgramStructure (String subprogramId)
    {
        SubProgram subProgram = _ametysObjectResolver.resolveById(subprogramId);
        return new SubProgramElement(subProgram, _odfHelper);
    }
    
    /**
     * Get the structure of a subprogram, including the structure of child subprograms until the given depth
     * @param subprogramId The id of subprogram
     * @param depth Set a positive number to get structure of child subprograms until given depth. Set a negative number to get the whole structure recursively, including the structure of child subprograms. This parameter concerns only subprograms.
     * @return Node with the subprogram structure
     */
    public static Node getSubProgramStructure (String subprogramId, int depth)
    {
        SubProgram subProgram = _ametysObjectResolver.resolveById(subprogramId);
        return new SubProgramElement(subProgram, depth, null, _odfHelper);
    }
    
    /**
     * Get the parent program information
     * @param subprogramId The id of subprogram
     * @return a node for each program's information
     */
    public static NodeList getParentProgram (String subprogramId)
    {
        return getParentProgramStructure(subprogramId, 0);
    }
    
    /**
     * Get the certification label of a {@link AbstractProgram}.
     * Returns null if the program is not certified.
     * @param abstractProgramId the id of program or subprogram
     * @return the certification label
     */
    public static String getCertificationLabel(String abstractProgramId)
    {
        AbstractProgram abstractProgram = _ametysObjectResolver.resolveById(abstractProgramId);
        if (abstractProgram.isCertified())
        {
            String degreeId = null;
            if (abstractProgram instanceof Program)
            {
                degreeId = ((Program) abstractProgram).getDegree();
            }
            else if (abstractProgram instanceof SubProgram)
            {
                // Get degree from parent
                Set<Program> rootPrograms = _odfHelper.getParentPrograms(abstractProgram);
                if (rootPrograms.size() > 0)
                {
                    degreeId = rootPrograms.iterator().next().getDegree();
                }
            }
            
            if (StringUtils.isNotEmpty(degreeId))
            {
                Content degree = _ametysObjectResolver.resolveById(degreeId);
                return degree.getValue("certificationLabel");
            }
        }
        
        return null;
    }
    
    /**
     * Get the program information
     * @param programId The id of program
     * @return Node with the program's information
     */
    public static Node getProgram (String programId)
    {
        return getProgramStructure(programId, 0);
    }
    
    /**
     * Get the structure of a parent programs, including the structure of child subprograms until the given depth.
     * @param subprogramId The id of subprogram
     * @param depth Set a positive number to get structure of child subprograms until given depth. Set a negative number to get the whole structure recursively, including the structure of child subprograms. This parameter concerns only subprograms.
     * @return a node for each program's structure
     */
    public static NodeList getParentProgramStructure (String subprogramId, int depth)
    {
        List<ProgramElement> programs = new ArrayList<>();
        
        SubProgram subProgram = _ametysObjectResolver.resolveById(subprogramId);
        
        Set<Program> rootPrograms = _odfHelper.getParentPrograms(subProgram);
        if (rootPrograms.size() > 0)
        {
            programs.add(new ProgramElement(rootPrograms.iterator().next(), depth, null, _odfHelper));
        }
        
        return new AmetysNodeList(programs);
    }
    
    /**
     * Get the paths of a {@link ProgramItem} util the root program(s)
     * Paths are built with content's title
     * @param programItemId the id of program items
     * @param separator The path separator
     * @return the paths in program
     */
    public static NodeList getProgramPaths(String programItemId, String separator)
    {
        ProgramItem programItem = _ametysObjectResolver.resolveById(programItemId);
        List<String> paths = _odfHelper.getPaths(programItem, separator, p -> ((Content) p).getTitle(), false);
        
        List<Element> result = paths.stream()
            .map(path -> new StringElement("path", (Map<String, String>) null, path))
            .collect(Collectors.toList());
        
        return new AmetysNodeList(result);
    }
    
    /**
     * Get the structure of a program until the given depth.
     * @param programId The id of program
     * @param depth Set a positive number to get structure of child subprograms until given depth. Set a negative number to get the whole structure recursively, including the structure of child subprograms. This parameter concerns only subprograms.
     * @return Node with the program structure
     */
    public static Node getProgramStructure (String programId, int depth)
    {
        Program program = _ametysObjectResolver.resolveById(programId);
        return new ProgramElement(program, depth, null, _odfHelper);
    }
    
    /**
     * Get the items of a reference table
     * @param tableRefId the id of reference table
     * @param lang the language to use for labels
     * @return the items
     */
    public static Node getTableRefItems(String tableRefId, String lang)
    {
        return getTableRefItems(tableRefId, lang, false, true);
    }
    
    /**
     * Get the items of a reference table
     * @param tableRefId the id of reference table
     * @param lang the language to use for labels
     * @param ordered true to sort items by 'order' attribute
     * @return the items
     */
    public static Node getTableRefItems(String tableRefId, String lang, boolean ordered)
    {
        return getTableRefItems(tableRefId, lang, ordered, true);
    }
    
    /**
     * Get the items of a reference table
     * @param tableRefId the id of reference table
     * @param lang the language to use for labels
     * @param ordered true to sort items by 'order' attribute
     * @param includeArchived true to include archived items
     * @return the items
     */
    public static Node getTableRefItems(String tableRefId, String lang, boolean ordered, boolean includeArchived)
    {
        return new OdfReferenceTableElement(tableRefId, _odfRefTableHelper, lang, ordered, includeArchived);
    }
    
    /**
     * Get the id of root orgunit
     * @return The id of root
     */
    public static String getRootOrgUnitId()
    {
        return _rootOrgUnitProvider.getRootId();
    }
    
    /**
     * Get the id of the first orgunit matching the given UAI code
     * @param uaiCode the UAI code
     * @return the id of orgunit or null if not found
     */
    public static String getOrgUnitIdByUAICode(String uaiCode)
    {
        return Optional.ofNullable(uaiCode)
                .map(_odfHelper::getOrgUnitByUAICode)
                .map(OrgUnit::getId)
                .orElse(null);
    }
    
    /**
     * Get the more recent educational booklet for a {@link ProgramItem}
     * @param programItemId the program item id
     * @return the pdf as an ametys node list
     */
    public static AmetysNodeList getEducationalBooklet(String programItemId)
    {
        try
        {
            Content programItem = _ametysObjectResolver.resolveById(programItemId);
            if (programItem.hasValue(ProgramItem.EDUCATIONAL_BOOKLETS))
            {
                ModelAwareRepeater repeater = programItem.getRepeater(ProgramItem.EDUCATIONAL_BOOKLETS);
                ZonedDateTime dateToCompare = null;
                ModelAwareRepeaterEntry entry = null;
                for (ModelAwareRepeaterEntry repeaterEntry : repeater.getEntries())
                {
                    ZonedDateTime date = repeaterEntry.getValue("date");
                    if (dateToCompare == null || date.isAfter(dateToCompare))
                    {
                        dateToCompare = date;
                        entry = repeaterEntry;
                    }
                }
                
                if (entry != null)
                {
                    SAXTransformerFactory saxTransformerFactory = (SAXTransformerFactory) TransformerFactory.newInstance();
                    TransformerHandler th = saxTransformerFactory.newTransformerHandler();
                    
                    DOMResult result = new DOMResult();
                    th.setResult(result);
                    
                    th.startDocument();
                    XMLUtils.startElement(th, "value");
                    if (entry.hasValue("pdf"))
                    {
                        programItem.dataToSAX(th, ProgramItem.EDUCATIONAL_BOOKLETS + "[" + entry.getPosition() + "]/pdf");
                    }
                    XMLUtils.endElement(th, "value");
                    th.endDocument();
                    
                    List<Node> values = new ArrayList<>();
                    
                    // #getChildNodes() returns a NodeList that contains the value(s) saxed
                    // we cannot returns directly this NodeList because saxed values should be wrapped into a <value> tag.
                    NodeList childNodes = result.getNode().getFirstChild().getChildNodes();
                    for (int i = 0; i < childNodes.getLength(); i++)
                    {
                        Node n = childNodes.item(i);
                        values.add(n);
                    }
                    
                    return new AmetysNodeList(values);
                }
            }
        }
        catch (Exception e)
        {
            return null;
        }
        
        return null;
    }

    /**
     * Count the hours accumulation in the {@link ProgramItem}
     * @param contentId The id of the {@link ProgramItem}
     * @return The hours accumulation
     */
    public static Double getCumulatedHours(String contentId)
    {
        return _odfHelper.getCumulatedHours(_ametysObjectResolver.<ProgramItem>resolveById(contentId));
    }
    
    /**
     * Get the ECTS value at the given educational path
     * @param courseId The course id
     * @param coursePathAsString The course path as string (semicolon separated). Can be a partial path.
     * @return the ECTS value for the given educational path
     */
    public static NodeList getEcts(String courseId, String coursePathAsString)
    {
        return getValueForPath(courseId, Course.ECTS_BY_PATH, coursePathAsString, Course.ECTS);
    }
    
    /**
     * Get the attribute of a content at the given path for a given educational path
     * @param programItemId The program item id
     * @param dataPath The path of attribute to retrieve. The path must contain the path of repeater holding the value (ex: path/to/repeater/attributeName)
     * @param programItemPathAsString The program item path as string (semicolon separated). Can be a partial path
     * @param defaultValuePath The data path of the default value in there is no specific value at the given educational path
     * @return The value into a "value" node or null if an error occurred
     */
    public static NodeList getValueForPath(String programItemId, String dataPath, String programItemPathAsString, String defaultValuePath)
    {
        ProgramItem programItem = _ametysObjectResolver.resolveById(programItemId);
        
        String[] programItemIds = programItemPathAsString.split(EducationalPath.PATH_SEGMENT_SEPARATOR);
        List<ProgramItem> programItemPath = Stream.of(programItemIds).map(id -> _ametysObjectResolver.<ProgramItem>resolveById(id)).toList();
        List<EducationalPath> educationaPaths = _odfHelper.getEducationPathFromPath(programItemPath);
                
        int position = _odfHelper.getRepeaterEntryPositionForPath(programItem, dataPath, educationaPaths);
        
        String repeaterPath = StringUtils.substringBeforeLast(dataPath, "/");
        String attributeName = StringUtils.substringAfterLast(dataPath, "/");
        
        return AmetysXSLTHelper.contentAttribute(programItemId, position != -1 ? repeaterPath + "[" + position + "]/" + attributeName : defaultValuePath);
    }
    
    /**
     * Convert a duration in minutes to a string representing the duration in hours.
     * @param duree in minutes
     * @return the duration in hours
     */
    public static String minute2hour(int duree)
    {
        int h = duree / 60;
        int m = duree % 60;
        return String.format("%dh%02d", h, m);
    }
}
