/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.NoLiveVersionException;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.LOMSheet;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.person.ContactData;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.odf.program.WebsiteLink;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Component able to generate CDM-fr for a {@link CDMEntity}.
 */
public class ExportCDMfrManager extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = ExportCDMfrManager.class.getName();
    
    /** Request attribute to get if the export is for Ametys application */
    public static final String REQUEST_ATTRIBUTE_EXPORT_FOR_AMETYS = ExportCDMfrManager.class.getName() + "$forAmetys";
    
    private static final String[] __FIXED_DOMAIN_NAMES = new String[]{"ALL", "DEG", "SHS", "STS", "STA"};
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The source resolver */
    protected SourceResolver _sourceResolver;
    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    /** The root orgunit provider */
    protected RootOrgUnitProvider _rootOrgUnitProvider;
    /** The avalon context */
    protected Context _context;
    /** The CDMfr extension point */
    protected CDMfrExtensionPoint _cdmFrExtensionPoint;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    /** Helper for content types */
    protected ContentTypesHelper _cTypeHelper;
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _refTableHelper = (OdfReferenceTableHelper) smanager.lookup(OdfReferenceTableHelper.ROLE);
        _rootOrgUnitProvider = (RootOrgUnitProvider) smanager.lookup(RootOrgUnitProvider.ROLE);
        _cdmFrExtensionPoint = (CDMfrExtensionPoint) smanager.lookup(CDMfrExtensionPoint.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _cTypeHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Entry point to generate the CDM for a program
     * @param contentHandler The content handler to sax into
     * @param program The program
     * @throws SAXException if failed to generate CDM
     */
    public void generateCDM(ContentHandler contentHandler, Program program) throws SAXException
    {
        contentHandler.startDocument();
        contentHandler.startPrefixMapping("", "http://cdm-fr.fr/2013/CDM");
        contentHandler.startPrefixMapping("xsi", "http://www.w3.org/2001/XMLSchema-instance");
        contentHandler.startPrefixMapping("cdmfr", "http://cdm-fr.fr/2013/CDM-frSchema");
        contentHandler.startPrefixMapping("xhtml", "http://www.w3.org/1999/xhtml");
        contentHandler.startPrefixMapping("ametys-cdm", "http://www.ametys.org/cdm/1.0");
            
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("http://www.w3.org/2001/XMLSchema-instance", "schemaLocation", "xsi:schemaLocation", "http://cdm-fr.fr/2013/CDM http://cdm-fr.fr/2013/schemas/CDMFR.xsd");
        attrs.addCDATAAttribute("profile", "CDM-fr");
        attrs.addCDATAAttribute("language", program.getLanguage());
        XMLUtils.startElement(contentHandler, "CDM", attrs);
        
        saxHabilitation(contentHandler, program);
        
        XMLUtils.startElement(contentHandler, "properties");
        XMLUtils.createElement(contentHandler, "datasource", "Ametys v4.2");
        attrs = new AttributesImpl();
        attrs.addCDATAAttribute("date", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        XMLUtils.createElement(contentHandler, "datetime", attrs);
        XMLUtils.endElement(contentHandler, "properties");
        
        program2CDM(contentHandler, program);
        
        XMLUtils.endElement(contentHandler, "CDM");
        
        contentHandler.endDocument();
    }
    
    /**
     * Switch to Live version if it was required
     * @param ao the Ametys object
     * @return true if the ametys object is switched
     */
    protected boolean switchToLiveVersionIfNeeded(DefaultAmetysObject ao)
    {
        try
        {
            _odfHelper.switchToLiveVersionIfNeeded(ao);
            return true;
        }
        catch (NoLiveVersionException e)
        {
            getLogger().warn("Cannot export content to CDM : Live label is required but there is no Live version for content " + ao.getId());
        }
        
        return false;
    }
    
    /**
     * Determines if this CDM export if for a Ametys application
     * @return true if this CDM export if for a Ametys application
     */
    protected boolean isExportForAmetys()
    {
        Request request = ContextHelper.getRequest(_context);
        return request.getAttribute(ExportCDMfrManager.REQUEST_ATTRIBUTE_EXPORT_FOR_AMETYS) != null;
    }
    
    /**
     * SAX CDMfr events for program's habilitation
     * @param contentHandler the content handler to sax into
     * @param program the program
     * @throws SAXException if an error occurred
     */
    public void saxHabilitation(ContentHandler contentHandler, Program program) throws SAXException
    {
        /*
         * <cdmfr:habilitation>
         *      <cdmfr:habiliId>ID</cdmfr:habiliId>
         *      <cdmfr:cohabilitation>
         *          <cdmfr:exists>true|false</cdmfr:exists> // Habilité ou non
         *          <cdmfr:listOfOrgUnit>
         *              <cdmfr:habOrgUnit>
         *                  <cdmfr:refOrgUnit> // Etablisssement cohabilité
         *                  <cdmfr:domainName>
         *                      <cdmfr:fixedDomain> // Domaine concerné par cette habilitation
         *                  </cdmfr:domainName>
         *              </cdmfr:habOrgUnit>
         *          </cdmfr:listOfOrgUnit>
         *      </cdmfr:cohabilitation>
         *      <cdmfr:field>
         *          <fieldname>
         *              <free>MENTION</free>
         *          </fieldname>
         *          <speciality>
         *              <specialityId/>
         *              <specialityName/>
         *          </speciality>
         *      </cdmfr:field>
         *      <cdmfr:partnership>
         *          <cdmfr:training>
         *              <cdmfr:trainingStrategy /> // Stages
         *          </cdmfr:training>
         *      </cdmfr:partnership>
         * </cdmfr:habilitation>
         * 
         */
        
        if (program.isPublishable() && switchToLiveVersionIfNeeded(program))
        {
            Request request = ContextHelper.getRequest(_context);
            Content currentContent = (Content) request.getAttribute(Content.class.getName());

            try
            {
                request.setAttribute(Content.class.getName(), program);
                
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID, "");
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_DIPLOMA_TYPE, getDiplomaType(program));
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_HABILITATION, attrs);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_HABILITATION_ID, program.getCDMId());
                
                _saxCohabilitation(contentHandler, program);
                _saxHabilitationJointOrgUnit(contentHandler, program);
                _saxHabilitationDiploma(contentHandler, program);
                _saxHabilitationPartnerShip(contentHandler, program);
                
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_HABILITATION);
            }
            finally
            {
                request.setAttribute(Content.class.getName(), currentContent);
            }
        }
    }
    
    /**
     * Sax habilitation informations of the diploma.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationDiploma(ContentHandler contentHandler, Program program) throws SAXException
    {
        String speciality = program.getSpeciality();
        String mention = program.getMention();
        
        if (StringUtils.isNotEmpty(mention) || StringUtils.isNotEmpty(speciality))
        {
            // Formation LMD : Mention - Speciality
            /*
             * <cdmfr:field>
             *      <cdmfr:fieldName>
             *          <cdmfr:controlled fieldNameCode="code_mention">
             *              <cdmfr:registeredName>Libelle mention</cdmfr:registeredName>
             *          </cdmfr:controlled>
             *      </cdmfr:fieldName>
             *      <cdmfr:speciality>
             *          <cdmfr:specialityName/>
             *          <cdmfr:specialityId/>
             *          <cdmfr:refProgram/>
             *      </cdmfr:speciality>
             * </cdmfr:field>
             */
            
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "field");
            _saxHabilitationMention(contentHandler, program, mention);
            _saxHabilitationSpeciality(contentHandler, program, speciality);
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "field");
        }
        else if (OdfReferenceTableHelper.MENTION_LICENCEPRO.equals(_getMentionType(program)))
        {
            _saxHabilitationLicPro(contentHandler, program);
        }
        else
        {
            _saxHabilitationOtherDiploma(contentHandler, program);
        }
    }
    
    /**
     * Sax habilitation informations of professional license.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationLicPro(ContentHandler contentHandler, Program program) throws SAXException
    {
        // Licence professionelle
        /*
         * <cdmfr:licPro>
         *      <cdmfr:specialite>
         *          <cdm:text/>
         *      </cdmfr:specialite>
         * </cdmfr:licPro>
         */
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "licPro");
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "specialite");
        XMLUtils.createElement(contentHandler, "text", program.getSpeciality());
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "specialite");
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "licPro");
    }

    /**
     * Sax habilitation informations for another diploma : no mention, not a license.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationOtherDiploma(ContentHandler contentHandler, Program program) throws SAXException
    {
        // Autre diplôme habilité ne s'inscrivant pas dans une structure mention-spécialité
        /*
         * <cdmfr:otherDiploma>
         *      <cdmfr:refProgram/>
         * </cdmfr:otherDiploma>
         */
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "otherDiploma");
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "refProgram", program.getCDMId());
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "otherDiploma");
    }

    /**
     * Sax habilitation informations of mention.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @param mention The mention to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationMention(ContentHandler contentHandler, Program program, String mention) throws SAXException
    {
        if (StringUtils.isNotEmpty(mention))
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "fieldName");
            
            if (ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID.equals(program.getType("mention").getId()))
            {
                OdfReferenceTableEntry entry = _refTableHelper.getItem(mention);
                
                String mentionLabel = entry.getCdmValue();
                if (StringUtils.isEmpty(mentionLabel))
                {
                    mentionLabel = entry.getLabel(program.getLanguage());
                }
                
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("fieldNameCode", entry.getCode());
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "controlled", attrs);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "registeredName", mentionLabel);
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "controlled");
            }
            else
            {
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "free", mention);
            }
            
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "fieldName");
        }
    }

    /**
     * Sax habilitation informations of speciality.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @param speciality The speciality to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationSpeciality(ContentHandler contentHandler, Program program, String speciality) throws SAXException
    {
        if (StringUtils.isNotEmpty(speciality))
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "speciality");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "specialityName", speciality);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "specialityId", program.getCDMId() + "SP01"); // FIXME "SP + n° ordre de la spécialité"
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "refProgram", program.getCDMId());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "speciality");
        }
    }

    /**
     * Sax cohabilitation informations.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxCohabilitation(ContentHandler contentHandler, Program program) throws SAXException
    {
        String[] domains = program.getDomain();
        boolean isHabilited = false;
        for (String domain : domains)
        {
            isHabilited |= !"HD".equals(_refTableHelper.getItemCode(domain));
        }
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "cohabilitation");
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "exists", String.valueOf(isHabilited));
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "listOfOrgUnit");
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "habOrgUnit");
        
        String rneCode = _getRootOrgUnitId(program);
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "refOrgUnit", rneCode);
        
        _saxHabilitationDomains(contentHandler, domains);
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "habOrgUnit");
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "listOfOrgUnit");
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "cohabilitation");
    }

    /**
     * Sax habilitation informations of domains.
     * @param contentHandler The content handler
     * @param domains The domains to SAX
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationDomains(ContentHandler contentHandler, String[] domains) throws SAXException
    {
        for (String domain : domains)
        {
            Content content = _resolver.resolveById(domain);
            OdfReferenceTableEntry entry = new OdfReferenceTableEntry(content);
            String domainName = entry.getCdmValue();
            String domainCode = entry.getCode();
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "domainName");
            if (ArrayUtils.contains(__FIXED_DOMAIN_NAMES, domainCode))
            {
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "fixedDomain", domainName);
            }
            else
            {
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "openDomain", domainName);
            }
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "domainName");
        }
    }

    /**
     * Sax habilitation informations of joint orgunits.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationJointOrgUnit(ContentHandler contentHandler, Program program) throws SAXException
    {
        // Etablissement cohabilité
        String[] jointOrgUnit = program.getJointOrgUnit();
        if (jointOrgUnit.length > 0)
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "cohabilitation");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "exists", String.valueOf(true));
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("cohabilitationRole", "cohabilité");
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "listOfOrgUnit", attrs);
            for (String id : jointOrgUnit)
            {
                if (StringUtils.isNotEmpty(id))
                {
                    String uaiCode = _refTableHelper.getItemCode(id);
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "habOrgUnit");
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "refOrgUnit", uaiCode);
                    String ouLabel = _refTableHelper.getItemLabel(id, program.getLanguage());
                    if (ouLabel != null)
                    {
                        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, ouLabel);
                    }
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "habOrgUnit");
                }
            }
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "listOfOrgUnit");
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "cohabilitation");
        }
    }

    /**
     * Sax habilitation informations of partner ship.
     * @param contentHandler The content handler
     * @param program The program to sax
     * @throws SAXException if an error occurs
     */
    protected void _saxHabilitationPartnerShip(ContentHandler contentHandler, Program program) throws SAXException
    {
        /*
         * <cdmfr:partnership>
         *     <cdmfr:training>
         *          <cdmfr:trainingStrategy />
         *     </cdmfr:training>
         * </cdmfr:partnership>
         */
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "partnership");
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "training");
        DataContext richTextContext = RepositoryDataContext.newInstance()
                                                           .withObject(program)
                                                           .withDataPath(AbstractProgram.TRAINING_STRATEGY);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "trainingStrategy", program.getTrainingStrategy(), richTextContext, _sourceResolver, false, isExportForAmetys());
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "training");
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + "partnership");
    }
    
    /**
     * Get the root organization unit uai code
     * @param program the exported program
     * @return the uai code of the root organization unit corresponding to the exported program
     */
    protected String _getRootOrgUnitId(Program program)
    {
        return _rootOrgUnitProvider.getRoot().getUAICode();
    }
    
    /**
     * Get the type of diploma
     * @param program the program
     * @return diploma type
     */
    protected String getDiplomaType (Program program)
    {
        String degreeMentionType = _getMentionType(program);
        if (OdfReferenceTableHelper.MENTION_LICENCE.equals(degreeMentionType))
        {
            return "L";
        }
        else if (OdfReferenceTableHelper.MENTION_MASTER.equals(degreeMentionType))
        {
            return "M";
        }
        else if (OdfReferenceTableHelper.MENTION_LICENCEPRO.equals(degreeMentionType))
        {
            return "Lpro";
        }
        return "autre";
    }

    //***********************************************************************//
    //                              PROGRAM                                  //
    //***********************************************************************//
    
    /**
     * Export this entity in CDM-fr format.
     * @param contentHandler the target handler
     * @param program the program
     * @throws SAXException if an error occurs during export.
     */
    public void program2CDM(ContentHandler contentHandler, Program program) throws SAXException
    {
        boolean rootOrgUnitSaxed = false;
        
        Set<String> persons = new HashSet<>();
        Set<String> orgUnits = new HashSet<>();
        Set<String> coursesToSax = new HashSet<>();
        
        abstractProgram2CDM(contentHandler, program, persons, orgUnits, coursesToSax);
        
        Set<String> coursesAlreadySax = new HashSet<>();
        
        while (!coursesToSax.isEmpty())
        {
            Set<String> courses = new HashSet<>();
            for (String courseId : coursesToSax)
            {
                coursesAlreadySax.add(courseId);
                Course course = _resolver.resolveById(courseId);
                courses.addAll(course2CDM(contentHandler, course, orgUnits, persons));
            }
            
            coursesToSax.clear();
            coursesToSax.addAll(courses);
            coursesToSax.removeAll(coursesAlreadySax);
        }

        for (String orgUnitId : orgUnits)
        {
            OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
            if (orgUnitId.equals(_rootOrgUnitProvider.getRootId()))
            {
                rootOrgUnitSaxed = true;
            }
            
            _orgunit2CDM(contentHandler, orgUnit, persons);
        }
        
        if (!rootOrgUnitSaxed)
        {
            _orgunit2CDM(contentHandler, _rootOrgUnitProvider.getRoot(), persons);
        }
        
        for (String personId : persons)
        {
            Person person = _resolver.resolveById(personId);
            person2CDM(contentHandler, person);
        }
    }
    
    /**
     * Send the CDM-fr representation of their ProgramPart to the given Contenthandler.<br>
     * Also collects referenced orgUnits, persons and courses, so that they could be sent afterwards.
     * @param contentHandler the receiving contentHandler.
     * @param program the program
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @param courses collected {@link Course} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void abstractProgram2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program, Set<String> persons, Set<String> orgUnits, Set<String> courses) throws SAXException
    {
        DataContext programRichTextsContext = RepositoryDataContext.newInstance()
                                                                   .withObject(program);
        if (program.isPublishable() && switchToLiveVersionIfNeeded(program))
        {
            Request request = ContextHelper.getRequest(_context);
            Content currentContent = (Content) request.getAttribute(Content.class.getName());
            
            try
            {
                request.setAttribute(Content.class.getName(), program);
                
                String tagName = program.getCDMTagName();
            
                AttributesImpl attrs = new AttributesImpl();
                
                // <program id="FRUAI{rootOrgUnitId}PR{code}" language="...">
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID, program.getCDMId());
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LANGUAGE, program.getLanguage());
                XMLUtils.startElement(contentHandler, tagName, attrs);
                
                // <programID>code</programID>
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PROGRAM_ID, program.getCDMId());
        
                // <programName><text>name</text></programName>
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_NAME);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_TEXT, program.getTitle());
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_NAME);
                
                _abstractProgram2CDMCode(contentHandler, program);
                
                // <webLink>
                //      <href>href</href>
                //      <linkName>linkName</linkName>
                // </webLink>
                for (WebsiteLink website : program.getWebsiteLinks())
                {
                    website.toCDM(contentHandler);
                }
                
                // <programDescription>...</programDescription>
                attrs = new AttributesImpl();
                String programType = program.getEducationKind();
                if (StringUtils.isNotEmpty(programType))
                {
                    programType = _refTableHelper.getItemCDMfrValue(programType, false);
                    _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_NATURE, programType);
                }
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_DESCRIPTION, attrs);
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, program.getPresentation(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.PRESENTATION), _sourceResolver, false, isExportForAmetys());
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_DESCRIPTION);
                
                // <qualification>...</qualification>
                _qualification2CDM(contentHandler, program, programRichTextsContext);
                
                // <levelCode @codeSet="bac|bac+1|bac+2|bac+3|..."/>
                // <level @level="L|M|Doc"/>
                _level2CDM(contentHandler, program);
                
                // <learningObjectives>...</learningObjectives>
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_LEARNING_OBJECTIVES, program.getObjectives(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.OBJECTIVES), _sourceResolver, false, isExportForAmetys());
                
                // <admissionInfo>
                //      <cdmfr:admissionDescription>...</cdmfr:admissionDescription>
                // </admissionInfo>
                _admissionInfo2CDM(contentHandler, program, programRichTextsContext);
                
                // <recommendedPrerequisites>...</recommendedPrerequisites>
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_RECOMMANDED_PREREQUISITES, program.getRecommendedPrerequisite(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.RECOMMENDED_PREREQUISITE), _sourceResolver, false, isExportForAmetys());
        
                // <formalPrerequisites>...</formalPrerequisites>
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_FORMAL_PREREQUISITES, program.getNeededPrerequisite(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.NEEDED_PREREQUISITE), _sourceResolver, true, isExportForAmetys());
                
                // <teachingPlace>...</teachingPlace>
                _place2CDM(contentHandler, program);
                
                // <targetGroup>...</targetGroup>
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_TARGET_GROUP, program.getTargetGroup(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.TARGET_GROUP), _sourceResolver, false, isExportForAmetys());
        
                // <formOfTeaching org="" method=""/>
                String formOfTeachingMethod = _refTableHelper.getItemCDMfrValue(program.getDistanceLearning(), false);
                for (String formOfTeachingOrg: program.getFormOfTeachingOrgs())
                {
                    attrs = new AttributesImpl();
                    String formOfTeachingOrgValue = _refTableHelper.getItemCDMfrValue(formOfTeachingOrg, false);
                    _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_METHOD, formOfTeachingMethod);
                    _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_ORG, formOfTeachingOrgValue);
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_FORM_OF_TEACHING, attrs);
                }
    
                // <instructionLanguage teachingLang="fr"/>
                String[] langs = program.getEducationLanguage();
                for (String lang : langs)
                {
                    String teachingLanguageValue = _refTableHelper.getItemCDMfrValue(lang, true);
                    attrs = new AttributesImpl();
                    attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_TEACHING_LANG, teachingLanguageValue);
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INSTRUCTION_LANGUAGE, attrs);
                }
                
                // <studyAbroad>...</studoAbroad>
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_STUDY_ABROAD, program.getStudyAbroad(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.STUDY_ABROAD), _sourceResolver, false, isExportForAmetys());
                
                // <programDuration>...</programDuration>
                String duration = program.getDuration();
                if (StringUtils.isNotBlank(duration))
                {
                    String durationValue = _refTableHelper.getItemCDMfrValue(duration, true);
                    if (durationValue != null)
                    {
                        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PROGRAM_DURATION, durationValue);
                    }
                }
                
                // <cdmfr:programStructure>...</cdmfr:programStructure>
                RichText teachingOrganization = program.getTeachingOrganization();
                if (teachingOrganization != null && teachingOrganization.getLength() > 0)
                {
                    XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_STRUCTURE);
                    CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, program.getTeachingOrganization(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.TEACHING_ORGANIZATION), _sourceResolver, false, isExportForAmetys());
                    XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_STRUCTURE);
                }
                
                // Sax courses list
                ContentValue[] children = program.getValue(TraversableProgramPart.CHILD_PROGRAM_PARTS, false, new ContentValue[0]);
                for (ContentValue child : children)
                {
                    Optional<ModifiableContent> content = child.getContentIfExists();
                    if (content.isPresent() && content.get() instanceof CourseList courseList)
                    {
                        courselist2CDM(contentHandler, courseList, courses);
                    }
                }
                
                // <regulations/>
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_REGULATIONS);
        
                // <expenses>...</expenses>
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_EXPENSES, program.getExpenses(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.EXPENSES), _sourceResolver, false, isExportForAmetys());
                
                // <universalAjustment>...</universalAjustment>
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_UNIVERSAL_ADJUSTMENT, program.getUniversalAdjustment(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.UNIVERSAL_ADJUSTMENT), _sourceResolver, false, isExportForAmetys());
        
                // <contacts>
                //      <refPerson idRef="ID"/>
                //      <refOrgUnit idRef="ID"/>
                // </contacts>
                _abstractProgram2CDMContacts(contentHandler, program, persons, orgUnits);
                
                // <infoBlock>...</infoBlock>
                attrs = new AttributesImpl();
                if (tagName.equals(CDMFRTagsConstants.TAG_SUB_PROGRAM))
                {
                    attrs.addCDATAAttribute("userDefined", "subProgram");
                }
                
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, program.getAdditionalInformations(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.ADDITIONNAL_INFORMATIONS), _sourceResolver, false, isExportForAmetys(), attrs);
        
                // <infoBlock userDefined="ametys-extension>...</infoBlock>
                saxExtensions(contentHandler, program, persons, orgUnits);
                
                // <programStructure>...</programStructure>
                _programPart2CDMSubPrograms(contentHandler, program, persons, orgUnits, courses);
                
                // <searchword>keyword</searchword>
                for (String keyword : program.getKeywords())
                {
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_SEARCH_WORD, keyword);
                }
                
                // </program>
                XMLUtils.endElement(contentHandler, tagName);
            }
            finally
            {
                request.setAttribute(Content.class.getName(), currentContent);
            }
        }
    }
    
    /**
     * Export CDM extensions handling all elements that do not belong to the CDM-fr schema
     * @param contentHandler the receiving contentHandler.
     * @param content the current ODF Content
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    protected void saxExtensions(ContentHandler contentHandler, Content content, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("userDefined", "ametys-extension");
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, attrs);
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_EXTENSION_BLOCK);
        
        for (String id : _cdmFrExtensionPoint.getExtensionsIds())
        {
            CDMfrExtension cdmFrExtension = _cdmFrExtensionPoint.getExtension(id);
            
            if (content instanceof AbstractProgram<?> abstractProgram)
            {
                if (abstractProgram instanceof Program program)
                {
                    cdmFrExtension.program2CDM(contentHandler, program, persons, orgUnits);
                }
                else if (abstractProgram instanceof SubProgram subProgram)
                {
                    cdmFrExtension.subProgram2CDM(contentHandler, subProgram, persons, orgUnits);
                }
                
                cdmFrExtension.abstractProgram2CDM(contentHandler, abstractProgram, persons, orgUnits);
            }
            else if (content instanceof Container container)
            {
                cdmFrExtension.container2CDM(contentHandler, container, persons, orgUnits);
            }
            else if (content instanceof Course course)
            {
                cdmFrExtension.course2CDM(contentHandler, course, persons, orgUnits);
            }
            else if (content instanceof Person person)
            {
                cdmFrExtension.person2CDM(contentHandler, person);
            }
            else if (content instanceof OrgUnit orgunit)
            {
                cdmFrExtension.orgunit2CDM(contentHandler, orgunit);
            }
        }
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_EXTENSION_BLOCK);
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK);
    }

    /**
     * SAX CDM-fr teaching places
     * @param contentHandler the content handler to sax into
     * @param program the program or subprogram
     * @throws SAXException if an error occurred
     */
    protected void _place2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        // <teachingPlace>
        //      <cdmfr:refOrgUnit />
        //      <adr>
        //          <pcode></pcode>
        //      </adr>
        // </teachingPlace>
        String[] places = program.getPlace();
        for (String id : places)
        {
            if (StringUtils.isNotEmpty(id))
            {
                String postalCode = _refTableHelper.getItemCode(id);
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_TEACHING_PLACE);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REF_ORG_UNIT);
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ADDRESS);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PCODE, postalCode);
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ADDRESS);
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_TEACHING_PLACE);
            }
        }
    }

    /**
     * SAX CDM-fr qualification
     * @param contentHandler the content handler to sax into
     * @param program the program or subprogram
     * @param programRichTextsContext the context for program's rich texts
     * @throws SAXException if an error occurred
     */
    protected void _qualification2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program, DataContext programRichTextsContext) throws SAXException
    {
        // <qualification>
        //      <qualificatioName><text></text></qualificationName>
        //      <qualificationDescription>
        //          <infoBlock>...</infoBlock>
        //          <statistics>success rate</statistics>
        //      </qualificationDescription>
        //      <degree degree=""/>
        //      <profession>...</profession>
        //      <studyQualification>...</studyQualification>
        // </qualification>
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_QUALIFICATION);
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_QUALIFICATION_NAME);
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEXT);
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_QUALIFICATION_NAME);
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_QUALIFICATION_DESCRIPTION);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, program.getQualification(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.QUALIFICATION), _sourceResolver, true, isExportForAmetys());
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_STATISTICS, program.getSuccessRate());
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_QUALIFICATION_DESCRIPTION);
        
        AttributesImpl attrs = new AttributesImpl();
        String ects = _refTableHelper.getItemCDMfrValue(program.getEcts(), true);
        _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_ECTS_CREDITS, ects);
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_CREDITS, attrs);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, program.getKnowledgeCheck(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.KNOWLEDGE_CHECK), _sourceResolver, true, isExportForAmetys());
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_CREDITS);

        _degree2CDM(contentHandler, program);

        // <profession>
        //      <infoBlock>...</infoBlock>
        //      <romeData romeCode="code" romeLibel=""/>
        // </profession>
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROFESSION);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, program.getJobOpportunities(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.JOB_OPPORTUNITIES), _sourceResolver, true, isExportForAmetys());
        _romeCodes2CDM(contentHandler, program);
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROFESSION);
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_STUDY_QUALIFICATION);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, program.getFurtherStudy(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.FURTHER_STUDY), _sourceResolver, true, isExportForAmetys());
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_COMPETENCE_DESC);
        attrs = new AttributesImpl();
        attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LIMITED, "false");
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_COMPETENCE_VALIDITY, attrs);
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_STUDY_QUALIFICATION);
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_QUALIFICATION);
    }
    
    /**
     * SAX CDMfr degree
     * @param contentHandler the content handler to sax into
     * @param abstractProgram the program or subprogram
     * @throws SAXException if an error occurred
     */
    protected void _degree2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> abstractProgram) throws SAXException
    {
        if (abstractProgram instanceof Program program)
        {
            AttributesImpl attrs = new AttributesImpl();
            String degreeId = program.getDegree();
            if (StringUtils.isNotEmpty(degreeId))
            {
                String cdmCode = _refTableHelper.getItemCDMfrValue(degreeId, false);
                if (StringUtils.isNotEmpty(cdmCode))
                {
                    _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_DEGREE, cdmCode);
                }
                else
                {
                    _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_DEGREE_NOT_LMD, _refTableHelper.getItemCode(degreeId));
                }
                
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_DEGREE, attrs);
            }
        }
    }

    /**
     * SAX CDMfr rome codes
     * @param contentHandler the content handler to sax into
     * @param program the program or subprogram
     * @throws SAXException if an error occurred
     */
    protected void _romeCodes2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        String[] romeCode = program.getRomeCode();
        for (String id : romeCode)
        {
            AttributesImpl attrs = new AttributesImpl();
            String code = _refTableHelper.getItemCode(id);
            attrs.addCDATAAttribute("romeCode", code);
            String label = _refTableHelper.getItemCDMfrValue(id, false);
            if (StringUtils.isNotBlank(label))
            {
                attrs.addCDATAAttribute("romeLibel", label);
            }
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_ROME_DATA, attrs);
        }
    }

    /**
     * SAX CDMfr level
     * @param contentHandler the content handler to sax into
     * @param program the program or subprogram
     * @throws SAXException if an error occurred
     */
    protected void _level2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        String educationLevel = program.getEducationLevel();
        _addNotNullAttribute(attrs, "codeSet", _refTableHelper.getItemCDMfrValue(educationLevel, true));
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_LEVEL_CODE, attrs);
        
        attrs = new AttributesImpl();
        String educationLevelCode = _refTableHelper.getItemCode(educationLevel);
        if (educationLevelCode.startsWith("licence"))
        {
            _addNotNullAttribute(attrs, "level", "L");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_LEVEL, attrs);
        }
        else if (educationLevelCode.startsWith("master"))
        {
            _addNotNullAttribute(attrs, "level", "M");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_LEVEL, attrs);
        }
        else if (educationLevelCode.startsWith("doctorat"))
        {
            _addNotNullAttribute(attrs, "level", "Doc");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_LEVEL, attrs);
        }
    }

    /**
     * SAX CDMfr admission info
     * @param contentHandler the content handler to sax into
     * @param program the program or subprogram
     * @param programRichTextsContext the context for program's rich texts
     * @throws SAXException if an error occurred
     */
    protected void _admissionInfo2CDM(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program, DataContext programRichTextsContext) throws SAXException
    {
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ADMISSION_INFO);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_ADMISSION_DESCRIPTION, program.getAccessCondition(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.ACCESS_CONDITION), _sourceResolver, false, isExportForAmetys());
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_STUDENT_STATUS, "");
        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_ECTS_REQUIRED, "");
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_STUDENT_PLACES, program.getEffectives(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.NUMBER_OF_STUDENTS), _sourceResolver, false, isExportForAmetys());
        
        LocalDate teachingStart = program.getTeachingStart();
        if (teachingStart != null)
        {
            CDMHelper.date2CDM(contentHandler, CDMFRTagsConstants.TAG_TEACHING_START, teachingStart);
        }
        
        LocalDate registrationDeadline = program.getRegistrationDeadline();
        if (registrationDeadline != null)
        {
            CDMHelper.date2CDM(contentHandler, CDMFRTagsConstants.TAG_REGISTRATION_DEADLINE, registrationDeadline);
        }
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REGISTRATION_DETAIL);
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REGISTRATION_PROCESS);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REGISTRATION_MODALITIES, program.getInscription(), programRichTextsContext.cloneContext().withDataPath(AbstractProgram.INSCRIPTION), _sourceResolver, false, isExportForAmetys());
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REGISTRATION_PROCESS);
        LocalDate registrationStart = program.getRegistrationStart();
        if (registrationStart != null)
        {
            CDMHelper.date2CDM(contentHandler, CDMFRTagsConstants.TAG_REGISTRATION_START, registrationStart);
        }
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REGISTRATION_DETAIL);
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ADMISSION_INFO);
    }
    
    /**
     * SAX CDMfr events for a {@link AbstractProgram} codes
     * @param contentHandler the content handler to sax into
     * @param program the program or subprogram
     * @throws SAXException if an error occurred
     */
    protected void _abstractProgram2CDMCode(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        
        // <programCode codeSet="userDefined">code</programCode>
        String code = program.getCode();
        if (StringUtils.isNotBlank(code))
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "userDefined");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, code);
        }
        
        // <programCode codeSet="codeCNIS-NSF">code</programCode>
        // <programCode libSet="codeCNIS-NSF">label</programCode>
        String nsfId = ContentDataHelper.getContentIdFromContentData(program, "nsfCode");
        if (StringUtils.isNotBlank(nsfId))
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "codeCNIS-NSF");
            
            String nsfCode = _refTableHelper.getItemCode(nsfId);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, nsfCode);
            
            String codeLib = _refTableHelper.getItemCDMfrValue(nsfId, false);
            if (StringUtils.isNotEmpty(codeLib))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LIB_SET, "codeCNIS-NSF");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, codeLib);
            }
        }
        
        // <programCode codeSet="sectDGESIP">code</programCode>
        // <programCode libSet="sectDGESIP">label</programCode>
        for (String id : program.getDGESIPCode())
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "sectDGESIP");
            
            String dgesipCode = _refTableHelper.getItemCode(id);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, dgesipCode);
            
            String codeLib = _refTableHelper.getItemCDMfrValue(id, false);
            if (StringUtils.isNotEmpty(codeLib))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LIB_SET, "sectDGESIP");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, codeLib);
            }
        }
        
        // <programCode codeSet="ERASMUS">code</programCode>
        // <programCode libSet="ERASMUS">label</programCode>
        for (String id : program.getErasmusCode())
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "ERASMUS");
            
            String erasmusCode = _refTableHelper.getItemCode(id);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, erasmusCode);
            
            String codeLib = _refTableHelper.getItemCDMfrValue(id, false);
            if (StringUtils.isNotEmpty(codeLib))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LIB_SET, "ERASMUS");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, codeLib);
            }
        }
        
        // <programCode codeSet="cite97">code</programCode>
        // <programCode libSet="cite97">label</programCode>
        for (String id : program.getCite97Code())
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "cite97");
            String cite97Code = _refTableHelper.getItemCode(id);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, cite97Code);
            
            String codeLib = _refTableHelper.getItemCDMfrValue(id, false);
            if (StringUtils.isNotEmpty(codeLib))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LIB_SET, "cite97");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, codeLib);
            }
        }
        
        // <programCode codeSet="FAP">code</programCode>
        // <programCode libSet="FAP">label</programCode>
        for (String id : program.getFapCode())
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "FAP");
            String fapCode = _refTableHelper.getItemCode(id);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, fapCode);
            
            String codeLib = _refTableHelper.getItemCDMfrValue(id, false);
            if (StringUtils.isNotEmpty(codeLib))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LIB_SET, "FAP");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, codeLib);
            }
        }
        
        // <programCode codeSet="codeSectDiscipSISE">code</programCode>
        // <programCode libSet="codeSectDiscipSISE">label</programCode>
        for (String id : program.getSiseCode())
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "sectDiscipSISE");
            
            String siseCode = _refTableHelper.getItemCode(id);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, siseCode);
            
            String codeLib = _refTableHelper.getItemCDMfrValue(id, false);
            if (StringUtils.isNotEmpty(codeLib))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LIB_SET, "sectDiscipSISE");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, codeLib);
            }
        }
    }
    
    /**
     * SAX CDMfr events for a {@link AbstractProgram} contacts
     * @param contentHandler the content handler to sax into
     * @param program the program or subprogram
     * @param persons the contacts
     * @param orgUnits the orgunits
     * @throws SAXException if an error occurred
     */
    protected void _abstractProgram2CDMContacts(ContentHandler contentHandler, AbstractProgram<? extends ProgramFactory> program, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_CONTACTS);
        
        Map<String, List<String>> contactsByRole = program.getContactsByRole();
        
        for (Entry<String, List<String>> entry : contactsByRole.entrySet())
        {
            String role = entry.getKey();
            List<String> contactIds = entry.getValue();
            
            String roleCode = null;
            if (StringUtils.isNotEmpty(role))
            {
                roleCode = _refTableHelper.getItemCode(role);
            }
            
            for (String personID : contactIds)
            {
                try
                {
                    Person person = _resolver.resolveById(personID);
                    persons.add(person.getId());
                    attrs = new AttributesImpl();
                    attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, person.getCDMId());
                    if (roleCode != null)
                    {
                        attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ROLE, roleCode);
                    }
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_REF_PERSON, attrs);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Contact does exist anymore, ignore it
                }
            }
        }

        for (String orgUnitId : program.getOrgUnits())
        {
            if (!"".equals(orgUnitId))
            {
                try
                {
                    OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
                    orgUnits.add(orgUnit.getId());
                    attrs = new AttributesImpl();
                    attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, orgUnit.getCDMId());
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_REF_ORG_UNIT, attrs);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // OrgUnit does exist anymore, ignore it
                }
            }
        }
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_CONTACTS);
    }
    
    /**
     * SAX CDMfr events for a {@link ProgramPart} strcuture
     * @param contentHandler the content handler to sax into
     * @param programPart the program part
     * @param persons the contacts
     * @param orgUnits the orgunits
     * @param courses the courses
     * @throws SAXException if an error occurred
     */
    protected void _programPart2CDMSubPrograms(ContentHandler contentHandler, TraversableProgramPart programPart, Set<String> persons, Set<String> orgUnits, Set<String> courses) throws SAXException
    {
        ContentValue[] children = programPart.getValue(TraversableProgramPart.CHILD_PROGRAM_PARTS, false, new ContentValue[0]);
        for (ContentValue child : children)
        {
            Optional<ModifiableContent> content = child.getContentIfExists();
            if (content.isPresent())
            {
                if (content.get() instanceof AbstractProgram<?> program)
                {
                    abstractProgram2CDM(contentHandler, program, persons, orgUnits, courses);
                }
                else if (content.get() instanceof Container container)
                {
                    container2CDM(contentHandler, container, persons, orgUnits, courses);
                }
            }
        }
    }
    
    
    //***********************************************************************//
    //                             CONTAINER                                 //
    //***********************************************************************//
    
    /**
     * Send the CDM-fr representation of their ProgramPart to the given Contenthandler.<br>
     * Also collects referenced orgUnits, persons and courses, so that they could be sent afterwards.
     * @param contentHandler the receiving contentHandler.
     * @param container The container
     * @param persons collected {@link Person} ids.
     * @param orgUnits collected {@link OrgUnit} ids.
     * @param courses collected {@link Course} ids.
     * @throws SAXException if an error occurs during CDM processing.
     */
    public void container2CDM(ContentHandler contentHandler, Container container, Set<String> persons, Set<String> orgUnits, Set<String> courses) throws SAXException
    {
        if (container.isPublishable() && switchToLiveVersionIfNeeded(container))
        {
            Request request = ContextHelper.getRequest(_context);
            Content currentContent = (Content) request.getAttribute(Content.class.getName());
            
            try
            {
                request.setAttribute(Content.class.getName(), container);
                
                AttributesImpl attrs = new AttributesImpl();
                
                // <program id="FR_RNE_{rootOrgUnitId}_PR_{code}">
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID, container.getCDMId());
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_SUB_PROGRAM, attrs);
                
                // <programID>code</programID>
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PROGRAM_ID, container.getCDMId());
                
                // <programName><text>name</text></programName>
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_NAME);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_TEXT, container.getTitle());
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_NAME);
                
                // <programCode codeSet="userDefined">code</programCode>
                String code = container.getCode();
                if (StringUtils.isNotBlank(code))
                {
                    attrs = new AttributesImpl();
                    attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "userDefined");
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_CODE, attrs, code);
                }
                
                // <programDescription>...</programDescription>
                attrs = new AttributesImpl();
                String nature = container.getNature();
                if (StringUtils.isNotEmpty(nature))
                {
                    nature = _refTableHelper.getItemCDMfrValue(nature, true);
                    _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_NATURE, nature);
                }
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_DESCRIPTION, attrs);
                
                // <qualification>...</qualification>
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_QUALIFICATION);
                _addPositiveAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_ECTS_CREDITS, container.getEcts());
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_CREDITS, attrs);
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, null, null, null, true, isExportForAmetys());
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_CREDITS);
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_QUALIFICATION);
                
                // <programStructure>...</programStructure>
                ContentValue[] children = container.getValue(TraversableProgramPart.CHILD_PROGRAM_PARTS, false, new ContentValue[0]);
                for (ContentValue child : children)
                {
                    Optional<ModifiableContent> content = child.getContentIfExists();
                    if (content.isPresent() && content.get() instanceof CourseList courseList)
                    {
                        courselist2CDM(contentHandler, courseList, courses);
                    }
                }
                
                // <infoBlock>...</infoBlock>
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute("userDefined", "container");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, attrs);
                
                // <infoBlock userDefined="ametys-extension>...</infoBlock>
                saxExtensions(contentHandler, container, persons, orgUnits);
                
                // <subProgram>...</subProgram>
                _programPart2CDMSubPrograms(contentHandler, container, persons, orgUnits, courses);
                
                // </program>
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_SUB_PROGRAM);
            }
            finally
            {
                request.setAttribute(Content.class.getName(), currentContent);
            }
        }
    }
    
    /**
     * SAX the course list structure
     * @param contentHandler The content handler to SAX into
     * @param cl The course list
     * @param courses References of saxed courses
     * @throws SAXException if an error occurs
     */
    public void courselist2CDM (ContentHandler contentHandler, CourseList cl, Set<String> courses) throws SAXException
    {
        if (cl.isPublishable() && switchToLiveVersionIfNeeded(cl))
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_STRUCTURE);
            
            _courselist2CDM (contentHandler, cl);
            
            for (Course course  : cl.getCourses())
            {
                if (course.isPublishable())
                {
                    courses.add(course.getId());
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, course.getCDMId());
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REF_COURSE, attrs);
                }
            }
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROGRAM_STRUCTURE);
        }
    }
    
    /**
     * Convert course list info to CDM (without child courses)
     * @param contentHandler The content handler to sax into
     * @param cl The course list
     * @throws SAXException if an error occurs
     */
    protected void _courselist2CDM (ContentHandler contentHandler, CourseList cl) throws SAXException
    {
        if (switchToLiveVersionIfNeeded(cl))
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("userDefined", "listCode");
            String code = cl.getCode();
            if (StringUtils.isEmpty(code))
            {
                code = org.ametys.core.util.StringUtils.generateKey().toUpperCase();
            }
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, attrs, code);
            
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute("userDefined", "listName");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, attrs, cl.getTitle());
            
            ChoiceType type = cl.getType();
            if (type != null)
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute("userDefined", "listChoiceType");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, attrs, cl.getType().toString().toLowerCase());
            }
            
            if (type != null && type.equals(ChoiceType.CHOICE))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute("userDefined", "listMinNumberOfCourses");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, attrs, String.valueOf(cl.getMinNumberOfCourses()));
                
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute("userDefined", "listMaxNumberOfCourses");
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, attrs, String.valueOf(cl.getMaxNumberOfCourses()));
            }
        }
    }
    
    //***********************************************************************//
    //                               COURSE                                  //
    //***********************************************************************//
    
    /**
     * Export this entity in CDM-fr format.
     * @param contentHandler the target handler
     * @param course The course
     * @throws SAXException if an error occurs during export.
     */
    public void course2CDM(ContentHandler contentHandler, Course course) throws SAXException
    {
        boolean rootOrgUnitSaxed = false;
        
        Set<String> orgUnits = new HashSet<>();
        Set<String> persons = new HashSet<>();
        
        Set<String> coursesAlreadySax = new HashSet<>();
        Set<String> coursesToSax = new HashSet<>();
        coursesToSax.add(course.getId());
        
        while (!coursesToSax.isEmpty())
        {
            Set<String> courses = new HashSet<>();
            for (String courseId : coursesToSax)
            {
                coursesAlreadySax.add(courseId);
                Course courseToSax = _resolver.resolveById(courseId);
                courses.addAll(course2CDM(contentHandler, courseToSax, orgUnits, persons));
            }
            coursesToSax.clear();
            coursesToSax.addAll(courses);
            coursesToSax.removeAll(coursesAlreadySax);
        }
        
        for (String orgUnitId : orgUnits)
        {
            OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
            if (orgUnitId.equals(_rootOrgUnitProvider.getRootId()))
            {
                rootOrgUnitSaxed = true;
            }
            _orgunit2CDM(contentHandler, orgUnit, persons);
        }
        
        if (!rootOrgUnitSaxed)
        {
            OrgUnit orgUnit = _rootOrgUnitProvider.getRoot();
            _orgunit2CDM(contentHandler, orgUnit, persons);
        }
        
        for (String personId : persons)
        {
            Person person = _resolver.resolveById(personId);
            person2CDM(contentHandler, person);
        }
        
        // Set content in request
        ContextHelper.getRequest(_context).setAttribute(Content.class.getName(), course);
    }
    
    /**
     * Export this entity in CDM-fr format.
     * @param contentHandler the content handler to SAX into
     * @param course the course
     * @param orgUnits the org units
     * @param persons the persons
     * @return the saxed coursed
     * @throws SAXException if failed to generate CDM
     */
    public Set<String> course2CDM(ContentHandler contentHandler, Course course, Set<String> orgUnits, Set<String> persons) throws SAXException
    {
        DataContext courseRichTextsContext = RepositoryDataContext.newInstance()
                                                                  .withObject(course);

        if (!course.isPublishable() || !switchToLiveVersionIfNeeded(course))
        {
            return new HashSet<>();
        }
        
        Request request = ContextHelper.getRequest(_context);
        Content currentContent = (Content) request.getAttribute(Content.class.getName());
        
        try
        {
            request.setAttribute(Content.class.getName(), course);
            
            AttributesImpl attrs = new AttributesImpl();
            
            // <course id="FR_RNE_{rootOrgUnitId}_CO_{code}" language="...">
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID, course.getCDMId());
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_LANGUAGE, course.getLanguage());
            String erasmusCode = _refTableHelper.getItemCode(course.getErasmusCode());
            _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_IDENT, erasmusCode);
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_COURSE, attrs);
            
            // <courseID>code</courseID>
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_COURSE_ID, course.getCDMId());
            
            // <courseName><text>name</text></courseName>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_COURSE_NAME);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEXT, course.getTitle());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_COURSE_NAME);
    
            // <courseCode codeSet="UE">code</couseCode>
            String courseType = _refTableHelper.getItemCode(course.getCourseType());
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, courseType);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_COURSE_CODE, attrs, course.getCdmCode());
            
            // <webLink>
            //      <href>href</href>
            //      <linkName>linkName</linkName>
            // </webLink>
            String webLinkURL = course.getWebLinkUrl();
            if (StringUtils.isNotEmpty(webLinkURL))
            {
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_WEB_LINK);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_HREF, webLinkURL);
                String webLinkLabel = course.getWebLinkLabel();
                if (StringUtils.isNotEmpty(webLinkLabel))
                {
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_LINK_NAME, course.getWebLinkLabel());
                }
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_WEB_LINK);
            }
    
            // <level>level</level>
            attrs = new AttributesImpl();
            String level = _refTableHelper.getItemCDMfrValue(course.getLevel(), true);
            _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_LEVEL, level);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_LEVEL, attrs);
            
            // <courseDescription>...</courseDescription>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_COURSE_DESCRIPTION);
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_DIGITAL_USE);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PERCENTAGE, "0");
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_LCMS);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_EXISTS, "false");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_RESOURCE_MGT);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PERC_OF_USE, "0");
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_LCMS);
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROD_PEDA);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_DIGIT_PROD_PERC, "0");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_DIGIT_PROD_UNT_PERC, "0");
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROD_PEDA);
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_DIGITAL_USE);
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_INFO_BLOCK, course.getDescription(), courseRichTextsContext.cloneContext().withDataPath(Course.DESCRIPTION), _sourceResolver, true, isExportForAmetys());
            
            Set<String> courses = _courseLists2CDM(contentHandler, course);
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_COURSE_DESCRIPTION);
            
            // <teachingTerm term="" timeOfDay="" start=""/>
            attrs = new AttributesImpl();
            LocalDate startDate = course.getStartDate();
            if (startDate != null)
            {
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_START, startDate.format(CDMHelper.CDM_DATE_FORMATTER));
            }
            
            String teachingTerm = _refTableHelper.getItemCDMfrValue(course.getTeachingTerm(), true);
            _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_TERM, teachingTerm);
            String timeSlot = _refTableHelper.getItemCDMfrValue(course.getTimeSlot(), true);
            _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_TIME_OF_DAY, timeSlot);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEACHING_TERM, attrs);
    
            // <credits ECTScredits="" totalWorkLoad="">
            //      <infoBlock>...</infoBlock>
            //      <globalVolume teachingType="CM/TP/TD">...</globalVolume>
            // </credits>
            _credits2CDM(contentHandler, course);
            
            // <learningObjectives>...</learningObjectives>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_LEARNING_OBJECTIVES);
            CDMHelper.richText2CDM(contentHandler, course.getObjectives(), courseRichTextsContext.cloneContext().withDataPath(Course.OBJECTIVES), _sourceResolver, isExportForAmetys());
            // <ametys-cdm:skills>...</ametys-cdm:skills>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + "skills", course.getSkills(), courseRichTextsContext.cloneContext().withDataPath(Course.SKILLS), _sourceResolver, false, isExportForAmetys());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_LEARNING_OBJECTIVES);
            
            // <admissionInfo>
            //      <studentPlaces places=""/>
            //      <teachingStart date=""/>
            // </admissionInfo>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ADMISSION_INFO);
    
            String maxNumberOfStudents = course.getMaxNumberOfStudents();
            if (StringUtils.isNotEmpty(maxNumberOfStudents))
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_PLACES, maxNumberOfStudents);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_STUDENT_PLACES, attrs);
            }
            
            if (startDate != null)
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_DATE, new SimpleDateFormat("yyyy-MM-dd").format(DateUtils.asDate(startDate)));
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEACHING_START, attrs);
            }
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ADMISSION_INFO);
            
            // <formalPrerequisites>...</formalPrerequisites>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_FORMAL_PREREQUISITES, course.getNeededPrerequisite(), courseRichTextsContext.cloneContext().withDataPath(Course.NEEDED_PREREQUISITE), _sourceResolver, true, isExportForAmetys());
    
            // <teachingPlace>
            //      <cdmfr:refOrgUnit />
            //      <adr>
            //          <pcode></pcode>
            //      </adr>
            // </teachingPlace>
            _teachingLocation2CDM(contentHandler, course);
            
            // <formOfTeaching method="" org=""/>
            String formOfTeachingMethod = _refTableHelper.getItemCDMfrValue(course.getFormOfTeachingMethod(), false);
            for (String formOfTeachingOrg: course.getFormOfTeachingOrgs())
            {
                attrs = new AttributesImpl();
                String formOfTeachingOrgValue = _refTableHelper.getItemCDMfrValue(formOfTeachingOrg, false);
                _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_METHOD, formOfTeachingMethod);
                _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_ORG, formOfTeachingOrgValue);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_FORM_OF_TEACHING, attrs);
            }
            
            // <formOfAssessment>...</formOfAssessment>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_FORM_OF_ASSESSMENT, course.getFormOfAssessment(), courseRichTextsContext.cloneContext().withDataPath(Course.FORM_OF_ASSESSMENT), _sourceResolver, false, isExportForAmetys());
            
            // <instructionLanguage teachingLang=""/>
            String[] teachingLanguages = course.getTeachingLanguage();
            for (String teachingLanguage : teachingLanguages)
            {
                String teachingLanguageValue = _refTableHelper.getItemCDMfrValue(teachingLanguage, true);
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_TEACHING_LANG, teachingLanguageValue);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_INSTRUCTION_LANGUAGE, attrs);
            }
            
            // <syllabus>
            // ...
            //   <webLink role="learningObjectMetadata">
            //      <href>...</href>
            //      <linkName>...</linkName>
            //   </webLink>
            // </syllabus>
            _syllabus2CDM(contentHandler, course, courseRichTextsContext);
            
            // <teachingActivity method=""/>
            attrs = new AttributesImpl();
            String teachingActivity = _refTableHelper.getItemCDMfrValue(course.getTeachingActivity(), true);
            _addNotNullAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_METHOD, teachingActivity);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEACHING_ACTIVITY, attrs);
            
            // <contacts>
            //      <refPerson idRef=ID/>
            // </contacts>
            _course2CDMContacts(contentHandler, course, persons, orgUnits);
            
            // <infoBlock>...</infoBlock>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, course.getAdditionalInformations(), courseRichTextsContext.cloneContext().withDataPath(Course.ADDITIONAL_INFORMATIONS), _sourceResolver, false, isExportForAmetys());
            
            // <infoBlock userDefined="ametys-extension>...</infoBlock>
            saxExtensions(contentHandler, course, persons, orgUnits);
            
            // <searchword>keyword</searchword>
            for (String keyword : course.getKeywords())
            {
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_SEARCH_WORD, keyword);
            }
           
            // </course>
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_COURSE);

            return courses;
        }
        finally
        {
            request.setAttribute(Content.class.getName(), currentContent);
        }
    }
    
    /**
     * Sax the courses lists for a course.
     * @param contentHandler the content handler to SAX into
     * @param course The courselist container.
     * @return The list of courses into the course list.
     * @throws SAXException if failed to generate CDM
     */
    protected Set<String> _courseLists2CDM(ContentHandler contentHandler, Course course) throws SAXException
    {
        Set<String> courses = new HashSet<>();
        
        for (CourseList coursesList : course.getCourseLists())
        {
            if (coursesList.isPublishable())
            {
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_COURSE_CONTENTS);
                
                // SAX infos on course list. This DO NOT respect the CDM-fr schema
                _courselist2CDM(contentHandler, coursesList);
                
                for (Course childCourse : coursesList.getCourses())
                {
                    if (childCourse.isPublishable())
                    {
                        courses.add(childCourse.getId());
                        AttributesImpl attrs = new AttributesImpl();
                        attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, childCourse.getCDMId());
                        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REF_COURSE, attrs);
                    }
                }
                
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_COURSE_CONTENTS);
            }
        }
        
        return courses;
    }
    
    /**
     * Sax the course credits.
     * @param contentHandler the content handler to SAX into
     * @param course The course to sax
     * @throws SAXException if failed to generate CDM
     */
    protected void _credits2CDM(ContentHandler contentHandler, Course course) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        _addPositiveAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_TOTAL_WORK_LOAD, course.getNumberOfHours());
        _addPositiveAttribute(attrs, CDMFRTagsConstants.ATTRIBUTE_ECTS_CREDITS, course.getEcts());
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_CREDITS, attrs);
        CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, null, null, _sourceResolver, true, isExportForAmetys());
        
        Map<String, Double> hoursByNature = new HashMap<>();
        List<CoursePart> courseParts = course.getCourseParts();
        for (CoursePart coursePart : courseParts)
        {
            String nature = coursePart.getNature();
            Double hours = MapUtils.getDouble(hoursByNature, nature, 0d) + coursePart.getNumberOfHours();
            if (hours > 0)
            {
                hoursByNature.put(nature, hours);
            }
        }
        
        for (String nature : hoursByNature.keySet())
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_TEACHING_TYPE, _refTableHelper.getItemCDMfrValue(nature, true));
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_GLOBAL_VOLUME, attrs, String.valueOf(hoursByNature.get(nature)));
        }
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_CREDITS);
    }

    /**
     * SAX the syllabus elements of the course.
     * @param contentHandler The content handler
     * @param course The course to SAX
     * @param courseRichTextsContext the context for course's rich texts
     * @throws SAXException if an error occurs
     */
    protected void _syllabus2CDM(ContentHandler contentHandler, Course course, DataContext courseRichTextsContext) throws SAXException
    {
        RichText syllabus = course.getSyllabus();
        RichText bibliography = course.getBibliography();
        Set<LOMSheet> lomSheets = course.getLOMSheets();
        if (syllabus != null || bibliography != null || !lomSheets.isEmpty())
        {
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_SYLLABUS);
            
            // Syllabus
            if (syllabus != null)
            {
                CDMHelper.richText2CDM(contentHandler, syllabus, courseRichTextsContext.cloneContext().withDataPath(Course.SYLLABUS), _sourceResolver, isExportForAmetys());
            }
            
            // Bibliography
            if (bibliography != null)
            {
                CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_AMETYS_CDM + CDMFRTagsConstants.TAG_BIBLIOGRAPHY, bibliography, courseRichTextsContext.cloneContext().withDataPath(Course.BIBLIOGRAPHY), _sourceResolver, false, isExportForAmetys());
            }
            
            // LOM sheets
            for (LOMSheet lomSheet : lomSheets)
            {
                lomSheet.toCDM(contentHandler);
            }
            
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_SYLLABUS);
        }
    }
    
    /**
     * Sax the course teaching locations.
     * @param contentHandler the content handler to SAX into
     * @param course The course to sax
     * @throws SAXException if failed to generate CDM
     */
    protected void _teachingLocation2CDM(ContentHandler contentHandler, Course course) throws SAXException
    {
        for (String id : course.getTeachingLocation())
        {
            if (StringUtils.isNotEmpty(id))
            {
                String postalCode = _refTableHelper.getItemCode(id);
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_TEACHING_PLACE);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REF_ORG_UNIT);
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ADDRESS);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PCODE, postalCode);
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ADDRESS);
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_TEACHING_PLACE);
            }
        }
    }
    
    /**
     * SAX CDMfr events for a {@link Course} contacts
     * @param contentHandler the content handler to sax into
     * @param course the course
     * @param persons the contacts
     * @param orgUnits the orgunits
     * @throws SAXException if an error occurred
     */
    protected void _course2CDMContacts(ContentHandler contentHandler, Course course, Set<String> persons, Set<String> orgUnits) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        
        XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_CONTACTS);
        
        Map<String, List<String>> contactsByRole = course.getContactsByRole();
        
        for (Entry<String, List<String>> entry : contactsByRole.entrySet())
        {
            String role = entry.getKey();
            List<String> contactIds = entry.getValue();
            
            String roleCode = null;
            if (StringUtils.isNotEmpty(role))
            {
                roleCode = _refTableHelper.getItemCode(role);
            }
            
            for (String personID : contactIds)
            {
                try
                {
                    Person person = _resolver.resolveById(personID);
                    persons.add(person.getId());
                    attrs = new AttributesImpl();
                    attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, person.getCDMId());
                    if (roleCode != null)
                    {
                        attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ROLE, roleCode);
                    }
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_REF_PERSON, attrs);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Contact does exist anymore, ignore it
                }
            }
        }
        
        for (String orgUnitID : course.getOrgUnits())
        {
            if (!"".equals(orgUnitID))
            {
                try
                {
                    OrgUnit orgUnit = _resolver.resolveById(orgUnitID);
                    orgUnits.add(orgUnit.getId());
                    attrs = new AttributesImpl();
                    attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, orgUnit.getCDMId());
                    XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_REF_ORG_UNIT, attrs);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // OrgUnit does exist anymore, ignore it
                }
            }
        }
        
        XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_CONTACTS);
    }
    
    
    //***********************************************************************//
    //                              ORGUNIT                                  //
    //***********************************************************************//
    
    /**
     * Export this entity in CDM-fr format.
     * @param contentHandler the target handler
     * @param orgunit the orgunit
     * @throws SAXException if an error occurs during export.
     */
    public void orgunit2CDM(ContentHandler contentHandler, OrgUnit orgunit) throws SAXException
    {
        Set<String> persons = new HashSet<>();
        
        _orgunit2CDM(contentHandler, orgunit, persons, true);
        
        for (String personId : persons)
        {
            Person person = _resolver.resolveById(personId);
            person2CDM(contentHandler, person);
        }
        
        if (!orgunit.getId().equals(_rootOrgUnitProvider.getRootId()))
        {
            OrgUnit rootOrgunit = _resolver.resolveById(_rootOrgUnitProvider.getRootId());
            _orgunit2CDM(contentHandler, rootOrgunit, persons);
        }
        
        // Set content in request
        ContextHelper.getRequest(_context).setAttribute(Content.class.getName(), orgunit);
    }
    
    /**
     * SAX CDMfr event for a {@link OrgUnit}
     * @param contentHandler the content handler to sax into
     * @param orgunit the orgunit
     * @param persons the contacts of orgunit
     * @throws SAXException if an error occurred
     */
    protected void _orgunit2CDM(ContentHandler contentHandler, OrgUnit orgunit, Set<String> persons) throws SAXException
    {
        _orgunit2CDM(contentHandler, orgunit, persons, false);
    }
    
    /**
     * SAX CDMfr event for a {@link OrgUnit}
     * @param contentHandler the content handler to sax into
     * @param orgunit the orgunit
     * @param persons the contacts of orgunit
     * @param showChildren true to sax child orgunits
     * @throws SAXException if an error occurred
     */
    protected void _orgunit2CDM(ContentHandler contentHandler, OrgUnit orgunit, Set<String> persons, boolean showChildren) throws SAXException
    {
        Request request = ContextHelper.getRequest(_context);
        Content currentContent = (Content) request.getAttribute(Content.class.getName());
        DataContext orgunitRichTextsContext = RepositoryDataContext.newInstance()
                                                                   .withObject(orgunit);
        
        try
        {
            _odfHelper.switchToLiveVersionIfNeeded(orgunit);
            
            request.setAttribute(Content.class.getName(), orgunit);
        
            AttributesImpl attrs = new AttributesImpl();
            
            // <orgunit id="FR_RNE_{rootOrgUnitId}_OR_{login}">
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID, orgunit.getCDMId());
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT, attrs);
            
            // <orgUnitID>RNE</orgUnitID>
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT_ID, orgunit.getCDMId());
            
            // <orgUnitName><text>name</text></orgUnitID>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT_NAME);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEXT, orgunit.getTitle());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT_NAME);
            
            // <orgUnitAcronym>acronyme</orgUnitAcronym>
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT_ACRONYM, orgunit.getAcronym());
            
            // <orgUnitCode codeSet="codeUAI">code</orgUnitCode>
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_CODE_SET, "codeUAI");
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT_CODE, attrs, orgunit.getUAICode());
    
            // <orgUnitKind orgUnitKindCodeValueSet=""/>
            attrs = new AttributesImpl();
            String orgunitType = _refTableHelper.getItemCDMfrValue(orgunit.getType(), false);
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ORG_UNIT_KIND_CODE_VALUESET, orgunitType);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT_KIND);
            
            // <webLink>
            //      <href>href</href>
            //      <linkName>linkName</linkName>
            // </webLink>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_WEB_LINK);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_HREF, orgunit.getWebLinkURL());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_LINK_NAME, orgunit.getWebLinkLabel());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_WEB_LINK);
            
            // <orgUnitDescription>...</orgUnitDescription>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT_DESCRIPTION, orgunit.getDescription(), orgunitRichTextsContext.cloneContext().withDataPath(OrgUnit.DESCRIPTION), _sourceResolver, false, isExportForAmetys());
            
            // <admissionInfo>
            //      <admissionDescription>...</admissionDescription>
            // </admissionInfo>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ADMISSION_INFO);
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_ADMISSION_DESCRIPTION, orgunit.getAdmissionInfo(), orgunitRichTextsContext.cloneContext().withDataPath(OrgUnit.ADMISSION_INFO), _sourceResolver);
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ADMISSION_INFO);
            
            // <regulations>...</regulations>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_REGULATIONS, orgunit.getRegulations(), orgunitRichTextsContext.cloneContext().withDataPath(OrgUnit.REGULATIONS), _sourceResolver, false, isExportForAmetys());
            
            // <expenses>...</expenses>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_EXPENSES, orgunit.getExpenses(), orgunitRichTextsContext.cloneContext().withDataPath(OrgUnit.EXPENSES), _sourceResolver, false, isExportForAmetys());
            
            // <studentFacilities>...</studentFacilities>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_STUDENT_FACILITIES, orgunit.getStudentFacilities(), orgunitRichTextsContext.cloneContext().withDataPath(OrgUnit.STUDENT_FACILITIES), _sourceResolver, false, isExportForAmetys());
            
            // <universalAjustment>...</universalAjustment>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_UNIVERSAL_ADJUSTMENT, orgunit.getUniversalAdjustment(), orgunitRichTextsContext.cloneContext().withDataPath(OrgUnit.UNIVERSAL_ADJUSTMENT), _sourceResolver, false, isExportForAmetys());
            
            // <contacts>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_CONTACTS);
            
            for (String contactID : orgunit.getContacts())
            {
                if (StringUtils.isNotEmpty(contactID))
                {
                    try
                    {
                        Person person = _resolver.resolveById(contactID);
                        persons.add(person.getId());
                        attrs = new AttributesImpl();
                        attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, person.getCDMId());
                        attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ROLE, "contact");
                        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_REF_PERSON, attrs);
                    }
                    catch (UnknownAmetysObjectException e)
                    {
                        // Contact does exist anymore, ignore it
                    }
                }
            }
            
            // </contacts>
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_CONTACTS);
            
            // <infoBlock>...</infoBlock>
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, orgunit.getAdditionnalInfos(), orgunitRichTextsContext.cloneContext().withDataPath(OrgUnit.ADDITIONNAL_INFOS), _sourceResolver, false, isExportForAmetys());
    
            Set<OrgUnit> orgUnits = new HashSet<>();
            if (showChildren)
            {
                // <refOrgUnit IDref=""/>
                for (String orgUnitID : orgunit.getSubOrgUnits())
                {
                    try
                    {
                        OrgUnit orgUnit = _resolver.resolveById(orgUnitID);
                        orgUnits.add(orgUnit);
                        attrs = new AttributesImpl();
                        attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID_REF, orgUnit.getCDMId());
                        XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_REF_ORG_UNIT, attrs);
                    }
                    catch (UnknownAmetysObjectException e)
                    {
                        // OrgUnit does exist anymore, ignore it
                    }
                }
            }
            
            // <infoBlock userDefined="ametys-extension>...</infoBlock>
            saxExtensions(contentHandler, orgunit, null, null);
                    
            // </orgunit>
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ORG_UNIT);
    
            for (OrgUnit childOrgunit : orgUnits)
            {
                _orgunit2CDM(contentHandler, childOrgunit, persons);
            }
        }
        catch (NoLiveVersionException e)
        {
            getLogger().error("Orgunit {} ({}) is not validated so it cannot be exported to CDM-fr.", orgunit.getTitle(), orgunit.getId());
        }
        finally
        {
            request.setAttribute(Content.class.getName(), currentContent);
        }
    }
    
    
    //***********************************************************************//
    //                              PERSON                                   //
    //***********************************************************************//
    
    /**
     * Export this entity in CDM-fr format.
     * @param contentHandler the target handler
     * @param person the person
     * @throws SAXException if an error occurs during export.
     */
    public void person2CDM(ContentHandler contentHandler, Person person) throws SAXException
    {
        Request request = ContextHelper.getRequest(_context);
        Content currentContent = (Content) request.getAttribute(Content.class.getName());
        
        try
        {
            _odfHelper.switchToLiveVersionIfNeeded(person);
            
            request.setAttribute(Content.class.getName(), person);
            
            AttributesImpl attrs = new AttributesImpl();
            
            // <person id="FRUAI{rootOrgUnitId}PE{login}">
            attrs.addCDATAAttribute(CDMFRTagsConstants.ATTRIBUTE_ID, person.getCDMId());
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_PERSON, attrs);
            
            // <personID>login</personID>
            String login = person.getLogin();
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PERSON_ID, StringUtils.isNotEmpty(login) ? login : person.getName());
            
            // <name>
            //      <given>prénom</given>
            //      <family>nom</family>
            // </name>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_NAME);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_GIVEN_NAME, person.getGivenName());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_FAMILY_NAME, person.getLastName());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_NAME);
    
            // <title><text>titre</text></title>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_TITLE);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEXT, person.getPersonTitle());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_TITLE);
            
            // <role><text>role</text></role>
            for (String role : person.getRole())
            {
                XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ROLE);
                XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_TEXT, role);
                XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ROLE);
            }
            
            // <affiliation>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_AFFILIATION);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_REF_ORG_UNIT);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_PROFESS_FIELD);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_SECT_CNU);
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.NAMESPACE_CDMFR + CDMFRTagsConstants.TAG_AFFILIATION);
            
            
            // <contactData>
            ContactData contactData = person.getContactData();
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_CONTACT_DATA);
            
            // <adr>
            //      <extadr>extadr</extadr>
            //      <street>address</street>
            //      <locality>locality</locality>
            //      <pcode>pcode</pcode>
            // </adr>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_ADDRESS);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_EXTADR, contactData.getAdditionalAddress());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_STREET, contactData.getAddress());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_LOCALITY, contactData.getTown());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PCODE, contactData.getZipCode());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_ADDRESS);
            
            // <telephone>telephone</telephone>
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_PHONE, contactData.getPhone());
            
            // <fax>fax</fax>
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_FAX, contactData.getFax());
            
            // <email>email</email>
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_EMAIL, contactData.getMail());
            
            // <webLink>
            //      <href>href</href>
            //      <linkName>linkName</linkName>
            // </webLink>
            XMLUtils.startElement(contentHandler, CDMFRTagsConstants.TAG_WEB_LINK);
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_HREF, contactData.getWebLinkUrl());
            XMLUtils.createElement(contentHandler, CDMFRTagsConstants.TAG_LINK_NAME, contactData.getWebLinkLabel());
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_WEB_LINK);
            
            // </contactData>
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_CONTACT_DATA);
            
            // <infoBlock>...</infoBlock>
            DataContext context = RepositoryDataContext.newInstance()
                                                       .withObject(person)
                                                       .withDataPath(Person.ADDITIONAL_INFORMATIONS);
            CDMHelper.richText2CDM(contentHandler, CDMFRTagsConstants.TAG_INFO_BLOCK, person.getAdditionalInformations(), context, _sourceResolver, false, isExportForAmetys());
            
            // <infoBlock userDefined="ametys-extension>...</infoBlock>
            saxExtensions(contentHandler, person, null, null);
    
            // </person>
            XMLUtils.endElement(contentHandler, CDMFRTagsConstants.TAG_PERSON);
        }
        catch (NoLiveVersionException e)
        {
            getLogger().error("Person {} ({}) is not validated so it cannot be exported to CDM-fr.", person.getTitle(), person.getId());
        }
        finally
        {
            request.setAttribute(Content.class.getName(), currentContent);
        }
    }
    
    private String _getMentionType(Program program)
    {
        return Optional.ofNullable(program.<ContentValue>getValue(AbstractProgram.DEGREE))
            .flatMap(ContentValue::getContentIfExists)
            .map(c -> c.<String>getValue(OdfReferenceTableHelper.DEGREE_MENTION_TYPE))
            .orElse(null);
    }
    
    //***********************************************************************//
    //                               MISC                                    //
    //***********************************************************************//

    /**
     * SAX value as attributes if not null nor empty
     * @param attrs the XML attributes
     * @param localName the local name
     * @param value the value
     */
    protected void _addNotNullAttribute(AttributesImpl attrs, String localName, String value)
    {
        if (StringUtils.isNotEmpty(value))
        {
            attrs.addCDATAAttribute(localName, value);
        }
    }
    
    /**
     * SAX value as attributes if not null nor empty
     * @param attrs the XML attributes
     * @param localName the local name
     * @param value the value
     */
    protected void _addPositiveAttribute(AttributesImpl attrs, String localName, double value)
    {
        if (value > 0)
        {
            attrs.addCDATAAttribute(localName, String.valueOf(value));
        }
    }
}
