/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.clientsideelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.helper.DeleteODFContentHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.RootOrgUnitProvider;

/**
 * This element creates an action button to delete a ODF content
 */
public class DeleteContentClientSideElement extends org.ametys.cms.clientsideelement.DeleteContentClientSideElement
{
    /** The root orgunit provider */
    protected RootOrgUnitProvider _rootOrgUnitProvider;
    
    /** The delete ODF content helper */
    protected DeleteODFContentHelper _deleteODFContentHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _rootOrgUnitProvider = (RootOrgUnitProvider) manager.lookup(RootOrgUnitProvider.ROLE);
        _deleteODFContentHelper = (DeleteODFContentHelper) manager.lookup(DeleteODFContentHelper.ROLE);
    }

    @Override
    protected boolean _hasRight(Content content)
    {
        return _deleteODFContentHelper.hasRight(content);
    }
    
    @Override
    protected boolean _isModifiable(Content content)
    {
        if (content instanceof OrgUnit && _rootOrgUnitProvider.isRoot(content.getId()))
        {
            // The root orgunit can not be deleted
            return false;
        }
        
        return super._isModifiable(content);
    }
    
    @Override
    protected boolean _isContentReferenced(Content content)
    {
        return _deleteODFContentHelper.isContentReferenced(content);
    }
    
    /**
     * Delete ODF contents
     * @param contentsId The ids of contents to delete
     * @param parameters the additional parameters
     * @return the deleted and undeleted contents
     */
    @SuppressWarnings("unchecked")
    @Callable
    public Map<String, Object> deleteContents(List<String> contentsId, Map<String, Object> parameters)
    {
        Map<String, Map<String, Object>> initialContentParameters = new HashMap<>();
        for (String contentId : contentsId)
        {
            initialContentParameters.put(contentId, getContentDefaultParameters(_resolver.resolveById(contentId)));
        }
        
        Map<String, Object> deleteResults = _deleteODFContentHelper.deleteContents(contentsId, (String) parameters.get("mode"));
        
        Map<String, Object> updatedResults = new HashMap<>();
        Set<Content> allReferencedContents = new HashSet<>();
        Set<Content> allLockedContents = new HashSet<>();
        Set<Content> allUnauthorizedContents = new HashSet<>();
        Set<Content> allUnDeletedContents = new HashSet<>();
        for (String deletedContentId : deleteResults.keySet())
        {
            Map<String, Object> deleteResult = (Map<String, Object>) deleteResults.get(deletedContentId);
            Map<String, Object> updatedContentResults = new HashMap<>();
            
            if (deleteResult.containsKey("check-before-deletion-failed"))
            {
                updatedContentResults.put("check-before-deletion-failed", deleteResult.get("check-before-deletion-failed"));
            }
            
            String initialContentId = (String) deleteResult.get("initial-content");
            updatedContentResults.put("initial-content", initialContentParameters.get(initialContentId));
            
            List<Map<String, Object>> deletedContents = new ArrayList<>();
            for (String deleteContentId : (Set<String>) deleteResult.get("deleted-contents"))
            {
                Map<String, Object> deleteParameters = new HashMap<>();
                deleteParameters.put("id", deleteContentId);
                deletedContents.add(deleteParameters);
            }
            updatedContentResults.put("deleted-contents", deletedContents);
            
            List<Map<String, Object>> undeletedContents = new ArrayList<>();
            for (Content content : (Set<Content>) deleteResult.get("undeleted-contents"))
            {
                allUnDeletedContents.add(content);
                undeletedContents.add(getContentDefaultParameters(content));
            }
            updatedContentResults.put("undeleted-contents", undeletedContents);
            
            List<Map<String, Object>> referencedContents = new ArrayList<>();
            for (Content content : (Set<Content>) deleteResult.get("referenced-contents"))
            {
                allReferencedContents.add(content);
                referencedContents.add(getContentDefaultParameters(content));
            }
            updatedContentResults.put("referenced-contents", referencedContents);
            
            // Update the description for locked contents
            List<Map<String, Object>> updatedLockedContents = new ArrayList<>();
            for (Content content : (Set<Content>) deleteResult.get("locked-contents"))
            {
                allLockedContents.add(content);
                Map<String, Object> contentParameters = getContentDefaultParameters(content);
                contentParameters.put("description", _getLockedDescription(content));
                updatedLockedContents.add(contentParameters);
            }
            updatedContentResults.put("locked-contents", updatedLockedContents);
    
            // Update the description for unauthorized contents
            List<Map<String, Object>> updatedUnauthorizedContents = new ArrayList<>();
            for (Content content : (Set<Content>) deleteResult.get("unauthorized-contents"))
            {
                allUnauthorizedContents.add(content);
                Map<String, Object> contentParameters = getContentDefaultParameters(content);
                contentParameters.put("description", _getNoRightDescription(content));
                updatedUnauthorizedContents.add(contentParameters);
            }
            updatedContentResults.put("unauthorized-contents", updatedUnauthorizedContents);
            
            updatedResults.put(deletedContentId, updatedContentResults);
        }
        
        // Add the list of all referenced contents (removing duplicate contents)
        List<Map<String, Object>> allReferencedContentsMap = allReferencedContents.stream()
            .map(c -> getContentDefaultParameters(c))
            .collect(Collectors.toList());
        
        updatedResults.put("all-referenced-contents", allReferencedContentsMap);
        
        // Add the list of all locked contents (removing duplicate contents)
        List<Map<String, Object>> allLockedContentsMap = allLockedContents.stream()
                .map(c -> getContentDefaultParameters(c))
                .collect(Collectors.toList());
            
        updatedResults.put("all-locked-contents", allLockedContentsMap);
            
        // Add the list of all unauthorized contents (removing duplicate contents)
        List<Map<String, Object>> allUnauthorizedContentsMap = allUnauthorizedContents.stream()
                .map(c -> getContentDefaultParameters(c))
                .collect(Collectors.toList());
            
        updatedResults.put("all-unauthorized-contents", allUnauthorizedContentsMap);
        
        // Add the list of all undeleted contents (removing duplicate contents)
        List<Map<String, Object>> allUnDeletedContentsMap = allUnDeletedContents.stream()
                .map(c -> getContentDefaultParameters(c))
                .collect(Collectors.toList());
            
        updatedResults.put("all-undeleted-contents", allUnDeletedContentsMap);
            
        return updatedResults;
    }
    
    @Override
    protected Map<String, Object> getContentDefaultParameters(Content content)
    {
        Map<String, Object> contentDefaultParameters = super.getContentDefaultParameters(content);
        
        if (content instanceof ProgramItem && content.hasValue(ProgramItem.CODE) && contentDefaultParameters.containsKey("title"))
        {
            String title = (String) contentDefaultParameters.get("title");
            contentDefaultParameters.put("title", title + " (" + content.getValue(ProgramItem.CODE) + ")");
        }
        
        return contentDefaultParameters;
    }
}
