/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content;

import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.data.EducationalPath;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Generates additional information about a ODF content
 *
 */
public class AdditionalOdfContentPropertiesGenerator extends ServiceableGenerator
{
    private static final ListProgramItemComparator __LIST_PROGRAM_ITEM_COMPARATOR = new ListProgramItemComparator();
    
    /** The AmetysObject resolver. */
    protected AmetysObjectResolver _resolver;
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) serviceManager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String contentId = parameters.getParameter("contentId", request.getParameter("contentId"));
        
        Content content = null;
        if (StringUtils.isNotEmpty(contentId))
        {
            content = _resolver.resolveById(contentId);
        }
        else
        {
            content = (Content) request.getAttribute(Content.class.getName());
        }
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "odf");
        
        if (content instanceof ProgramItem)
        {
            XMLUtils.startElement(contentHandler, "programItem");
            saxProgramItemProperties((ProgramItem) content);
            XMLUtils.endElement(contentHandler, "programItem");
        }
        
        XMLUtils.endElement(contentHandler, "odf");
        contentHandler.startDocument();
        contentHandler.endDocument();
    }
    
    /**
     * SAX specifics properties for a program item
     * @param programItem the program item
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxProgramItemProperties(ProgramItem programItem) throws SAXException
    {
        _saxProgramItemAncestorPaths(programItem);
        _saxChildProgramItems(programItem);
    }
    
    /**
     * SAX all paths of program item
     * @param programItem The program item
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxProgramItemAncestorPaths(ProgramItem programItem) throws SAXException
    {
        if (_odfHelper.hasParentProgramItems(programItem))
        {
            List<EducationalPath> educationalPaths = _odfHelper.getEducationalPaths(programItem);
            if (!educationalPaths.isEmpty())
            {
                List<List<ProgramItem>> ancestorPaths = educationalPaths.stream()
                        .map(p -> p.getProgramItems(_resolver))
                        .collect(Collectors.toList());
                
                if (!ancestorPaths.isEmpty())
                {
                    // Sort the list to display it
                    ancestorPaths.sort(__LIST_PROGRAM_ITEM_COMPARATOR);
                    
                    XMLUtils.startElement(contentHandler, "ancestors");
                    for (List<ProgramItem> ancestorPath : ancestorPaths)
                    {
                        XMLUtils.startElement(contentHandler, "path");
                        for (ProgramItem ancestor : ancestorPath)
                        {
                            _saxProgramItem(ancestor);
                        }
                        XMLUtils.endElement(contentHandler, "path");
                    }
                        
                    XMLUtils.endElement(contentHandler, "ancestors");
                }
            }
        }
    }
    
    /**
     * SAX the child program items
     * @param programItem the program item
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxChildProgramItems(ProgramItem programItem) throws SAXException
    {
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        
        if (!children.isEmpty())
        {
            XMLUtils.startElement(contentHandler, "children");
            for (ProgramItem child : children)
            {
                _saxProgramItem(child);
            }
            XMLUtils.endElement(contentHandler, "children");
        }
    }
    
    /**
     * Sax a program item
     * @param item the program item
     * @throws SAXException if an error occurs while saxing
     */
    protected void _saxProgramItem(ProgramItem item) throws SAXException
    {
        Content content = (Content) item;
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("code", item.getCode());
        XMLUtils.createElement(contentHandler, "item", attrs, content.getTitle());
    }

    /**
     * Comparator of {@link List} composing of {@link ProgramItem} by title.
     */
    private static final class ListProgramItemComparator implements Comparator<List<ProgramItem>>
    {
        public int compare(List<ProgramItem> l1, List<ProgramItem> l2)
        {
            int l1Size = l1.size();
            int l2Size = l2.size();
            
            for (int i = 0; i < l1Size; i++)
            {
                if (l2Size <= i)
                {
                    // l1 is greater than l2
                    return 1;
                }
                else
                {
                    // Compare title
                    int comparing = ((Content) l1.get(i)).getTitle().compareTo(((Content) l2.get(i)).getTitle());
                    if (comparing != 0)
                    {
                        return comparing;
                    }
                }
            }
            
            // Lists are the same at the end of reading l1
            // Then compare size of the lists
            return Integer.compare(l1Size, l2Size);
        }
    }
}
