/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.courselist;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseContainer;
import org.ametys.odf.program.AbstractProgramPart;

/**
 * Class representing a list of {@link Course}
 */
public class CourseList extends AbstractProgramPart<CourseListFactory> implements CourseContainer
{
    /** Constants for choice type attribute */
    public static final String CHOICE_TYPE = "choiceType";
    /** Constants for minimum number of courses. This attribute is valued only if type is {@link ChoiceType}.CHOICE */
    public static final String MIN_COURSES = "min";
    /** Constants for maximum number of courses. This attribute is valued only if type is {@link ChoiceType}.CHOICE */
    public static final String MAX_COURSES = "max";
    /** Constants for child courses attribute */
    public static final String CHILD_COURSES = "courses";
    /** Name of the attribute holding the direct parent courses */
    public static final String PARENT_COURSES = "parentCourses";
    
    /**
     * The type of the {@link CourseList}
     */
    public enum ChoiceType
    {
        /** mandatory type */
        MANDATORY,
        /** optional type */
        OPTIONAL,
        /** choice type */
        CHOICE
    }

    /**
     * Constructor
     * @param node The JCR node
     * @param parentPath The parent path
     * @param factory the factory
     */
    public CourseList(Node node, String parentPath, CourseListFactory factory)
    {
        super(node, parentPath, factory);
    }

    @Override
    public List<Course> getCourses()
    {
        return Arrays.stream(getValue(CHILD_COURSES, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(Course.class::cast)
                .collect(Collectors.toList());
    }

    @Override
    public boolean containsCourse(String courseId)
    {
        List<String> courseIds = ContentDataHelper.getContentIdsListFromMultipleContentData(this, CHILD_COURSES);
        return courseIds.contains(courseId);
    }
    
    @Override
    public boolean hasCourses()
    {
        return !ContentDataHelper.isMultipleContentDataEmpty(this, CHILD_COURSES);
    }
    
    // --------------------------------------------------------------------------------------//
    // GETTERS AND SETTERS
    // --------------------------------------------------------------------------------------//
    /**
     * If this set of teaching unit element is a type of choice this method
     * return the minimum number of teaching unit elements that the student must
     * choose else -1.
     * 
     * @return the minimum number of teaching unit element that the student must choose. 0 if not set
     */
    public long getMinNumberOfCourses()
    {
        return getValue(CourseList.MIN_COURSES, false, 0L);
    }

    /**
     * If this set of teaching unit element is a type of choice this method
     * return the maximum number of teaching unit element that this set of
     * teaching unit element must contains else -1.
     * 
     * @return the maximum number of teaching unit element that this set of
     *         teaching unit element must contains.
     */
    public long getMaxNumberOfCourses()
    {
        return getValue(CourseList.MAX_COURSES, false, 0L);
    }
    
    /**
     * Indicates whether the teaching unit elements contained are mandatory,
     * optional or a choice.
     * 
     * @return the type of the teaching unit elements contained.
     */
    public ChoiceType getType()
    {
        try
        {
            String choiceType = getValue(CHOICE_TYPE);
            return Optional.ofNullable(choiceType).map(ChoiceType::valueOf).orElse(null);
        }
        catch (IllegalArgumentException e)
        {
            // No corresponding choice
            return null;
        }
    }

    /**
     * Get the ids of parent courses
     * @return the ids of parent courses
     */
    public List<Course> getParentCourses()
    {
        return Arrays.stream(getValue(PARENT_COURSES, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(Course.class::cast)
                .collect(Collectors.toList());
    }
        
    // --------------------------------------------------------------------------------------//
    // CDM-fr
    // --------------------------------------------------------------------------------------//
    @Override
    protected String getCDMType()
    {
        return "CL";
    }
}
