/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.enumeration;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemContainer;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * This component handles ODF reference tables
 *
 */
public class OdfReferenceTableHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = OdfReferenceTableHelper.class.getName();
 
    /** Abstract table ref */
    public static final String ABSTRACT_TABLE_REF = "odf-enumeration.AbstractTableRef";
    /** Domain */
    public static final String DOMAIN = "odf-enumeration.Domain";
    /** Degree */
    public static final String DEGREE = "odf-enumeration.Degree";
    /** Level */
    public static final String LEVEL = "odf-enumeration.Level";
    /** Program type. */
    public static final String PROGRAM_TYPE = "odf-enumeration.ProgramType";
    /** Form of teaching */
    public static final String FORMOFTEACHING_METHOD = "odf-enumeration.FormofteachingMethod";
    /** Orgnization of teaching */
    public static final String FORMOFTEACHING_ORG = "odf-enumeration.FormofteachingOrg";
    /** Teaching method */
    public static final String TEACHING_ACTIVITY = "odf-enumeration.TeachingActivity";
    /** Type of training course */
    public static final String INTERNSHIP = "odf-enumeration.Internship";
    /** Distance learning modalities */
    public static final String DISTANCE_LEARNING_MODALITIES = "odf-enumeration.DistanceLearningModalities";
    /** Place */
    public static final String PLACE = "odf-enumeration.Place";
    /** Teaching term. */
    public static final String TEACHING_TERM = "odf-enumeration.TeachingTerm";
    /** Time slot */
    public static final String TIME_SLOT = "odf-enumeration.TimeSlot";
    /** Code ROME */
    public static final String CODE_ROME = "odf-enumeration.CodeRome";
    /** Code ERASMUS */
    public static final String CODE_ERASMUS = "odf-enumeration.CodeErasmus";
    /** Code DGESIP */
    public static final String CODE_DGESIP = "odf-enumeration.CodeDgesip";
    /** Code SISE */
    public static final String CODE_SISE = "odf-enumeration.CodeSise";
    /** Code Cite97 */
    public static final String CODE_CITE97 = "odf-enumeration.CodeCite97";
    /** Code FAP */
    public static final String CODE_FAP = "odf-enumeration.CodeFap";
    /** Code NSF */
    public static final String CODE_NSF = "odf-enumeration.CodeNsf";
    /** RNCP level */
    public static final String RNCP_LEVEL = "odf-enumeration.RncpLevel";
    /** Join orgunit*/
    public static final String JOIN_ORGUNIT = "odf-enumeration.JoinOrgunit";
    /** Mention licence */
    public static final String ABSTRACT_MENTION = "odf-enumeration.Mention";
    /** Mention BUT */
    public static final String MENTION_BUT = "odf-enumeration.MentionBUT";
    /** Mention BUT */
    public static final String MENTION_BUT_ATTRIBUTE_PARCOURS = "parcours";
    /** Mention licence */
    public static final String MENTION_LICENCE = "odf-enumeration.MentionLicence";
    /** Mention licence pro */
    public static final String MENTION_LICENCEPRO = "odf-enumeration.MentionLicencepro";
    /** Mention master */
    public static final String MENTION_MASTER = "odf-enumeration.MentionMaster";
    /** Abstract table ref for category */
    public static final String ABSTRACT_TABLE_REF_CATEGORY = "odf-enumeration.AbstractTableRefCategory";
    /** Apprenticeship contract */
    public static final String APPRENTICESHIP_CONTRACT = "odf-enumeration.ApprenticeshipContract";
    /** Available certification */
    public static final String AVAILABLE_CERTIFICATION = "odf-enumeration.AvailableCertification";
    /** Campus */
    public static final String CAMPUS = "odf-enumeration.Campus";
    /** Category for code Erasmus */
    public static final String CODE_ERASMUS_CATEGORY = "odf-enumeration.CodeErasmusCategory";
    /** Category for code FAP */
    public static final String CODE_FAP_CATEGORY = "odf-enumeration.CodeFapCategory";
    /** Nature of container */
    public static final String CONTAINER_NATURE = "odf-enumeration.ContainerNature";
    /** Nature of course */
    public static final String COURSE_NATURE = "odf-enumeration.CourseNature";
    /** Discipline */
    public static final String DISCIPLINE = "odf-enumeration.Discipline";
    /** Duration */
    public static final String DURATION = "odf-enumeration.Duration";
    /** ECTS */
    public static final String ECTS = "odf-enumeration.Ects";
    /** Foreign place */
    public static final String FOREIGN_PLACE = "odf-enumeration.ForeignPlace";
    /** International education */
    public static final String INTERNATIONAL_EDUCATION = "odf-enumeration.InternationalEducation";
    /** Language */
    public static final String LANGUAGE = "odf-enumeration.Language";
    /** OrgUnit type */
    public static final String ORGUNIT_TYPE = "odf-enumeration.OrgUnitType";
    /** Period */
    public static final String PERIOD = "odf-enumeration.Period";
    /** Period type */
    public static final String PERIOD_TYPE = "odf-enumeration.PeriodType";
    /** Person role */
    public static final String PERSON_ROLE = "odf-enumeration.PersonRole";
    /** Program field */
    public static final String PROGRAM_FIELD = "odf-enumeration.ProgramField";
    /** Sectors */
    public static final String SECTORS = "odf-enumeration.Sectors";
    /** Nature of course part */
    public static final String ENSEIGNEMENT_NATURE = "odf-enumeration.EnseignementNature";
    /** Category of nature of course part */
    public static final String ENSEIGNEMENT_NATURE_CATEGORY = "odf-enumeration.EnseignementNatureCategory";
    /** Skill */
    public static final String SKILL = "odf-enumeration.Skill";
    /** Skill set */
    public static final String SKILL_SET = "odf-enumeration.SkillSet";
    /** Attribute name for mention type in table refe degree */
    public static final String DEGREE_MENTION_TYPE = "mentionType";
    
    private AmetysObjectResolver _resolver;

    private ContentTypeExtensionPoint _cTypeEP;

    private ContentTypesHelper _cTypeHelper;

    private ODFHelper _odfHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _cTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    /**
     * Determines if the content type is a ODF table reference
     * @param cTypeId The id of content type
     * @return true if the content type is a ODF table reference
     */
    public boolean isTableReference(String cTypeId)
    {
        return _cTypeHelper.getAncestors(cTypeId).contains(ABSTRACT_TABLE_REF);
    }
    
    /**
     * Determines if the content is an entry of a ODF table reference
     * @param content The content
     * @return <code>true</code> if the content is an entry of a ODF table reference
     */
    public boolean isTableReferenceEntry(Content content)
    {
        String[] cTypeIds = content.getTypes();
        for (String cTypeId : cTypeIds)
        {
            if (isTableReference(cTypeId))
            {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Get the id of table references
     * @return The content type's id
     */
    public Set<String> getTableReferenceIds()
    {
        Set<String> tableRefIds = new HashSet<>();
        
        for (String cTypeId : _cTypeEP.getExtensionsIds())
        {
            if (_cTypeHelper.getAncestors(cTypeId).contains(ABSTRACT_TABLE_REF))
            {
                tableRefIds.add(cTypeId);
            }
        }
        return tableRefIds;
    }
    
    /**
     * Get the attribute definitions for table references for a given content type
     * @param cTypeId The id of content type
     * @return The attribute definitions for table references
     */
    public Map<String, ContentAttributeDefinition> getTableRefAttributeDefinitions(String cTypeId)
    {
        ContentType cType = _cTypeEP.getExtension(cTypeId);
        return _getTableRefAttributes(cType);
    }
    
    private Map<String, ContentAttributeDefinition> _getTableRefAttributes(ModelItemContainer modelItemContainer)
    {
        Map<String, ContentAttributeDefinition> tableRefAttributes = new LinkedHashMap<>();
        
        for (ModelItem modelItem : modelItemContainer.getModelItems())
        {
            if (modelItem instanceof ContentAttributeDefinition)
            {
                ContentAttributeDefinition attributeDefinition = (ContentAttributeDefinition) modelItem;
                if (isTableReference(attributeDefinition.getContentTypeId()))
                {
                    tableRefAttributes.put(attributeDefinition.getPath(), attributeDefinition);
                }
            }
            else if (modelItem instanceof ModelItemContainer && !(modelItem instanceof RepeaterDefinition))
            {
                // enumerated attributes in repeaters are not supported
                tableRefAttributes.putAll(_getTableRefAttributes((ModelItemContainer) modelItem));
            }
        }
        
        return tableRefAttributes;
    }
    
    /**
     * Get the attribute definitions for table references for a given content type and a view
     * @param cTypeId The id of content type
     * @param viewName the name of the view. Cannot be null.
     * @return The attributes definitions for table references
     */
    public Map<String, ContentAttributeDefinition> getTableRefAttributeDefinitions(String cTypeId, String viewName)
    {
        ContentType cType = _cTypeEP.getExtension(cTypeId);
        
        View view = cType.getView(viewName);
        return _getTableRefAttributes(view);
    }
    
    private Map<String, ContentAttributeDefinition> _getTableRefAttributes(ViewItemContainer viewItemContainer)
    {
        Map<String, ContentAttributeDefinition> tableRefAttributes = new LinkedHashMap<>();
        
        for (ViewItem viewItem : viewItemContainer.getViewItems())
        {
            if (viewItem instanceof ViewElement)
            {
                ElementDefinition definition = ((ViewElement) viewItem).getDefinition();
                if (definition instanceof ContentAttributeDefinition)
                {
                    tableRefAttributes.put(definition.getPath(), (ContentAttributeDefinition) definition);
                }
            }
            else if (viewItem instanceof ViewItemContainer)
            {
                tableRefAttributes.putAll(_getTableRefAttributes((ViewItemContainer) viewItem));
            }
        }
        
        return tableRefAttributes;
    }
    
    /**
     * Get the content type for mention for this degree
     * @param degreeIds The ids of degrees
     * @return A map with the id of content type or null if there is no mention for this degree
     */
    @Callable
    public Map<String, String> getMentionContentTypes(List<String> degreeIds)
    {
        Map<String, String> mentionTypes = new HashMap<>();
        for (String degreeId : degreeIds)
        {
            mentionTypes.put(degreeId, getMentionForDegree(degreeId));
        }
        return mentionTypes;
    }
    
    /**
     * Get the available BUT training paths for a BUT mention
     * @param mentionId the id of BUT mention
     * @return the BUT training paths
     */
    public ContentValue[] getBUTParcoursForMention(String mentionId)
    {
        Content content = _resolver.resolveById(mentionId);
        if (_cTypeHelper.isInstanceOf(content, MENTION_BUT))
        {
            return content.getValue(MENTION_BUT_ATTRIBUTE_PARCOURS, false, new ContentValue[0]);
        }
        return new ContentValue[0];
    }
    
    /**
     * Get the available BUT training paths for a content
     * @param contentId the content's id
     * @param mentionId the id of mention. Can be null or empty
     * @return the available BUT training paths
     */
    @Callable
    public List<Map<String, String>> getBUTParcoursItems(String contentId, String mentionId)
    {
        ContentValue[] values = null;
        if (StringUtils.isNotEmpty(mentionId))
        {
            // Get items from mention
            values = getBUTParcoursForMention(mentionId);
        }
        else
        {
            // Get items from parent program
            values = getBUTParcoursItems(contentId);
        }
        
        if (values != null)
        {
            return Arrays.stream(values)
                .map(ContentValue::getContent)
                .map(c -> Map.of("id", c.getId(), "title", c.getTitle()))
                .collect(Collectors.toList());
        }
        else
        {
            return List.of();
        }
    }
    /**
     * Get the available BUT training paths for a {@link ProgramItem}
     * @param programItemId the id of program item
     * @return the BUT training paths
     */
    public ContentValue[] getBUTParcoursItems(String programItemId)
    {
        Content content = _resolver.resolveById(programItemId);
        
        // Get first parent program
        Set<Program> parentPrograms = _odfHelper.getParentPrograms((AbstractProgram) content);
        if (!parentPrograms.isEmpty())
        {
            Program program = parentPrograms.iterator().next();
            
            if (isBUTDiploma(program))
            {
                return Optional.ofNullable(program.<ContentValue>getValue(AbstractProgram.MENTION))
                    .flatMap(ContentValue::getContentIfExists)
                    .map(c -> c.<ContentValue[]>getValue(MENTION_BUT_ATTRIBUTE_PARCOURS))
                    .orElse(new ContentValue[0]);
                
            }
        }
        
        // No BUT training paths
        return new ContentValue[0];
    }
    
    /**
     * Determines if the program is a BUT diploma
     * @param program the program
     * @return true if the program has a BUT
     */
    public boolean isBUTDiploma(Program program)
    {
        String mentionType = getMentionType(program);
        return mentionType != null && MENTION_BUT.equals(mentionType);
    }
    
    /**
     * Get the type of mention for a program
     * @param program the program
     * @return the type of mention or null if there is no mention for this program
     */
    public String getMentionType(Program program)
    {
        return Optional.ofNullable(program.<ContentValue>getValue(AbstractProgram.DEGREE))
                .flatMap(ContentValue::getContentIfExists)
                .map(c -> c.<String>getValue(DEGREE_MENTION_TYPE))
                .orElse(null);
    }
    
    /**
     * Get the mention for a given degree.
     * @param degreeId The degree ID
     * @return The associated mention reference table, null if there isn't
     */
    public String getMentionForDegree(String degreeId)
    {
        try
        {
            Content content = _resolver.resolveById(degreeId);
            return content.getValue(DEGREE_MENTION_TYPE);
        }
        catch (UnknownAmetysObjectException e)
        {
            // Nothing to do
        }
        return null;
    }
    
    /**
     * Get the CDM-fr value associated with the given code
     * @param tableRefId The id of content type
     * @param code The code
     * @return The CDM-fr value or empty string if not found
     */
    public String getCDMfrValue (String tableRefId, String code)
    {
        ContentTypeExpression cTypeExpr = new ContentTypeExpression(Operator.EQ, tableRefId);
        StringExpression codeExpr = new StringExpression(OdfReferenceTableEntry.CODE, Operator.EQ, code);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, codeExpr));
        AmetysObjectIterable<Content> contents = _resolver.query(xpathQuery);
        AmetysObjectIterator<Content> it = contents.iterator();
        
        if (it.hasNext())
        {
            OdfReferenceTableEntry entry = new OdfReferenceTableEntry(it.next());
            return entry.getCdmValue();
        }
        
        return "";
    }
    
    /**
     * Get all items of an enumeration and their label
     * @param tableRefId The id of content type
     * @return items of enumeration
     */
    public List<OdfReferenceTableEntry> getItems (String tableRefId)
    {
        return getItems(tableRefId, new SortField[0]);
    }
    
    /**
     * Get all items of an enumeration and their label, sorted by given fields
     * @param tableRefId The id of content type
     * @param sortFields The sort fields to order results
     * @return items of enumeration
     */
    public List<OdfReferenceTableEntry> getItems (String tableRefId, SortField... sortFields)
    {
        return getItems(tableRefId, true, sortFields);
    }
    
    /**
     * Get all items of an enumeration and their label, sorted by given fields
     * @param tableRefId The id of content type
     * @param includeArchived <code>true</code> to include archived entries
     * @param sortFields The sort fields to order results
     * @return items of enumeration
     */
    public List<OdfReferenceTableEntry> getItems (String tableRefId, boolean includeArchived, SortField... sortFields)
    {
        Expression expr = new ContentTypeExpression(Operator.EQ, tableRefId);
        if (!includeArchived)
        {
            Expression notArchivedExpr = new NotExpression(new BooleanExpression(OdfReferenceTableEntry.ARCHIVED, true));
            expr = new AndExpression(expr, notArchivedExpr);
        }
        
        SortCriteria sortCriteria = new SortCriteria();
        for (SortField sortField : sortFields)
        {
            sortCriteria.addCriterion(sortField.getName(), sortField.getAscending(), sortField.getNormalize());
        }
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr, sortCriteria);
        AmetysObjectIterable<Content> contents = _resolver.query(xpathQuery);
        
        return contents.stream().map(OdfReferenceTableEntry::new).collect(Collectors.toList());
    }
    
    /**
     * Returns the label of an reference table entry
     * @param contentId The content id
     * @param lang The requested language of label
     * @return the item label or <code>null</code> if not found
     */
    public String getItemLabel(String contentId, String lang)
    {
        try
        {
            Content content = _resolver.resolveById(contentId);
            OdfReferenceTableEntry entry = new OdfReferenceTableEntry(content);
            return entry.getLabel(lang);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Returns the label of an reference table entry
     * @param tableRefId The id of content type (useless)
     * @param contentId The content id
     * @param lang The requested language of label
     * @return the item label or <code>null</code> if not found
     * @deprecated Use {@link #getItemLabel(String, String)} instead
     */
    @Deprecated
    public String getItemLabel (String tableRefId, String contentId, String lang)
    {
        return getItemLabel(contentId, lang);
    }

    /**
     * Returns the CMD value of an reference table entry
     * @param contentId The content id
     * @param returnCodeIfEmpty <code>true</code> to return the code if CDM-fr value is empty
     * @return the CDM-fr value or empty value if not found
     */
    public String getItemCDMfrValue(String contentId, boolean returnCodeIfEmpty)
    {
        if (StringUtils.isEmpty(contentId))
        {
            return "";
        }
        
        Content content = _resolver.resolveById(contentId);
        OdfReferenceTableEntry entry = new OdfReferenceTableEntry(content);
        
        String cdmValue = entry.getCdmValue();
        
        if (StringUtils.isEmpty(cdmValue) && returnCodeIfEmpty)
        {
            return entry.getCode();
        }
        return cdmValue;
    }
    
    /**
     * Returns the CMD value of an reference table entry
     * @param tableRefId The id of content type (useless)
     * @param contentId The content id
     * @param returnCodeIfEmpty <code>true</code> to return the code if CDM-fr value is empty
     * @return the CDM-fr value or empty value if not found
     * @deprecated Use {@link #getItemCDMfrValue(String, boolean)} instead
     */
    @Deprecated
    public String getItemCDMfrValue (String tableRefId, String contentId, boolean returnCodeIfEmpty)
    {
        return getItemCDMfrValue(contentId, returnCodeIfEmpty);
    }
    
    /**
     * Returns the code of an reference table entry from its CDM value
     * @param contentId The id of content
     * @return the code or empty value if not found
     */
    public String getItemCode(String contentId)
    {
        if (StringUtils.isEmpty(contentId))
        {
            return "";
        }
        
        try
        {
            Content content = _resolver.resolveById(contentId);
            OdfReferenceTableEntry entry = new OdfReferenceTableEntry(content);
            return entry.getCode();
        }
        catch (UnknownAmetysObjectException e)
        {
            return "";
        }
    }
    
    /**
     * Returns the code of an reference table entry from its CDM value
     * @param tableRefId The id of content type (useless)
     * @param contentId The id of content
     * @return the code or empty value if not found
     * @deprecated Use {@link #getItemCode(String)} instead
     */
    @Deprecated
    public String getItemCode (String tableRefId, String contentId)
    {
        return getItemCode(contentId);
    }
    
    /**
     * Returns the code of an reference table entry from its CDM value
     * @param tableRefId The id of content type
     * @param cdmValue The CDM-fr value
     * @return the code or <code>null</code> if not found
     */
    public String getItemCodeFromCDM (String tableRefId, String cdmValue)
    {
        ContentTypeExpression cTypeExpr = _getContentTypeExpression(tableRefId);
        StringExpression codeExpr = new StringExpression(OdfReferenceTableEntry.CDM_VALUE, Operator.EQ, cdmValue);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, codeExpr));
        AmetysObjectIterable<Content> contents = _resolver.query(xpathQuery);
        AmetysObjectIterator<Content> it = contents.iterator();
        
        if (it.hasNext())
        {
            OdfReferenceTableEntry entry = new OdfReferenceTableEntry(it.next());
            return entry.getCode();
        }
        return null;
    }

    /**
     * Returns the entry of an reference table entry from its cdmValue
     * @param tableRefId The id of content type
     * @param cdmValue The CDM-fr value
     * @return the entry or <code>null</code> if not found
     */
    public OdfReferenceTableEntry getItemFromCDM(String tableRefId, String cdmValue)
    {
        ContentTypeExpression cTypeExpr = _getContentTypeExpression(tableRefId);
        StringExpression cdmExpr = new StringExpression(OdfReferenceTableEntry.CDM_VALUE, Operator.EQ, cdmValue);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, cdmExpr));

        return _resolver.<Content>query(xpathQuery).stream()
            .findFirst()
            .map(OdfReferenceTableEntry::new)
            .orElse(null);
    }
    
    /**
     * Returns the entry of an reference table entry from its code
     * @param tableRefId The id of content type
     * @param code The code
     * @return the entry or <code>null</code> if not found
     */
    public OdfReferenceTableEntry getItemFromCode(String tableRefId, String code)
    {
        ContentTypeExpression cTypeExpr = _getContentTypeExpression(tableRefId);
        StringExpression codeExpr = new StringExpression(OdfReferenceTableEntry.CODE, Operator.EQ, code);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, codeExpr));
        
        return _resolver.<Content>query(xpathQuery).stream()
            .findFirst()
            .map(OdfReferenceTableEntry::new)
            .orElse(null);
    }
    
    /**
     * Returns the entry of a reference table entry from its code for the connector (Apogée, Pégase...)
     * @param tableRefId The id of content type
     * @param connectorCode The code
     * @param codeFieldName The field name containing the connector code
     * @return the entry or <code>null</code> if not found
     */
    public OdfReferenceTableEntry getItemFromConnector(String tableRefId, String connectorCode, String codeFieldName)
    {
        ContentTypeExpression cTypeExpr = _getContentTypeExpression(tableRefId);
        StringExpression cdmExpr = new StringExpression(codeFieldName, Operator.EQ, connectorCode);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, cdmExpr));

        return _resolver.<Content>query(xpathQuery).stream()
            .findFirst()
            .map(OdfReferenceTableEntry::new)
            .orElse(null);
    }
    
    private ContentTypeExpression _getContentTypeExpression(String tableRefId)
    {
        Set<String> tableRefids;
        if (tableRefId.equals(OdfReferenceTableHelper.ABSTRACT_MENTION))
        {
            tableRefids = _cTypeEP.getSubTypes(tableRefId);
        }
        else
        {
            tableRefids = Collections.singleton(tableRefId);
        }
        
        return new ContentTypeExpression(Operator.EQ, tableRefids.toArray(new String[tableRefids.size()]));
    }

    /**
     * Returns the reference table entry from its CDM value
     * @param contentId The id of content
     * @return the item as an {@link OdfReferenceTableEntry} or null if not found
     */
    public OdfReferenceTableEntry getItem(String contentId)
    {
        try
        {
            Content content = _resolver.resolveById(contentId);
            return new OdfReferenceTableEntry(content);
        }
        catch (UnknownAmetysObjectException e)
        {
            // Can be an empty ID or an invalid ID (workspace or simply deleted element)
            return null;
        }
    }

    /**
     * SAX items of a reference table
     * @param contentHandler The content handler to sax into
     * @param tableRefId The id of reference table
     * @throws SAXException if an error occurred while saxing
     */
    public void saxItems (ContentHandler contentHandler, String tableRefId) throws SAXException
    {
        saxItems(contentHandler, tableRefId, null);
    }
    
    /**
     * SAX items of a reference table
     * @param contentHandler the content handler to sax into
     * @param attributeDefinition the metadata definition
     * @throws SAXException if an error occurs while saxing
     */
    public void saxItems (ContentHandler contentHandler, ContentAttributeDefinition attributeDefinition) throws SAXException
    {
        saxItems(contentHandler, attributeDefinition, null);
    }
    
    /**
     * SAX items of a reference table
     * @param contentHandler The content handler to sax into
     * @param tableRefId The id of reference table
     * @param lang the language to use to display items, can be <code>null</code> then odf language is used
     * @throws SAXException if an error occurred while saxing
     */
    public void saxItems (ContentHandler contentHandler, String tableRefId, String lang) throws SAXException
    {
        _saxItems(contentHandler, new AttributesImpl(), tableRefId, lang);
    }
    
    /**
     * SAX items of a reference table
     * @param contentHandler the content handler to sax into
     * @param attributeDefinition the metadata definition
     * @param lang the language to use to display items, can be <code>null</code> then odf language is used
     * @throws SAXException if an error occurs while saxing
     */
    public void saxItems(ContentHandler contentHandler, ContentAttributeDefinition attributeDefinition, String lang) throws SAXException
    {
        String cTypeId = attributeDefinition.getContentTypeId();
        if (cTypeId.startsWith("odf-enumeration."))
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("metadataName", attributeDefinition.getName());
            attrs.addCDATAAttribute("metadataPath", attributeDefinition.getPath());
            
            _saxItems(contentHandler, attrs, cTypeId, lang);
        }
    }
    
    private void _saxItems(ContentHandler contentHandler, AttributesImpl rootAttrs, String tableRefId, String lang) throws SAXException
    {
        String langToUse = lang != null ? lang : Config.getInstance().getValue("odf.programs.lang");
        boolean hasShortLabel = _cTypeEP.getExtension(tableRefId).hasModelItem("shortLabel");
        
        rootAttrs.addCDATAAttribute("contentTypeId", tableRefId);
        XMLUtils.startElement(contentHandler, "items", rootAttrs);
        
        List<OdfReferenceTableEntry> entries = getItems(tableRefId);
        for (OdfReferenceTableEntry entry : entries)
        {
            AttributesImpl valueAttrs = new AttributesImpl();
            valueAttrs.addCDATAAttribute("id", entry.getId());
            valueAttrs.addCDATAAttribute("order", entry.getOrder().toString());
            valueAttrs.addCDATAAttribute("code", entry.getCode());
            valueAttrs.addCDATAAttribute("cdmValue", entry.getCdmValue());
            valueAttrs.addCDATAAttribute("archived", entry.isArchived().toString());
            if (hasShortLabel)
            {
                valueAttrs.addCDATAAttribute("shortLabel", entry.getContent().getValue("shortLabel", false, StringUtils.EMPTY));
            }
            
            XMLUtils.createElement(contentHandler, "item", valueAttrs, entry.getLabel(langToUse));
        }
        
        XMLUtils.endElement(contentHandler, "items");
    }
    
    /**
     * This class represents a sort field for reference table.
     *
     */
    public static final class SortField
    {
        private String _name;
        private boolean _ascending;
        private boolean _normalize;

        /**
         * Create a sort field
         * @param name the name of field to sort on
         * @param ascending <code>true</code> to sort in ascending order, <code>false</code> otherwise
         */
        public SortField(String name, boolean ascending)
        {
            this(name, ascending, false);
        }
        
        /**
         * Create a sort field
         * @param name the name of field to sort on
         * @param ascending <code>true</code> to sort in ascending order, <code>false</code> otherwise
         * @param normalize <code>true</code> to normalize string properties (remove accents and lower case)
         */
        public SortField(String name, boolean ascending, boolean normalize)
        {
            _name = name;
            _ascending = ascending;
            _normalize = normalize;
        }

        /**
         * Get the name of the sort field
         * @return the name of the sort field
         */
        public String getName()
        {
            return _name;
        }

        /**
         * Get the order for sorting results
         * @return <code>true</code> to sort results in ascending order, <code>false</code> otherwise
         */
        public boolean getAscending()
        {
            return _ascending;
        }
        
        /**
         * Return the normalize status for this sort field
         * @return <code>true</code> if string properties should be normalized (remove accents and lower case)
         */
        public boolean getNormalize()
        {
            return _normalize;
        }

    }
}
