/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.export.indesign;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.collections.MapUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper storing mappings of xsl stylesheets (used in InDesign transformations) names with their labels. 
 */
public class IndesignTransformationHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Indesign Transformation Helper's role */
    public static final String ROLE = IndesignTransformationHelper.class.getName();
    
    /** The path of the single program transformation stylesheets */
    private static final String __XSLT_DIRECTORY_URI = "context://WEB-INF/stylesheets/indesign";
    
    /** The path of the catalog transformation stylesheets */
    private static final String __GLOBAL_XSLT_DIRECTORY_URI = "context://WEB-INF/stylesheets/indesign/catalogue";

    private SourceResolver _srcResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    /**
     * Get available XSLT files for single program InDesign export
     * @return the mapping of xsl names with their labels 
     */
    public Map<String, I18nizableText> getIndesignXslt()
    {
        try
        {
            return _parseXSLTFiles(__XSLT_DIRECTORY_URI);
        }
        catch (IOException e)
        {
            getLogger().error("Failed to get XSLT for indesign transformation", e);
        }
        
        return MapUtils.EMPTY_MAP;
    }
    
    /**
     * Get available XSLT files for ODF catalog InDesign export
     * @return the mapping of xsl names with their labels 
     */
    public Map<String, I18nizableText> getIndesignGlobalXslt()
    {
        try
        {
            return _parseXSLTFiles(__GLOBAL_XSLT_DIRECTORY_URI);
        }
        catch (IOException e)
        {
            getLogger().error("Failed to get XSLT for ODF catalog indesign transformation", e);
        }
        
        return MapUtils.EMPTY_MAP;
    }

    /**
     * Get the xslt data from the given directory
     * @param rootURI The directory where to find xsl files
     * @return the data under the form of a Map
     * @throws IOException if error occurred while parsing files
     */
    private Map<String, I18nizableText> _parseXSLTFiles(String rootURI) throws IOException
    {
        Map<String, I18nizableText> xsltFiles = new HashMap<> ();
        
        FileSource rootSrc = null;
        try
        {
            rootSrc = (FileSource) _srcResolver.resolveURI(rootURI);
            if (rootSrc.exists())
            {
                File rootDir = rootSrc.getFile();
                File[] xslFiles = rootDir.listFiles(new FileFilter() 
                {
                    public boolean accept(File file)
                    {
                        String lowerCasedFileName = file.getName().toLowerCase();
                        return file.isFile() && lowerCasedFileName.endsWith(".xsl");
                    }
                });
                
                // Generate the map
                for (File xslFile : xslFiles)
                {
                    I18nizableText label = _parseLabel(rootURI, xslFile.getName());
                    xsltFiles.put(xslFile.getName(), label);
                }
            }
        }
        finally
        {
            if (rootSrc != null)
            {
                _srcResolver.release(rootSrc);
            }
        }
        
        return xsltFiles;
    }
    
    private I18nizableText _parseLabel(String rootURI, String value)
    {
        String valueWithNoExtension = value.substring(0, value.length() - 4);
        String url = rootURI + "/" + valueWithNoExtension + ".xml";
        
        Source source = null;
        try
        {
            source = _srcResolver.resolveURI(url);
            if (source.exists())
            {
                try (InputStream inputStream = source.getInputStream())
                {
                    Configuration conf = new DefaultConfigurationBuilder().build(inputStream);
                    
                    Configuration node = conf.getChild("label");
                    I18nizableText text = _parseI18nizableText(node, "application");
                    return text;
                }
            }
        }
        catch (Exception e) 
        {
            // Nothing
        }
        finally
        {
            _srcResolver.release(source);
        }
        
        int i = valueWithNoExtension.lastIndexOf('/');
        String shortFilename = valueWithNoExtension.substring(i + 1);
        return new I18nizableText(shortFilename);
    }
    
    private I18nizableText _parseI18nizableText(Configuration textConfig, String defaultCatalogue) throws ConfigurationException
    {
        boolean i18nSupported = textConfig.getAttributeAsBoolean("i18n", false);
        String text = textConfig.getValue();
        
        if (i18nSupported)
        {
            String catalogue = textConfig.getAttribute("catalogue", null);
            
            if (catalogue == null)
            {
                catalogue = defaultCatalogue;
            }
            
            return new I18nizableText(catalogue, text);
        }
        else
        {
            return new I18nizableText(text);
        }
    }
}
